/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.tools.poi;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;

import nuts.core.io.IOUtils;
import nuts.core.tool.AbstractFileTool;
import nuts.tools.poi.xls.XlsTextExtractor;

import org.apache.commons.cli.CommandLine;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;

/**
 * 
 */
public class ExcelExtractor extends AbstractFileTool {
	public static class Main extends AbstractFileTool.Main {
		/**
		 * @param args arguments
		 */
		public static void main(String[] args) {
			Main m = new Main();
			ExcelExtractor c = new ExcelExtractor();
			m.execute(c, args);
		}

		
		/**
		 * Constructor 
		 */
		public Main() {
			hasTarget = true;
		}

		@Override
		protected void addCommandLineOptions() throws Exception {
			super.addCommandLineOptions();
			
			addCommandLineFlag("es", "summary", "extract summary");
			addCommandLineFlag("eh", "header", "extract header");
			addCommandLineFlag("ef", "footer", "extract footer");
		}

		@Override
		protected void getCommandLineOptions(CommandLine cl) throws Exception {
			super.getCommandLineOptions(cl);

			if (cl.hasOption("es")) {
				setParameter("extractSummary", true);
			}

			if (cl.hasOption("eh")) {
				setParameter("extractHeader", true);
			}
			
			if (cl.hasOption("ef")) {
				setParameter("extractFooter", true);
			}
		}
	}

	//---------------------------------------------------------------------------------------
	// properties
	//---------------------------------------------------------------------------------------

	private int cntSucceed = 0;
	private XlsTextExtractor xlsTextExtractor = new XlsTextExtractor(); 
	
	/**
	 * Constructor
	 */
	public ExcelExtractor() {
		includes = new String[] { "**/*.xls" };
	}

	/**
	 * @param extractSummary the extractSummary to set
	 */
	public void setExtractSummary(boolean extractSummary) {
		xlsTextExtractor.setExtractSummary(extractSummary);
	}

	/**
	 * @param extractHeader the extractHeader to set
	 */
	public void setExtractHeader(boolean extractHeader) {
		xlsTextExtractor.setExtractHeader(extractHeader);
	}

	/**
	 * @param extractFooter the extractFooter to set
	 */
	public void setExtractFooter(boolean extractFooter) {
		xlsTextExtractor.setExtractFooter(extractFooter);
	}

	@Override
	protected void afterProcess() throws Exception {
		super.afterProcess();
		
		println0(cntSucceed + " of " + cntFile + " files extracted successfully");
	}

	@Override
	protected void processFile(File file) throws Exception {
		println0("Extracting " + file.getPath());

		File xml = getTargetFile(file, ".xml");
		FileInputStream fis = null;
		FileOutputStream fos = null;
		HSSFWorkbook wb = null;

		try {
			fis = new FileInputStream(file);
			fos = new FileOutputStream(xml);

			wb = new HSSFWorkbook(fis);

			xlsTextExtractor.extractToXml(wb, fos);
			
			cntSucceed++;
		} 
		finally {
			IOUtils.closeQuietly(fis);
			IOUtils.closeQuietly(fos);
		}
	}
}
