/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.tools.codegen;

import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import nuts.core.lang.ClassUtils;
import nuts.tools.codegen.bean.Action;
import nuts.tools.codegen.bean.ActionProperty;
import nuts.tools.codegen.bean.InputUI;
import nuts.tools.codegen.bean.ListUI;
import nuts.tools.codegen.bean.Model;
import nuts.tools.codegen.bean.Module;
import freemarker.template.Configuration;
import freemarker.template.Template;

/**
 * action source generator
 */
public class ActionGenerator extends AbstractCodeGenerator {
	/**
	 * Main class for ActionGenerator
	 */
	public static class Main extends AbstractCodeGenerator.Main {
		/**
		 * @param args arguments
		 */
		public static void main(String[] args) {
			Main cgm = new Main();
			
			AbstractCodeGenerator cg = new ActionGenerator();

			cgm.execute(cg, args);
		}
	}

	//---------------------------------------------------------------------------------------
	// properties
	//---------------------------------------------------------------------------------------
	private Template tplAction;
	private Template tplActionV;
	private Template tplActionLV;
	private Template tplActionVV;
	private Template tplActionIV;
	private Template tplActionUV;

	private int cntAction = 0;
	
	@Override
	protected void loadTemplates(Configuration cfg) throws Exception {
		tplAction = cfg.getTemplate("action/Action.java.ftl");
		tplActionV = cfg.getTemplate("action/Action-validation.xml.ftl");
		tplActionLV = cfg.getTemplate("action/Action-list-validation.xml.ftl");
		tplActionVV = cfg.getTemplate("action/Action-view-validation.xml.ftl");
		tplActionIV = cfg.getTemplate("action/Action-insert-validation.xml.ftl");
		tplActionUV = cfg.getTemplate("action/Action-update-validation.xml.ftl");
	}

	@Override
	protected void processModule(Module module) throws Exception {
		for (Action action : module.getActionList()) {
			if (Boolean.TRUE.equals(action.getGenerate())) {
				print2("Processing action - " + action.getName());
				
				Model am = null;
				for (Model model : module.getModelList()) {
					if (model.getName().equals(action.getModel())) {
						am = model;
						break;
					}
				}
				
				if (am == null) {
					throw new Exception("Can not find model[" + action.getModel() + "] of action[" + action.getName() + "]");
				}
	
				processJavaAction(module, action, am);
				
				cntAction++;
			}
		}
	}

	@Override
	protected void postProcess() throws Exception {
		print0(cntModule + " modules processed, " + cntFile + " files of " + cntAction + " actions generated successfully.");
	}

	private void prepareImportList(List<ActionProperty> ps, Set<String> imports) {
		for (ActionProperty p : ps) {
			String type = p.getFullJavaType();
			if (type.endsWith("[]")) {
				type = type.substring(0, type.length() - 2);
			}
			ModelGenerator.addImportType(imports, type);
		}
	}

	private void processJavaAction(Module module, Action action, Model model) throws Exception {
		String pkg = ClassUtils.getPackageName(action.getFullActionClass());

		checkLicense(module, pkg);
		
		String cls = ClassUtils.getSimpleClassName(action.getFullActionClass());

		Map<String, Object> wrapper = getWrapper(module, action, model);

		Set<String> imports = new TreeSet<String>();
		prepareImportList(action.getPropertyList(), imports);

		imports.add(List.class.getName());
		imports.add(model.getModelBeanClass());
		imports.add(model.getModelExampleClass());
		imports.add(model.getModelMetaDataClass());
		imports.add(action.getActionBaseClass());
		
		setImports(wrapper, imports);
		processTpl(pkg, cls + ".java", wrapper, tplAction, true);

		if (!action.getPropertyList().isEmpty()) {
			processTpl(pkg, cls + "-validation.xml", wrapper, tplActionV);
		}

		for (ListUI lui : action.getListUIList()) {
			if (Boolean.TRUE.equals(lui.getGenerate())) {
				setActionUI(wrapper, lui);
				processTpl(pkg, cls + "-" + action.getName() + "_" + lui.getName()
						+ "-validation.xml", wrapper, tplActionLV);
			}
		}
		
		for (InputUI iui : action.getInputUIList()) {
			if (Boolean.TRUE.equals(iui.getGenerate())) {
				setActionUI(wrapper, iui);

				if (iui.getTemplates().contains("view") || iui.getTemplates().contains("print")) {
					processTpl(pkg, cls + "-" + action.getName() + "_" + iui.getName()
							+ "-validation.xml", wrapper, tplActionVV);
				}
				else if (iui.getTemplates().contains("delete")) {
					processTpl(pkg, cls + "-" + action.getName() + "_" + iui.getName()
							+ "-validation.xml", wrapper, tplActionVV);
					processTpl(pkg, cls + "-" + action.getName() + "_" + iui.getName()
							+ "_execute-validation.xml", wrapper, tplActionVV);
				}
				else if (iui.getTemplates().contains("insert")) {
					processTpl(pkg, cls + "-" + action.getName() + "_" + iui.getName()
							+ "_confirm-validation.xml", wrapper, tplActionIV);
					processTpl(pkg, cls + "-" + action.getName() + "_" + iui.getName()
							+ "_execute-validation.xml", wrapper, tplActionIV);
				}
				else if (iui.getTemplates().contains("copy")) {
					processTpl(pkg, cls + "-" + action.getName() + "_" + iui.getName()
							+ "-validation.xml", wrapper, tplActionVV);
					processTpl(pkg, cls + "-" + action.getName() + "_" + iui.getName()
							+ "_confirm-validation.xml", wrapper, tplActionIV);
					processTpl(pkg, cls + "-" + action.getName() + "_" + iui.getName()
							+ "_execute-validation.xml", wrapper, tplActionIV);
				}
				else if (iui.getTemplates().contains("update")) {
					processTpl(pkg, cls + "-" + action.getName() + "_" + iui.getName()
							+ "-validation.xml", wrapper, tplActionVV);
					processTpl(pkg, cls + "-" + action.getName() + "_" + iui.getName()
							+ "_confirm-validation.xml", wrapper, tplActionUV);
					processTpl(pkg, cls + "-" + action.getName() + "_" + iui.getName()
							+ "_execute-validation.xml", wrapper, tplActionUV);
				}
			}
		}
	}

	private Map<String, Object> getWrapper(Module module, Action action, Model model) {
		Map<String, Object> wrapper = new HashMap<String, Object>();
		
		if ("true".equals(module.getProps().getProperty("source.datetime"))) {
			wrapper.put("date", Calendar.getInstance().getTime());
		}
		wrapper.put("module", module);
		wrapper.put("props", module.getProps());
		wrapper.put("action", action);
		wrapper.put("model", model);
		wrapper.put("gen", this);
		
		return wrapper;
	}
	
	private void setActionUI(Map<String, Object> wrapper, Object actionUI) {
		wrapper.put("ui", actionUI);
	}
	
	private void setImports(Map<String, Object> wrapper, Object imports) {
		wrapper.put("imports", imports);
	}


}
