/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.gae.dao;

import java.util.List;

import nuts.core.orm.dao.AbstractModelDAO;
import nuts.core.orm.dao.DataAccessException;
import nuts.core.orm.dao.DataAccessSession;
import nuts.core.orm.dao.DataHandler;

/**
 * @param <T> model type
 * @param <E> model example type
 */
@SuppressWarnings("unchecked")
public abstract class GaeModelDAO<T, E extends GaeQueryParameter> extends AbstractModelDAO<T, E> {
	/**
	 * Constructor
	 */
	public GaeModelDAO() {
		super();
	}

	/**
	 * Constructor
	 *
	 * @param dataAccessSession the dataAccessSession to set
	 */
	public GaeModelDAO(DataAccessSession dataAccessSession) {
		super(dataAccessSession);
	}

	/**
	 * @return the GaeDataAccessClient
	 */
	public GaeDataAccessClient getGaeDataAccessClient() {
		return ((GaeDataAccessClient)getDataAccessClient());
	}

	/**
	 * @return the GaeDataAccessSession
	 */
	public GaeDataAccessSession getGaeDataAccessSession() {
		return ((GaeDataAccessSession)getDataAccessSession());
	}

	/**
	 * @return model id
	 */
	protected abstract String getModelId();
	
	/**
	 * exists
	 * 
	 * @param key T
	 * @return T
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public boolean exists(T key) throws DataAccessException {
		return getGaeDataAccessSession().exists(getModelId(), key);
	}
	
	/**
	 * countByExample
	 * 
	 * @param example E
	 * @return count
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public int countByExample(E example) throws DataAccessException {
		return getGaeDataAccessSession().countByExample(getModelId(), example);
	}

	/**
	 * selectByPrimaryKey
	 * 
	 * @param key T
	 * @return T
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public T selectByPrimaryKey(T key) throws DataAccessException {
		return (T)(getGaeDataAccessSession().selectByPrimaryKey(getModelId(), key));
	}

	/**
	 * selectByExampleWithDataHandler
	 * 
	 * @param example E
	 * @param dataHandler data handler
	 * @return data count
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public int selectByExampleWithDataHandler(E example, DataHandler<T> dataHandler) throws DataAccessException {
		return getGaeDataAccessSession().selectByExampleWithDataHandler(getModelId(), example, dataHandler);
	}

	/**
	 * selectByExample
	 * 
	 * @param example E
	 * @return list of T 
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public List<T> selectByExample(E example) throws DataAccessException {
		return getGaeDataAccessSession().selectByExample(getModelId(), example);
	}

	/**
	 * if (exists(data)) { updateByPrimaryKey(data); } else { insert(data}; }
	 * 
	 * @param data T
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public void save(T data) throws DataAccessException {
		getGaeDataAccessSession().save(getModelId(), data);
	}
	
	/**
	 * insert
	 * 
	 * @param data T
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public void insert(T data) throws DataAccessException {
		getGaeDataAccessSession().insert(getModelId(), data);
	}

	/**
	 * deleteByPrimaryKey
	 * 
	 * @param key T
	 * @return count of deleted records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public int deleteByPrimaryKey(T key) throws DataAccessException {
		return getGaeDataAccessSession().deleteByPrimaryKey(getModelId(), key);
	}

	/**
	 * deleteByExample
	 * 
	 * @param example E
	 * @return count of deleted records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public int deleteByExample(E example) throws DataAccessException {
		return getGaeDataAccessSession().deleteByExample(getModelId(), example);
	}

	/**
	 * updateByPrimaryKey
	 * 
	 * @param data T
	 * @return count of updated records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public int updateByPrimaryKey(T data) throws DataAccessException {
		return getGaeDataAccessSession().updateByPrimaryKey(getModelId(), data);
	}

	/**
	 * updateByPrimaryKeySelective (ignore null properties)
	 * 
	 * @param data T
	 * @return count of updated records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public int updateByPrimaryKeySelective(T data) throws DataAccessException {
		return getGaeDataAccessSession().updateByPrimaryKeySelective(getModelId(), data);
	}

	/**
	 * updateByExample
	 * 
	 * @param data T
	 * @param example E
	 * @return count of updated records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public int updateByExample(T data, E example) throws DataAccessException {
		return getGaeDataAccessSession().updateByExample(getModelId(), data, example);
	}

	/**
	 * updateByExampleSelective (ignore null properties)
	 * 
	 * @param data T
	 * @param example E
	 * @return count of updated records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public int updateByExampleSelective(T data, E example) throws DataAccessException {
		return getGaeDataAccessSession().updateByExampleSelective(getModelId(), data, example);
	}
}
