/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.exts.struts2.components;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import nuts.core.lang.StringUtils;
import nuts.exts.struts2.NutsStrutsConstants;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.views.annotations.StrutsTag;
import org.apache.struts2.views.annotations.StrutsTagAttribute;

import com.opensymphony.xwork2.inject.Inject;
import com.opensymphony.xwork2.util.ValueStack;

/**
 * <!-- START SNIPPET: javadoc -->
 * <p>
 * Renders a jquery date picker.
 * </p>
 * 
 * <!-- END SNIPPET: javadoc -->
 * 
 * <b>Examples</b>
 * 
 * <pre>
 * &lt;!-- START SNIPPET: expl1 --&gt;
 * 
 * Example 1:
 *     &lt;r:timepicker name=&quot;order.date&quot; label=&quot;Order Date&quot; /&gt;
 * Example 2:
 *     &lt;r:timepicker name=&quot;delivery.date&quot; label=&quot;Delivery Date&quot; format=&quot;date&quot;  /&gt;
 * 
 * &lt;!-- END SNIPPET: expl1 --&gt;
 * </pre>
 * <p/>
 * 
 */
@StrutsTag(
		name = "timepicker", 
		tldTagClass = "nuts.exts.struts2.views.jsp.ui.TimePickerTag", 
		description = "Render timepicker", 
		allowDynamicAttributes = true)
public class TimePicker extends UIBean {

	/**
	 * TEMPLATE = "n-timepicker";
	 */
	public static final String TEMPLATE = "n-timepicker";

	protected static final Log log = LogFactory.getLog(TimePicker.class);

	protected static final String DEFAULT_TIME_FORMAT = "time";

	protected static final String DEFAULT_TIME_PATTERN = "HH:mm:ss";

	protected String maxlength;
	protected String readonly;
	protected String size;

	protected String inline;
	protected String format;
	protected String pattern;
	protected String options;
	protected String defaults;
	protected String mask;
	
	/**
	 * Constructor
	 * 
	 * @param stack value stack
	 * @param request request
	 * @param response response
	 */
	public TimePicker(ValueStack stack, HttpServletRequest request,
			HttpServletResponse response) {
		super(stack, request, response);
	}

	protected String getDefaultTemplate() {
		return TEMPLATE;
	}

	protected Class getValueClassType() {
		return null;
	}

	/**
	 * Evaluate extra parameters
	 */
	protected void evaluateExtraParams() {
		super.evaluateExtraParams();

		if (size != null) {
			addParameter("size", findString(size));
		}

		if (maxlength != null) {
			addParameter("maxlength", findString(maxlength));
		}

		if (readonly != null) {
			addParameter("readonly", findValue(readonly, Boolean.class));
		}

		if (inline != null) {
			addParameter("inline", findValue(inline, Boolean.class));
		}

		addParameter("format", getTimeFormat());
		addParameter("pattern", getTimePattern());
		
		if (mask != null) {
			addParameter("mask", findString(mask));
		}
		
		if (options != null) {
			addParameter("options", findString(options));
		}

		if (defaults != null) {
			addParameter("defaults", defaults);
		}
	}

	/**
	 * get time format from text
	 * 
	 * @return time format
	 */
	protected String getTimeFormat() {
		String value = null;
		if (format != null) {
			value = findString(format);
		}
		if (value == null) {
			value = (String)parameters.get("format");
		}
		if (StringUtils.isEmpty(value)) {
			value = DEFAULT_TIME_FORMAT;
		}		
		return value;
	}

	/**
	 * get time pattern from text
	 * 
	 * @return time pattern
	 */
	protected String getTimePattern() {
		String value = null;
		if (pattern != null) {
			value = findString(pattern);
		}
		if (value == null) {
			value = (String)parameters.get("pattern");
		}
		if (StringUtils.isEmpty(value)) {
			String format = (String)parameters.get("format");
			value = CDate.getDatePattern(getStack(), format);
		}
		if (StringUtils.isEmpty(value)) {
			value = DEFAULT_TIME_PATTERN;
		}		
		return value;
	}

	/**
     * @param defaults defaults
     */
    @Inject(value=NutsStrutsConstants.NUTS_TIMEPICKER_DEFAULTS, required=false)
    public void setDefaults(String defaults) {
    	this.defaults = defaults;
    }

	/**
	 * @param maxlength the maxlength to set
	 */
	@StrutsTagAttribute(description = "HTML maxlength attribute", type = "Integer")
	public void setMaxlength(String maxlength) {
		this.maxlength = maxlength;
	}

	/**
	 * @param maxlength the maxlength to set
	 */
	@StrutsTagAttribute(description = "Deprecated. Use maxlength instead.", type = "Integer")
	public void setMaxLength(String maxlength) {
		this.maxlength = maxlength;
	}

	/**
	 * @param readonly the readonly to set
	 */
	@StrutsTagAttribute(description = "Whether the input is readonly", type = "Boolean", defaultValue = "false")
	public void setReadonly(String readonly) {
		this.readonly = readonly;
	}

	/**
	 * @param size the size to set
	 */
	@StrutsTagAttribute(description = "HTML size attribute", type = "Integer")
	public void setSize(String size) {
		this.size = size;
	}

	/**
	 * @param inline the inline to set
	 */
	@StrutsTagAttribute(description = "The inline attribute", required = false, type = "Boolean", defaultValue = "false")
	public void setInline(String inline) {
		this.inline = inline;
	}

	/**
	 * @param format the format to set
	 */
	@StrutsTagAttribute(description = "The format attribute", required = false)
	public void setFormat(String format) {
		this.format = format;
	}

	/**
	 * @param pattern the pattern to set
	 */
	@StrutsTagAttribute(description = "The pattern attribute", required = false)
	public void setPattern(String pattern) {
		this.pattern = pattern;
	}

	/**
	 * @param mask the mask to set
	 */
	@StrutsTagAttribute(description = "The mask attribute", required = false)
	public void setMask(String mask) {
		this.mask = mask;
	}

	/**
	 * @param options the options to set
	 */
	@StrutsTagAttribute(description = "The time picker javascript options", required = false)
	public void setOptions(String options) {
		this.options = options;
	}
}
