/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.exts.fileupload;

import java.io.IOException;
import java.util.UUID;

import nuts.core.io.IOUtils;
import nuts.core.lang.StringUtils;
import nuts.exts.struts2.NutsStrutsConstants;

import org.apache.commons.vfs2.FileObject;
import org.apache.commons.vfs2.FileSystemException;
import org.apache.commons.vfs2.FileSystemManager;
import org.apache.commons.vfs2.VFS;

import com.opensymphony.xwork2.inject.Inject;

/**
 */
public class UploadManager {
	/**
	 * File System Manager
	 */
	private static FileSystemManager fileSystemManager;

	/**
	 * The directory in which uploaded files will be stored, if stored on disk.
	 */
	private static FileObject repository;

	/**
	 * The directory in which uploaded files will be stored, if stored on disk.
	 */
	private static String saveDir = System.getProperty("java.io.tmpdir");

	/**
	 * @return FileSystemManager
	 * @throws FileSystemException if an error occurs
	 */
	public static FileSystemManager getFileSystemManager() throws FileSystemException {
		if (fileSystemManager == null) {
			fileSystemManager = VFS.getManager();
		}
		return fileSystemManager;
	}

	/**
	 * @param fileSystemManager the fileSystemManager to set
	 */
	@Inject(required=false)
	public static void setFileSystemManager(FileSystemManager fileSystemManager) {
		UploadManager.fileSystemManager = fileSystemManager;
	}

	/**
	 * @return the repository
	 * @throws IOException IOException
	 */
	public static FileObject getRepository() throws IOException {
		if (repository == null) {
			repository = getFileSystemManager().resolveFile(saveDir);
		}
		return repository;
	}

	/**
	 * @param repository the repository to set
	 */
	public static void setRepository(FileObject repository) {
		UploadManager.repository = repository;
	}

	/**
	 * @return the saveDir
	 */
	public static String getSaveDir() {
		return saveDir;
	}

	/**
	 * @param saveDir the saveDir to set
	 */
	@Inject(value=NutsStrutsConstants.NUTS_MULTIPART_SAVEDIR, required=false)
	public static void setSaveDir(String saveDir) {
		UploadManager.saveDir = saveDir;
	}

	/**
	 * @param baseName base file name
	 * @return A unique file name
	 */
	public static String createUniqueFileName(String baseName) {
		String uniqueFileName = UUID.randomUUID().toString();
		if (baseName != null) {
			uniqueFileName +=  "." + StringUtils.remove(IOUtils.getFileNameExtension(baseName), ' ');
		}
		return uniqueFileName;
	}
	
	/**
	 * Create a upload file
	 * @param baseName baseName
	 * @return a FileObject
	 * @throws IOException IOException
	 */
	public static FileObject createUploadFile(String baseName) throws IOException {
		String fileName = createUniqueFileName(baseName);
		return getFileSystemManager().resolveFile(getRepository(), fileName);
	}
	
	/**
	 * Create a upload file
	 * @param baseName baseName
	 * @return a FileObject
	 * @throws IOException IOException
	 */
	public static FileObject resolveUploadFile(String baseName) throws IOException {
		return getFileSystemManager().resolveFile(getRepository(), baseName);
	}
}
