/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.exts.xwork2.validator.validators;

import java.io.File;

import nuts.core.lang.StringUtils;
import nuts.exts.fileupload.UploadFile;

import org.apache.commons.vfs.FileObject;
import org.apache.commons.vfs.FileSystemException;
import org.apache.commons.vfs.FileType;

import com.opensymphony.xwork2.validator.ValidationException;

/**
 * FileLengthFieldValidator
 */
public class FileLengthFieldValidator extends AbstractFieldValidator {

	private Long minLength = null;
	private Long maxLength = null;
	private Long length = null;

	/**
	 * @return the maxLength
	 */
	public Long getMaxLength() {
		return maxLength;
	}

	/**
	 * @param maxLength the maxLength to set
	 */
	public void setMaxLength(Long maxLength) {
		this.maxLength = maxLength;
	}

	/**
	 * @return the minLength
	 */
	public Long getMinLength() {
		return minLength;
	}

	/**
	 * @param minLength the minLength to set
	 */
	public void setMinLength(Long minLength) {
		this.minLength = minLength;
	}

	/**
	 * @return the length
	 */
	public Long getLength() {
		return length;
	}

	/**
	 * @return minFileSize
	 */
	public String getMinFileSize() {
		return StringUtils.formatFileSize(minLength);
	}
	
	/**
	 * @return maxFileSize
	 */
	public String getMaxFileSize() {
		return StringUtils.formatFileSize(maxLength);
	}
	
	/**
	 * @return fileSize
	 */
	public String getFileSize() {
		return StringUtils.formatFileSize(length);
	}
	
	/**
	 * @see com.opensymphony.xwork2.validator.Validator#validate(java.lang.Object)
	 */
	public void validate(Object object) throws ValidationException {
		Object value = getFieldValue(getFieldName(), object);

		if (value instanceof UploadFile) {
			value = ((UploadFile)value).getFile();
		}

		if (value == null) {
			return;
		}

		if (value instanceof File) {
			File f = (File)value;

			if (!f.exists() || !f.isFile()) {
				addFieldError(getFieldName(), object);
				return;
			}
			length = f.length();
		}
		else if (value instanceof FileObject) {
			FileObject f = (FileObject)value;

			try {
				if (!f.exists() || !FileType.FILE.equals(f.getType())) {
					addFieldError(getFieldName(), object);
					return;
				}
				length = f.getContent().getSize();
			}
			catch (FileSystemException e) {
				throw new ValidationException("field [" + getFieldName()
						+ "] (" + value.getClass() + ").getSize() failed: "
						+ e.getMessage());
			}
		}
		else {
			throw new ValidationException("field [" + getFieldName() + "] ("
					+ value.getClass() + ") is not a instance of "
					+ File.class.getName() + " / " + FileObject.class.getName());
		}

		// only check for a minimum value if the min parameter is set
		if (minLength != null && length < minLength) {
			addFieldError(getFieldName(), object);
			return;
		}

		// only check for a maximum value if the max parameter is set
		if (maxLength != null && length > maxLength) {
			addFieldError(getFieldName(), object);
			return;
		}
	}
}
