/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.exts.xwork2.converter.converters;

import java.util.Locale;
import java.util.Map;

import nuts.exts.xwork2.util.ContextUtils;
import ognl.DefaultTypeConverter;

import com.opensymphony.xwork2.ActionContext;
import com.opensymphony.xwork2.TextProvider;
import com.opensymphony.xwork2.util.ValueStack;


/**
 * AbstractTypeConverter
 */
public abstract class AbstractTypeConverter extends DefaultTypeConverter  {

	/**
	 * @see ognl.DefaultTypeConverter#convertValue(java.util.Map, java.lang.Object, java.lang.Class)
	 */
	public Object convertValue(Map context, Object o, Class toClass) {
		if (toClass.equals(String.class)) {
			return convertToString(context, o);
		}
		else if (o instanceof String[]) {
			return convertFromString(context, (String[]) o, toClass);
		}
		else if (o instanceof String) {
			return convertFromString(context, new String[] { (String) o }, toClass);
		}
		else {
			return performFallbackConversion(context, o, toClass);
		}
	}

	/**
	 * Hook to perform a fallback conversion if every default options failed. By default
	 * this will ask Ognl's DefaultTypeConverter (of which this class extends) to
	 * perform the conversion.
	 *
	 * @param context
	 * @param o
	 * @param toClass
	 * @return The fallback conversion
	 */
	protected Object performFallbackConversion(Map context, Object o, Class toClass) {
		return super.convertValue(context, o, toClass);
	}


	/**
	 * Converts one or more String values to the specified class.
	 *
	 * @param context the action context
	 * @param values  the String values to be converted, such as those submitted from an HTML form
	 * @param toClass the class to convert to
	 * @return the converted object
	 */
	public abstract Object convertFromString(Map context, String[] values, Class toClass);

	/**
	 * Converts the specified object to a String.
	 *
	 * @param context the action context
	 * @param o       the object to be converted
	 * @return the converted String
	 */
	public abstract String convertToString(Map context, Object o);

	protected String getText(String name) {
		TextProvider tp = getTextProvider();
		return tp.getText(name, (String)null);
	}

	protected String getText(String name, String defaultValue) {
		TextProvider tp = getTextProvider();
		return tp.getText(name, defaultValue);
	}

	protected Locale getLocale() {
		return ContextUtils.getLocale();
	}

	protected ValueStack getValueStack() {
		ActionContext context = ContextUtils.getActionContext();
		if (context == null) {
			return null;
		}
		return context.getValueStack();
	}

	protected TextProvider getTextProvider() {
		ValueStack vs = getValueStack();
		if (vs == null) {
			return null;
		}
		return ContextUtils.findTextProviderInStack(vs);
	}
}
