/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.exts.ibatis.type;

import java.sql.CallableStatement;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import com.ibatis.sqlmap.engine.type.BaseTypeHandler;

/**
 * AbstractBooleanTypeHandler
 */
public abstract class AbstractBooleanTypeHandler extends BaseTypeHandler {
	protected abstract String getTrueText();
	protected abstract String getFalseText();

	/**
	 * @see com.ibatis.sqlmap.engine.type.TypeHandler#setParameter(java.sql.PreparedStatement, int, java.lang.Object, java.lang.String)
	 */
	public void setParameter(PreparedStatement ps, int i, Object parameter, String jdbcType) throws SQLException {
		ps.setString(i, parameter == null ? null : (((Boolean)parameter) ? getTrueText(): getFalseText()));
	}

	protected Boolean convertFromString(String s) throws SQLException {
		if (s == null) {
			return null;
		}
		else if (getTrueText().equalsIgnoreCase(s)) {
			return Boolean.TRUE;
		}
		else if (getFalseText().equalsIgnoreCase(s)) {
			return Boolean.FALSE;
		}
		else {
			throw new SQLException(
				"Unexpected value " + s + " found where " + getTrueText() + " or " + getFalseText() + " was expected.");
		}
	}

	/**
	 * @see com.ibatis.sqlmap.engine.type.TypeHandler#getResult(java.sql.ResultSet, java.lang.String)
	 */
	public Object getResult(ResultSet rs, String columnName) throws SQLException {
		String s = rs.getString(columnName);
		if (rs.wasNull()) {
			return null;
		}
		return convertFromString(s);
	}

	/**
	 * @see com.ibatis.sqlmap.engine.type.TypeHandler#getResult(java.sql.ResultSet, int)
	 */
	public Object getResult(ResultSet rs, int columnIndex) throws SQLException {
		String s = rs.getString(columnIndex);
		if (rs.wasNull()) {
			return null;
		}
		return convertFromString(s);
	}

	/**
	 * @see com.ibatis.sqlmap.engine.type.TypeHandler#getResult(java.sql.CallableStatement, int)
	 */
	public Object getResult(CallableStatement cs, int columnIndex) throws SQLException {
		String s = cs.getString(columnIndex);
		if (cs.wasNull()) {
			return null;
		}
		return convertFromString(s);
	}

	/**
	 * @see com.ibatis.sqlmap.engine.type.TypeHandler#valueOf(java.lang.String)
	 */
	public Object valueOf(String s) {
		if (s == null) {
			return null;
		}
		else if (getTrueText().equalsIgnoreCase(s)) {
			return new Boolean (true);
		}
		else if (getFalseText().equalsIgnoreCase(s)) {
			return new Boolean (false);
		}
		else {
			throw new IllegalArgumentException(
				"Unexpected value " + s + " found where " + getTrueText() + " or " + getFalseText() + " was expected.");
		}
	}
}
