/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.util.converter;

import nuts.core.lang.NumberUtils;



public abstract class PrimitiveTypeConverter<T> extends AbstractConverter<T> {
	public PrimitiveTypeConverter(Class<T> type) {
		super(type);
	}
	
	public static class BooleanConverter extends PrimitiveTypeConverter<Boolean> {
		public BooleanConverter() {
			super(Boolean.class);
		}
		
		@Override
		protected Boolean defaultValue() {
			return false;
		}

		@Override
		protected Boolean convertValue(Object value) {
			if (value instanceof Number) {
				return ((Number)value).intValue() != 0;
			}
			
			String s = value.toString();
			if (s.length() < 1) {
				return false;
			}
			
			switch (s.charAt(0)) {
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9':
			case 'Y':
			case 'y':
			case 'T':
			case 't':
			case 'O':
			case 'o':
				return true;
			default:
				return false;
			}
		}
	}

	public static class ByteConverter extends PrimitiveTypeConverter<Byte> {
		public ByteConverter() {
			super(Byte.class);
		}
		
		@Override
		protected Byte defaultValue() {
			return 0;
		}

		@Override
		protected Byte convertValue(Object value) {
			if (value instanceof Number) {
				return ((Number)value).byteValue();
			}
			
			return NumberUtils.toByte(value.toString(), defaultValue());
		}
	}


	public static class CharacterConverter extends PrimitiveTypeConverter<Character> {
		public CharacterConverter() {
			super(Character.class);
		}
		
		@Override
		protected Character defaultValue() {
			return 0;
		}

		@Override
		protected Character convertValue(Object value) {
			if (value instanceof Character) {
				return (Character)value;
			}
			
			if (value instanceof Number) {
				return (char)((Number)value).intValue();
			}

			String s = value.toString();
			if (s.length() > 0) {
				return s.charAt(0);
			}
			
			return defaultValue();
		}
	}

	public static class DoubleConverter extends PrimitiveTypeConverter<Double> {
		public DoubleConverter() {
			super(Double.class);
		}
		
		@Override
		protected Double defaultValue() {
			return (double)0;
		}

		@Override
		protected Double convertValue(Object value) {
			if (value instanceof Number) {
				return ((Number)value).doubleValue();
			}
			
			return NumberUtils.toDouble(value.toString(), defaultValue());
		}
	}

	public static class FloatConverter extends PrimitiveTypeConverter<Float> {
		public FloatConverter() {
			super(Float.class);
		}
		
		@Override
		protected Float defaultValue() {
			return (float)0;
		}

		@Override
		protected Float convertValue(Object value) {
			if (value instanceof Number) {
				return ((Number)value).floatValue();
			}
			
			return NumberUtils.toFloat(value.toString(), defaultValue());
		}
	}

	public static class IntegerConverter extends PrimitiveTypeConverter<Integer> {
		public IntegerConverter() {
			super(Integer.class);
		}
		
		@Override
		protected Integer defaultValue() {
			return 0;
		}

		@Override
		protected Integer convertValue(Object value) {
			if (value instanceof Number) {
				return ((Number)value).intValue();
			}
			
			return NumberUtils.toInt(value.toString(), defaultValue());
		}
	}

	public static class LongConverter extends PrimitiveTypeConverter<Long> {
		public LongConverter() {
			super(Long.class);
		}
		
		@Override
		protected Long defaultValue() {
			return 0L;
		}

		@Override
		protected Long convertValue(Object value) {
			if (value instanceof Number) {
				return ((Number)value).longValue();
			}
			
			return NumberUtils.toLong(value.toString(), defaultValue());
		}
	}

	public static class ShortConverter extends PrimitiveTypeConverter<Short> {
		public ShortConverter() {
			super(Short.class);
		}
		
		@Override
		protected Short defaultValue() {
			return 0;
		}

		@Override
		protected Short convertValue(Object value) {
			if (value instanceof Number) {
				return ((Number)value).shortValue();
			}
			
			return NumberUtils.toShort(value.toString(), defaultValue());
		}
	}


}
