/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.util.comparator;

import java.util.Comparator;

/**
 * abstract comparator
 * @param <T> the type of objects that may be compared by this comparator
 */
public abstract class AbstractComparator<T> implements Comparator<T> {
	
	protected boolean descend;

	/**
	 * constructor
	 * descend = false
	 */
	public AbstractComparator() {
		this(false);
	}

	/**
	 * constructor
	 * @param descend descend
	 */
	public AbstractComparator(boolean descend) {
		this.descend = descend;
	}
	
	/**
	 * @return the descend
	 */
	public boolean isDescend() {
		return descend;
	}

	/**
	 * @param descend the descend to set
	 */
	public void setDescend(boolean descend) {
		this.descend = descend;
	}

	/**
     * @param o1 the first object to be compared.
     * @param o2 the second object to be compared.
     * @return a negative integer, zero, or a positive integer as the
     * 	       first argument is less than, equal to, or greater than the
     *	       second.
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(T o1, T o2) {
		return descend ? _compare(o2, o1) : _compare(o1, o2);
	}

	/**
     * @param o1 the first object to be compared.
     * @param o2 the second object to be compared.
     * @return a negative integer, zero, or a positive integer as the
     * 	       first argument is less than, equal to, or greater than the
     *	       second.
	 */
	protected abstract int _compare(T o1, T o2);

	/**
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return getClass().getName() + (descend ? "(descend)@" : "") + Integer.toHexString(hashCode());
	}
}
