/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.sql.criterion;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import nuts.core.orm.restriction.Restrictions;
import nuts.core.sql.SqlConstants;
import nuts.core.sql.SqlUtils;

/**
 */
@SuppressWarnings("serial")
public class SqlRestrictions implements Restrictions, Cloneable, Serializable {

	private String conjunction = AND;
	
	private List<AbstractExpression> expressions;

	/**
	 * constructor
	 */
	public SqlRestrictions() {
		expressions = new ArrayList<AbstractExpression>();
	}
    
	/**
	 * @return conjunction
	 */
	public String getConjunction() {
		return conjunction;
	}

	/**
	 * @param conjunction the conjunction to set
	 */
	public void setConjunction(String conjunction) {
		if (conjunction == null) {
			throw new IllegalArgumentException(
					"the conjunction is required; it can not be null");
		}

		conjunction = conjunction.toUpperCase();
		if (!AND.equals(conjunction) && !OR.equals(conjunction)) {
			throw new IllegalArgumentException("Invalid conjunction ["
					+ conjunction + "], must be AND/OR");
		}

		this.conjunction = conjunction;
	}

	/**
	 * setConjunctionToAnd
	 */
	public void setConjunctionToAnd() {
		this.conjunction = AND;
	}

	/**
	 * setConjunctionToOr
	 */
	public void setConjunctionToOr() {
		this.conjunction = OR;
	}

	/**
	 * @return expressions
	 */
	public List<AbstractExpression> getExpressions() {
		return expressions;
	}

	/**
	 * @param expressions the expressions to set
	 */
	public void setExpressions(List<AbstractExpression> expressions) {
		this.expressions = expressions;
	}

	/**
	 * isEmpty
	 * @return true/false
	 */
	public boolean isEmpty() {
		return expressions.isEmpty();
	}
	
	/**
	 * clear
	 */
	public void clear() {
		conjunction = AND;
		expressions.clear();
	}

	private SqlRestrictions conjunction(String conjunction, boolean force) {
		if (isEmpty()) {
			if (force) {
				throw new IllegalArgumentException("Explicitly appending logical operator " + conjunction + " to empty filters is not allowed.");
			}
		}
		else {
			AbstractExpression last = expressions.get(expressions.size() - 1); 
			if (last instanceof ParenExpression) {
				ParenExpression ce = (ParenExpression)last;
				if (SqlConstants.CLOSE_PAREN.equals(ce.getOperator())) {
					return addConjunctionExpression(conjunction);
				}
				else {
					if (force) {
						throw new IllegalArgumentException("Explicitly appending explicit logical operator " + conjunction + " after '" + ce.getOperator() + "' is not allowed.");
					}
				}
			}
			else if (last instanceof ConjunctionExpression) {
				if (force) {
					throw new IllegalArgumentException("Explicitly appending logical operator " + conjunction + " after '" + ((ConjunctionExpression)last).getOperator() + "' is not allowed.");
				}
				// 'and' 'or' already appended, so skip
				return this;
			}
			else {
				return addConjunctionExpression(conjunction);
			}
		}
		return this;
	}

	private SqlRestrictions addParenExpression(String operator) {
		expressions.add(new ParenExpression(operator));
		return this;
	}

	private SqlRestrictions addConjunctionExpression(String operator) {
		expressions.add(new ConjunctionExpression(operator));
		return this;
	}

	private SqlRestrictions addSimpleExpression(String column, String operator) {
		conjunction();
		expressions.add(new SimpleExpression(column, operator));
		return this;
	}

	private SqlRestrictions addCompareValueExpression(String column, String operator, Object compareValue) {
		conjunction();
		expressions.add(new CompareValueExpression(column, operator, compareValue));
		return this;
	}

	private SqlRestrictions addCompareColumnExpression(String column, String operator, String compareColumn) {
		conjunction();
		expressions.add(new CompareColumnExpression(column, operator, compareColumn));
		return this;
	}

	private SqlRestrictions addCompareCollectionExpression(String column, String operator, Object[] values) {
		conjunction();
		expressions.add(new CompareCollectionExpression(column, operator, values));
		return this;
	}

	private SqlRestrictions addCompareCollectionExpression(String column, String operator, Collection values) {
		conjunction();
		expressions.add(new CompareCollectionExpression(column, operator, values));
		return this;
	}

	private SqlRestrictions addCompareRanageExpression(String column, String operator, Object minValue, Object maxValue) {
		conjunction();
		expressions.add(new CompareRangeExpression(column, operator, minValue, maxValue));
		return this;
	}

	/**
	 * add conjunction expression
	 * @return this
	 */
	private SqlRestrictions conjunction() {
		return conjunction(conjunction, false);
	}
	
	/**
	 * add AND expression 
	 * @return this
	 */
	public SqlRestrictions and() {
		return conjunction(AND, true);
	}
	
	/**
	 * add OR expression 
	 * @return this
	 */
	public SqlRestrictions or() {
		return conjunction(OR, true);
	}
	
	/**
	 * add open paren ( 
	 * @return this
	 */
	public SqlRestrictions open() {
		conjunction();
		return addParenExpression(SqlConstants.OPEN_PAREN);
	}
	
	/**
	 * add close paren ) 
	 * @return this
	 */
	public SqlRestrictions close() {
		return addParenExpression(SqlConstants.CLOSE_PAREN);
	}
	
	/**
	 * add "column IS NULL" expression
	 * @param column column 
	 * @return this
	 */
	public SqlRestrictions isNull(String column) {
		return addSimpleExpression(column, SqlConstants.IS_NULL);
	}

	/**
	 * add "column IS NOT NULL" expression 
	 * @param column column 
	 * @return this
	 */
	public SqlRestrictions isNotNull(String column) {
		return addSimpleExpression(column, SqlConstants.IS_NOT_NULL);
	}

	/**
	 * add "column = value" expression
	 * @param column column 
	 * @param value value
	 * @return this
	 */
	public SqlRestrictions equalTo(String column, Object value) {
		return addCompareValueExpression(column, SqlConstants.EQUAL, value);
	}

	/**
	 * add "column &lt;&gt; value" expression
	 * @param column column 
	 * @param value value
	 * @return this
	 */
	public SqlRestrictions notEqualTo(String column, Object value) {
		return addCompareValueExpression(column, SqlConstants.NOT_EQUAL, value);
	}

	/**
	 * add "column &gt; value" expression
	 * @param column column 
	 * @param value value
	 * @return this
	 */
	public SqlRestrictions greaterThan(String column, Object value) {
		return addCompareValueExpression(column, SqlConstants.GREAT_THAN, value);
	}

	/**
	 * add "column %gt;= value" expression
	 * @param column column 
	 * @param value value
	 * @return this
	 */
	public SqlRestrictions greaterThanOrEqualTo(String column, Object value) {
		return addCompareValueExpression(column, SqlConstants.GREAT_EQUAL, value);
	}

	/**
	 * add "column &lt; value" expression
	 * @param column column 
	 * @param value value
	 * @return this
	 */
	public SqlRestrictions lessThan(String column, Object value) {
		return addCompareValueExpression(column, SqlConstants.LESS_THAN, value);
	}

	/**
	 * add "column &lt;= value" expression
	 * @param column column 
	 * @param value value
	 * @return this
	 */
	public SqlRestrictions lessThanOrEqualTo(String column, Object value) {
		return addCompareValueExpression(column, SqlConstants.LESS_EQUAL, value);
	}

	/**
	 * add "column LIKE value" expression
	 * @param column column 
	 * @param value value
	 * @return this
	 */
	public SqlRestrictions like(String column, Object value) {
		return addCompareValueExpression(column, SqlConstants.LIKE, value);
	}

	/**
	 * add "column LIKE %value%" expression
	 * @param column column 
	 * @param value value
	 * @return this
	 */
	public SqlRestrictions match(String column, Object value) {
		return like(column, SqlUtils.stringLike(value.toString()));
	}

	/**
	 * add "column LIKE value%" expression
	 * @param column column 
	 * @param value value
	 * @return this
	 */
	public SqlRestrictions leftMatch(String column, Object value) {
		return like(column, SqlUtils.startsLike(value.toString()));
	}

	/**
	 * add "column LIKE %value" expression
	 * @param column column 
	 * @param value value
	 * @return this
	 */
	public SqlRestrictions rightMatch(String column, Object value) {
		return like(column, SqlUtils.endsLike(value.toString()));
	}

	/**
	 * add "column NOT LIKE value" expression
	 * @param column column 
	 * @param value value
	 * @return this
	 */
	public SqlRestrictions notLike(String column, Object value) {
		return addCompareValueExpression(column, SqlConstants.NOT_LIKE, value);
	}

	/**
	 * add "column = compareColumn" expression
	 * @param column column 
	 * @param compareColumn column to compare
	 * @return this
	 */
	public SqlRestrictions equalToColumn(String column, String compareColumn) {
		return addCompareColumnExpression(column, SqlConstants.EQUAL, compareColumn);
	}

	/**
	 * add "column &lt;&gt; compareColumn" expression
	 * @param column column 
	 * @param compareColumn column to compare
	 * @return this
	 */
	public SqlRestrictions notEqualToColumn(String column, String compareColumn) {
		return addCompareColumnExpression(column, SqlConstants.NOT_EQUAL, compareColumn);
	}

	/**
	 * add "column %gt; compareColumn" expression
	 * @param column column 
	 * @param compareColumn column to compare
	 * @return this
	 */
	public SqlRestrictions greaterThanColumn(String column, String compareColumn) {
		return addCompareColumnExpression(column, SqlConstants.GREAT_THAN, compareColumn);
	}

	/**
	 * add "column &gt;= compareColumn" expression
	 * @param column column 
	 * @param compareColumn column to compare
	 * @return this
	 */
	public SqlRestrictions greaterThanOrEqualToColumn(String column, String compareColumn) {
		return addCompareColumnExpression(column, SqlConstants.GREAT_EQUAL, compareColumn);
	}

	/**
	 * add "column &lt; compareColumn" expression
	 * @param column column 
	 * @param compareColumn column to compare
	 * @return this
	 */
	public SqlRestrictions lessThanColumn(String column, String compareColumn) {
		return addCompareColumnExpression(column, SqlConstants.LESS_THAN, compareColumn);
	}

	/**
	 * add "column %lt;= compareColumn" expression
	 * @param column column 
	 * @param compareColumn column to compare
	 * @return this
	 */
	public SqlRestrictions lessThanOrEqualToColumn(String column, String compareColumn) {
		return addCompareColumnExpression(column, SqlConstants.LESS_EQUAL, compareColumn);
	}

	/**
	 * add "column IN (value1, value2 ...)" expression
	 * @param column column 
	 * @param values values
	 * @return this
	 */
	public SqlRestrictions in(String column, Object[] values) {
		return addCompareCollectionExpression(column, SqlConstants.IN, values);
	}

	/**
	 * add "column IN (value1, value2 ...)" expression
	 * @param column column 
	 * @param values values
	 * @return this
	 */
	public SqlRestrictions in(String column, Collection values) {
		return addCompareCollectionExpression(column, SqlConstants.IN, values);
	}

	/**
	 * add "column NOT IN (value1, value2 ...)" expression
	 * @param column column 
	 * @param values values
	 * @return this
	 */
	public SqlRestrictions notIn(String column, Object[] values) {
		return addCompareCollectionExpression(column, SqlConstants.NOT_IN, values);
	}

	/**
	 * add "column NOT IN (value1, value2 ...)" expression
	 * @param column column 
	 * @param values values
	 * @return this
	 */
	public SqlRestrictions notIn(String column, Collection values) {
		return addCompareCollectionExpression(column, SqlConstants.NOT_IN, values);
	}

	/**
	 * add "column BETWEEN (value1, value2)" expression
	 * @param column column 
	 * @param value1 value from
	 * @param value2 value to
	 * @return this
	 */
	public SqlRestrictions between(String column, Object value1, Object value2) {
		return addCompareRanageExpression(column, SqlConstants.BETWEEN, value1, value2);
	}

	/**
	 * add "column NOT BETWEEN (value1, value2)" expression
	 * @param column column 
	 * @param value1 value from
	 * @param value2 value to
	 * @return this
	 */
	public SqlRestrictions notBetween(String column, Object value1, Object value2) {
		return addCompareRanageExpression(column, SqlConstants.NOT_BETWEEN, value1, value2);
	}

	/**
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((conjunction == null) ? 0 : conjunction.hashCode());
		result = prime * result + ((expressions == null) ? 0 : expressions.hashCode());
		return result;
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		SqlRestrictions other = (SqlRestrictions) obj;
		if (conjunction == null) {
			if (other.conjunction != null)
				return false;
		}
		else if (!conjunction.equals(other.conjunction))
			return false;
		if (expressions == null) {
			if (other.expressions != null)
				return false;
		}
		else if (!expressions.equals(other.expressions))
			return false;
		return true;
	}

	/**
	 * Clone
	 * @throws CloneNotSupportedException if clone not supported
	 * @return Clone Object
	 */
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}

	/**
     * @return  a string representation of the object.
	 */
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();

		sb.append("{ ");
		sb.append("conjunction: ").append(conjunction);
		sb.append(", ");
		sb.append("expressions: ").append(expressions);
		sb.append(" }");

		return sb.toString();
	}

}
