/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.sql;

import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * SqlLogger
 */
public abstract class SqlLogger {
	/**
	 * log for java.sql.Statement
	 */
	private static Log pslog = LogFactory.getLog(Statement.class);

	/**
	 * log for java.sql.ResultSet
	 */
	private static Log rslog = LogFactory.getLog(ResultSet.class);

	/**
	 * logResultHeader
	 * @param resultSet result set 
	 */
	public static void logResultHeader(ResultSet resultSet) {
		if (rslog.isDebugEnabled()) {
			try {
				ResultSetMetaData meta = resultSet.getMetaData();
				
				StringBuilder sb = new StringBuilder();
				sb.append("Result Header: [");
				int cnt = meta.getColumnCount();
				for (int i = 1; i <= cnt; i++) {
					sb.append(meta.getColumnLabel(i));
					if (i < cnt) {
						sb.append(", ");
					}
				}
				sb.append("]");
				rslog.debug(sb.toString());
			}
			catch (SQLException e) {
				rslog.warn("LOG ERROR", e);
			}
		}
	}
	
	/**
	 * logResultValues
	 * @param resultSet result set 
	 */
	public static void logResultValues(ResultSet resultSet) {
		if (rslog.isDebugEnabled()) {
			try {
				ResultSetMetaData meta = resultSet.getMetaData();
	
				StringBuilder sb = new StringBuilder();
				sb.append("Result Values: [");
				int cnt = meta.getColumnCount();
				for (int i = 1; i <= cnt; i++) {
					sb.append(String.valueOf(resultSet.getObject(i)));
					if (i < cnt) {
						sb.append(", ");
					}
				}
				sb.append("]");
				rslog.debug(sb.toString());
			}
			catch (SQLException e) {
				rslog.warn("LOG ERROR", e);
			}
		}		
	}

	/**
	 * logSatement
	 * @param sql sql statement 
	 */
	public static void logSatement(String sql) {
		if (pslog.isDebugEnabled()) {
			pslog.debug("Statement: " + sql);
		}
	}
	
	/**
	 * logParameters
	 * @param ps PreparedStatement
	 */
	public static void logParameters(PreparedStatement ps) {
		if (pslog.isDebugEnabled()) {
			try {
				ParameterMetaData pmd = ps.getParameterMetaData();

				if (pmd.getParameterCount() > 0) {
					StringBuilder sb = new StringBuilder();
					sb.append("Parameters: [");
		
					for (int i = 1; i <= pmd.getParameterCount(); i++) {
						if (i > 1) {
							sb.append(", ");
						}
						try {
							sb.append(pmd.getParameterClassName(i));
						}
						catch (SQLException e) {
							sb.append('?');
						}
					}
		
					sb.append("]");
					pslog.debug(sb.toString());
				}
			}
			catch (SQLException e) {
				pslog.warn("LOG ERROR", e);
			}
		}
	}
}
