/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.resource;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;

/**
 * A class for load external resource.
 */
public class ExternalResourceLoader16 extends ExternalResourceLoader {
	private Control control;
	
	/**
	 * Constructor
	 */
	public ExternalResourceLoader16() {
		control = new Control(this);
	}

	public ResourceBundle getBundle(String baseName) {
		return ResourceBundle.getBundle(baseName, control);
	}

	public ResourceBundle getBundle(String baseName, Locale locale) {
		return ResourceBundle.getBundle(baseName, locale, control);
	}

	public ResourceBundle getBundle(String baseName, Locale locale, ClassLoader loader) {
		return ResourceBundle.getBundle(baseName, locale, loader, control);
	}

	public static class Control extends ResourceBundle.Control {
		/**
		 * The default format <code>List</code>, which contains the strings
		 * <code>"nuts.ext"</code> and <code>"java.class"</code> and <code>"java.properties"</code>, in
		 * this order. This <code>List</code> is {@linkplain
		 * Collections#unmodifiableList(List) unmodifiable}.
		 */
		public static final List<String> FORMAT_DEFAULT
		    = Collections.unmodifiableList(Arrays.asList(
		    	"nuts.ext", "java.class", "java.properties"));

		private ExternalResourceLoader externalResourceLoader;
		
		public Control(ExternalResourceLoader externalResourceLoader) {
			this.externalResourceLoader = externalResourceLoader;
		}
		
		public List<String> getFormats(String baseName) {
			if (baseName == null) {
				throw new NullPointerException();
			}
			return FORMAT_DEFAULT;
		}

		@SuppressWarnings("unchecked")
		public ResourceBundle newBundle(String baseName, Locale locale, String format,
				ClassLoader loader, boolean reload) throws IllegalAccessException,
				InstantiationException, IOException {
			if (baseName == null || locale == null || format == null || loader == null) {
				throw new NullPointerException();
			}

			if (format.equals("nuts.ext")) {
				Map contents = externalResourceLoader.getContentsMap(toBundleName(baseName, locale));
				if (contents != null) {
					MapResourceBundle bundle = new MapResourceBundle(contents);
					return bundle;
				}
			}
			
			return super.newBundle(baseName, locale, format, loader, reload);
		}
	}
}
