/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.oxm.adapter;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Attr;
import org.w3c.dom.DOMException;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.TypeInfo;

/**
 * ProxyElementAdapter is a pass-through adapter for objects which already
 * implement the Element interface. All methods are proxied to the underlying
 * Node except getParent(), getNextSibling() and getPreviousSibling(), which are
 * implemented by the abstract adapter node to work with the parent adapter.
 * 
 * Note: this class wants to be (extend) both an AbstractElementAdapter and
 * ProxyElementAdapter, but its proxy-ness is winning right now.
 */
public class ProxyElementAdapter extends ProxyNodeAdapter implements Element {
	private static Log log = LogFactory.getLog(ProxyElementAdapter.class);

	public ProxyElementAdapter(AdapterFactory factory, NodeAdapter parent,
			Element value) {
		super(factory, parent, value);
	}

	/**
	 * Get the proxied Element
	 */
	protected Element element() {
		return (Element)getPropertyValue();
	}

	protected List<Node> buildChildAdapters() {
		List<Node> adapters = new ArrayList<Node>();
		NodeList children = node().getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			Node child = children.item(i);
			Node adapter = wrap(child);
			if (adapter != null) {
				if (log.isDebugEnabled()) {
					log.debug("wrapped child node: " + child.getNodeName());
				}
				adapters.add(adapter);
			}
		}
		return adapters;
	}

	// Proxied Element methods

	public String getTagName() {
		return element().getTagName();
	}

	public boolean hasAttribute(String name) {
		return element().hasAttribute(name);
	}

	public String getAttribute(String name) {
		return element().getAttribute(name);
	}

	public boolean hasAttributeNS(String namespaceURI, String localName) {
		return element().hasAttributeNS(namespaceURI, localName);
	}

	public Attr getAttributeNode(String name) {
		if (log.isDebugEnabled()) {
			log.debug("wrapping attribute");
		}
		return (Attr)wrap(element().getAttributeNode(name));
	}

	// I'm overriding this just for clarity. The base impl is correct.
	public NodeList getElementsByTagName(String name) {
//		return super.getElementsByTagName(name);
		return element().getElementsByTagName(name);
	}

	public NodeList getElementsByTagNameNS(String namespaceURI, String localName)
			throws DOMException {
		return element().getElementsByTagNameNS(namespaceURI, localName);
	}
	public String getAttributeNS(String namespaceURI, String localName) {
		return element().getAttributeNS(namespaceURI, localName);
	}

	public Attr getAttributeNodeNS(String namespaceURI, String localName) {
		return (Attr)wrap(element().getAttributeNodeNS(namespaceURI, localName));
	}

	public void removeAttribute(String name) throws DOMException {
		element().removeAttribute(name);
	}

	public void removeAttributeNS(String namespaceURI, String localName)
			throws DOMException {
		element().removeAttributeNS(namespaceURI, localName);
	}

	public void setAttribute(String name, String value) throws DOMException {
		element().setAttribute(name, value);
	}

	public Attr removeAttributeNode(Attr oldAttr) throws DOMException {
		return element().removeAttributeNode(oldAttr);
	}

	public Attr setAttributeNode(Attr newAttr) throws DOMException {
		return element().setAttributeNode(newAttr);
	}

	public Attr setAttributeNodeNS(Attr newAttr) throws DOMException {
		return element().setAttributeNodeNS(newAttr);
	}

	public void setAttributeNS(String namespaceURI, String qualifiedName,
			String value) throws DOMException {
		element().setAttributeNS(namespaceURI, qualifiedName, value);
	}

	// DOM level 3 methods
	public TypeInfo getSchemaTypeInfo() {
		return element().getSchemaTypeInfo();
	}

	public void setIdAttribute(String name, boolean isId) throws DOMException {
		element().setIdAttribute(name, isId);
	}

	public void setIdAttributeNS(String namespaceURI, String localName, boolean isId)
			throws DOMException {
		element().setIdAttributeNS(namespaceURI, localName, isId);
	}

	public void setIdAttributeNode(Attr idAttr, boolean isId) throws DOMException {
		element().setIdAttributeNode(idAttr, isId);
	}
	// end DOM level 3 methods
	// end proxied Element methods

	public String toString() {
		return "ProxyElement for: " + element();
	}
}
