/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.oxm.adapter;

import java.util.ArrayList;
import java.util.List;

import org.w3c.dom.Node;

/**
 * This class is the most general type of adapter, utilizing reflective
 * introspection to present a DOM view of all of the public properties of its
 * value. For example, a property returning a JavaBean such as:
 * 
 * <pre>
 * public Person getMyPerson() { ... }
 * ...
 * class Person {
 *      public String getFirstName();
 *      public String getLastName();
 * }
 * </pre>
 * 
 * would be rendered as: <myPerson> <firstName>...</firstName>
 * <lastName>...</lastName> </myPerson>
 */
public class DomElementAdapter extends AbstractElementAdapter {
	/**
	 * Constructors
	 */
	public DomElementAdapter() {
	}

	public DomElementAdapter(AdapterFactory adapterFactory, NodeAdapter parent,
			String propertyName, Object value) {
		setContext(adapterFactory, parent, propertyName, value);
	}

	protected Node node() {
		return (Node)getPropertyValue();
	}
	
	@Override
	protected List<Node> buildAttrAdapters() {
		Node node = getAdapterFactory().proxyNode(this, node());
		if (node == null || node.getNodeType() != Node.ATTRIBUTE_NODE) {
			return new ArrayList<Node>(0);
		}

		List<Node> children = new ArrayList<Node>(1);
		children.add(node);
		return children;
	}

	@Override
	protected List<Node> buildChildAdapters() {
		Node node = getAdapterFactory().proxyNode(this, node());
		if (node == null || node.getNodeType() == Node.ATTRIBUTE_NODE) {
			return new ArrayList<Node>(0);
		}

		List<Node> children = new ArrayList<Node>(1);
		children.add(node);
		return children;
	}
}
