/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.oxm.adapter;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.w3c.dom.Attr;
import org.w3c.dom.DOMException;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.TypeInfo;

/**
 * AbstractAdapterElement extends the abstract Node type and implements the DOM
 * Element interface.
 */
public abstract class AbstractElementAdapter extends AbstractNodeAdapter implements Element {

	private List<Node> attrAdapters;

	public AbstractElementAdapter() {
	}

	public String getNodeName() {
		return getTagName();
	}

	public short getNodeType() {
		return Node.ELEMENT_NODE;
	}

	public String getTagName() {
		return getPropertyName();
	}

	public NodeList getElementsByTagNameNS(String namespaceURI, String localName)
			throws DOMException {
		return getElementsByTagName(localName);
	}

	/**
	 * Lazily initialize attribute adapters
	 */
	protected List<Node> getAttrAdapters() {
		if (attrAdapters == null) {
			attrAdapters = buildAttrAdapters();
		}
		return attrAdapters;
	}

	/**
	 * subclasses override to produce their attributes
	 * 
	 * @return List of attribute adapters.
	 */
	protected List<Node> buildAttrAdapters() {
		return new ArrayList<Node>();
	}

	/**
	 * reset attr adapters
	 */
	protected void resetAttrAdapters() {
		attrAdapters = null;
	}

	public NamedNodeMap getAttributes() {
		return new SimpleAttrMap(this, getAttrAdapters());
	}

	public boolean hasAttributes() {
		return CollectionUtils.isNotEmpty(getAttrAdapters());
	}

	/**
	 * No attributes, return empty attributes if asked.
	 */
	public String getAttribute(String name) {
		NamedNodeMap nnm = getAttributes();
		Node n = nnm.getNamedItem(name);
		if (n == null) {
			return "";
		}
		return n.getNodeValue();
	}

	public void setAttribute(String name, String value) throws DOMException {
		throw new UnsupportedOperationException("setAttribute");
	}

	public void setAttributeNS(
			String namespaceURI, 
			String qualifiedName, 
			String value)
			throws DOMException {
		setAttribute(qualifiedName, value);
	}

	public String getAttributeNS(String namespaceURI, String localName) {
		return getAttribute(localName);
	}

	public Attr setAttributeNode(Attr attr) throws DOMException {
		throw new UnsupportedOperationException("setAttributeNode");
	}

	public Attr getAttributeNode(String name) {
		return (Attr)getAttributes().getNamedItem(name);
	}

	public Attr setAttributeNodeNS(Attr attr) throws DOMException {
		throw new UnsupportedOperationException("setAttributeNodeNS");
	}

	public Attr getAttributeNodeNS(String namespaceURI, String localName) {
		return getAttributeNode(localName);
	}

	public boolean hasAttribute(String name) {
		NamedNodeMap nnm = getAttributes();
		Node n = nnm.getNamedItem(name);
		return n != null;
	}

	public boolean hasAttributeNS(String namespaceURI, String localName) {
		return hasAttribute(localName);
	}

	public void removeAttribute(String name) throws DOMException {
		throw new UnsupportedOperationException("removeAttribute");
	}

	public void removeAttributeNS(String namespaceURI, String localName)
			throws DOMException {
		throw new UnsupportedOperationException("removeAttributeNS");
	}

	public Attr removeAttributeNode(Attr attr) throws DOMException {
		throw new UnsupportedOperationException("removeAttributeNode");
	}

	public void setIdAttributeNode(Attr attr, boolean b) throws DOMException {
		throw new UnsupportedOperationException("setIdAttributeNode");
	}

	public TypeInfo getSchemaTypeInfo() {
		throw new UnsupportedOperationException("getSchemaTypeInfo");
	}

	public void setIdAttribute(String name, boolean b) throws DOMException {
		throw new UnsupportedOperationException("setIdAttribute");
	}

	public void setIdAttributeNS(String namespaceURI, String localName, boolean b)
			throws DOMException {
		throw new UnsupportedOperationException("setIdAttributeNS");
	}

}
