/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.orm.type;

import java.sql.Types;
import java.util.HashMap;
import java.util.Map;

/**
 * JdbcTypeRegistry
 */
public abstract class JdbcTypeRegistry {
	private static final Map<String, Integer> NAME_MAP = new HashMap<String, Integer>();

	private static final Map<Integer, String> TYPE_MAP = new HashMap<Integer, String>();

	static {
		initializeTypes();
	}

	/**
	 * set type
	 * 
	 * @param name name
	 * @param type type
	 */
	public static void setType(String name, Integer type) {
		NAME_MAP.put(name, type);
		TYPE_MAP.put(type, name);
	}

	/**
	 * Looks up a type by name, and returns it's int value (from java.sql.Types)
	 * 
	 * @param name - the type name
	 * @return - the int value (from java.sql.Types)
	 */
	public static Integer getType(String name) {
		return NAME_MAP.get(name);
	}

	/**
	 * Looks up a type by name, and returns it's int value (from java.sql.Types)
	 * 
	 * @param type - the type
	 * @return - the int value (from java.sql.Types)
	 */
	public static String getType(Integer type) {
		return TYPE_MAP.get(type);
	}

	private static void initializeTypes() {
		setType("ARRAY", Types.ARRAY);
		setType("BIGINT", Types.BIGINT);
		setType("BINARY", Types.BINARY);
		setType("BIT", Types.BIT);
		setType("BLOB", Types.BLOB);
		setType("BOOLEAN", Types.BOOLEAN);
		setType("CHAR", Types.CHAR);
		setType("CLOB", Types.CLOB);
		setType("DATALINK", Types.DATALINK);
		setType("DATE", Types.DATE);
		setType("DECIMAL", Types.DECIMAL);
		setType("DISTINCT", Types.DISTINCT);
		setType("DOUBLE", Types.DOUBLE);
		setType("FLOAT", Types.FLOAT);
		setType("INTEGER", Types.INTEGER);
		setType("JAVA_OBJECT", Types.JAVA_OBJECT);
		setType("LONGVARBINARY", Types.LONGVARBINARY);
		setType("LONGVARCHAR", Types.LONGVARCHAR);
		setType("NULL", Types.NULL);
		setType("NUMERIC", Types.NUMERIC);
		setType("OTHER", Types.OTHER);
		setType("REAL", Types.REAL);
		setType("REF", Types.REF);
		setType("SMALLINT", Types.SMALLINT);
		setType("STRUCT", Types.STRUCT);
		setType("TIME", Types.TIME);
		setType("TIMESTAMP", Types.TIMESTAMP);
		setType("TINYINT", Types.TINYINT);
		setType("VARBINARY", Types.VARBINARY);
		setType("VARCHAR", Types.VARCHAR);
		setType("ORACLECURSOR", -10);
	}

}
