/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.orm.type;

import nuts.core.lang.JsonUtils;
import nuts.core.lang.StringUtils;

import java.sql.CallableStatement;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

/**
 * base collection implementation of TypeHandler
 */
public abstract class CollectionTypeHandler implements TypeHandler {
	@SuppressWarnings("unchecked")
	protected Object toCollection(String v, Class c) throws SQLException {
		if (StringUtils.isEmpty(v)) {
			try {
				return c.newInstance();
			}
			catch (InstantiationException e) {
				throw new SQLException(e);
			}
			catch (IllegalAccessException e) {
				throw new SQLException(e);
			}
		}
		else {
			try {
				return JsonUtils.fromJson(v, c);
			}
			catch (Exception e) {
				throw new SQLException(e);
			}
		}
	}

	protected abstract Object toCollection(String v) throws SQLException;
	
	protected String toString(Object v) throws SQLException {
		try {
			return JsonUtils.toJson(v);
		}
		catch (Exception e) {
			throw new SQLException(e);
		}
	}

	/**
	 * Gets a column from a result set
	 * 
	 * @param rs - the result set
	 * @param columnName - the column name to get
	 * @return - the column value
	 * @throws SQLException if getting the value fails
	 */
	public Object getResult(ResultSet rs, String columnName)
			throws SQLException {
		String s = rs.getString(columnName);
		if (rs.wasNull()) {
			return null;
		}
		else {
			return toCollection(s);
		}
	}

	/**
	 * Gets a column from a result set
	 * 
	 * @param rs - the result set
	 * @param columnIndex - the column to get (by index)
	 * @return - the column value
	 * @throws SQLException if getting the value fails
	 */
	public Object getResult(ResultSet rs, int columnIndex) throws SQLException {
		String s = rs.getString(columnIndex);
		if (rs.wasNull()) {
			return null;
		}
		else {
			return toCollection(s);
		}
	}

	/**
	 * Gets a column from a callable statement
	 * 
	 * @param cs - the statement
	 * @param columnIndex - the column to get (by index)
	 * @return - the column value
	 * @throws SQLException if getting the value fails
	 */
	public Object getResult(CallableStatement cs, int columnIndex)
			throws SQLException {
		String s = cs.getString(columnIndex);
		if (cs.wasNull()) {
			return null;
		}
		else {
			return toCollection(s);
		}
	}

	/**
	 * Update column value to result set
	 * 
	 * @param rs - the result set
	 * @param columnName - the column name to get
	 * @param value - the value to update
	 * @param jdbcType - the JDBC type of the parameter
	 * @throws SQLException if getting the value fails
	 */
	public void updateResult(ResultSet rs, String columnName, Object value, String jdbcType)
			throws SQLException {
		if (value == null) {
			rs.updateNull(columnName);
		}
		else {
			rs.updateString(columnName, toString(value));
		}
	}

	/**
	 * Update column value to result set
	 * 
	 * @param rs - the result set
	 * @param columnIndex - the column to get (by index)
	 * @param value - the value to update
	 * @param jdbcType - the JDBC type of the parameter
	 * @throws SQLException if getting the value fails
	 */
	public void updateResult(ResultSet rs, int columnIndex, Object value, String jdbcType)
			throws SQLException {
		if (value == null) {
			rs.updateNull(columnIndex);
		}
		else {
			rs.updateString(columnIndex, toString(value));
		}
	}

	/**
	 * Sets a parameter on a prepared statement
	 * 
	 * @param ps - the prepared statement
	 * @param i - the parameter index
	 * @param parameter - the parameter value
	 * @param jdbcType - the JDBC type of the parameter
	 * @throws SQLException if setting the parameter fails
	 */
	public void setParameter(PreparedStatement ps, int i, Object parameter, String jdbcType)
			throws SQLException {
		String val = null;
		if (parameter != null) {
			val = toString(parameter);
		}
		ps.setString(i, val);
	}
}
