/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.orm.sql.engine;

import java.sql.Connection;


/**
 * ExtendSqlExecutor (extends from SimpleSqlExecutor).
 * 
 * <pre>
 * Dynamic SQL: 
 *    &#64; --> (at mark) at something
 *    &#64;!  --> not at something
 * 
 *    1) SELECT * FROM SAMPLE
 *         &#64;[ WHERE
 *           &#64;[(&#64;id[ID=:id] &#64;name[OR NAME=:name] &#64;bool[OR BOOL=:bool])]
 *           &#64;list[AND LIST IN (:list)]
 *           &#64;kind[AND KIND=:kind]
 *           &#64;!kind[AND KIND IS NULL]
 *         ]
 *         &#64;orderCol[ ORDER BY ::orderCol ::orderDir ]
 * 
 *    descripton:
 *      &#64;id[ID=:id] --> append sql when id is not null
 *      &#64;bool[BOOL=:bool] --> append sql when bool is TRUE
 *      &#64;list[LIST IN (:list)]  --> append sql when list != null && list.size() > 0 (:list --> ?, ?, ?)
 *      &#64;[ WHERE ... ]  --> append sql when all the logical expression in &#64;[...] is true
 *      &#64;!kind[AND KIND IS NULL] --> append sql when kind is null
 * 
 *    2) UPDATE SAMPLE
 *         SET
 *           &#64;name[,NAME=:name]
 *           &#64;kind[,KIND=:kind]
 *           ,OTHER='@kind[,KIND=:kind]'
 *         WHERE
 *           ID=:id
 *
 *
 * @see SimpleSqlExecutor
 * 
 * </pre>
 */
public class ExtendSqlExecutor extends SimpleSqlExecutor {
	/**
	 * Constructor
	 * 
	 * @param connection connection
	 */
	public ExtendSqlExecutor(Connection connection) {
		super(connection);
	}

	/**
	 * Constructor
	 * 
	 * @param connection connection
	 * @param resultSetType one of the following ResultSet constants: ResultSet.TYPE_FORWARD_ONLY,
	 *            ResultSet.TYPE_SCROLL_INSENSITIVE, or ResultSet.TYPE_SCROLL_SENSITIVE
	 */
	public ExtendSqlExecutor(Connection connection, int resultSetType) {
		super(connection, resultSetType);
	}

	/**
	 * Constructor
	 * 
	 * @param connection connection
	 * @param resultSetType one of the following ResultSet constants: ResultSet.TYPE_FORWARD_ONLY,
	 *            ResultSet.TYPE_SCROLL_INSENSITIVE, or ResultSet.TYPE_SCROLL_SENSITIVE
	 * @param resultSetConcurrency one of the following ResultSet constants:
	 *            ResultSet.CONCUR_READ_ONLY or ResultSet.CONCUR_UPDATABLE
	 */
	public ExtendSqlExecutor(Connection connection, int resultSetType, int resultSetConcurrency) {
		super(connection, resultSetType, resultSetConcurrency);
	}

	/**
	 * Constructor
	 * 
	 * @param connection connection
	 * @param resultSetType one of the following ResultSet constants: ResultSet.TYPE_FORWARD_ONLY,
	 *            ResultSet.TYPE_SCROLL_INSENSITIVE, or ResultSet.TYPE_SCROLL_SENSITIVE
	 * @param resultSetConcurrency one of the following ResultSet constants:
	 *            ResultSet.CONCUR_READ_ONLY or ResultSet.CONCUR_UPDATABLE
	 * @param resultSetHoldability one of the following ResultSet constants:
	 *            ResultSet.HOLD_CURSORS_OVER_COMMIT or ResultSet.CLOSE_CURSORS_AT_COMMIT
	 */
	public ExtendSqlExecutor(Connection connection, int resultSetType, int resultSetConcurrency,
			int resultSetHoldability) {
		super(connection, resultSetType, resultSetConcurrency, resultSetHoldability);
	}

	/**
	 * createSqlParser
	 * @param sql sql
	 * @return SqlParser instance
	 * @throws Exception if an error occurs
	 */
	@Override
	protected SqlParser createSqlParser(String sql) throws Exception {
		return new ExtendSqlParser(sql);
	}
}
