/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.orm.dao;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

import nuts.core.orm.restriction.Orders;
import nuts.core.orm.restriction.Restrictions;

/**
 * QueryParameter
 */
@SuppressWarnings("serial")
public class SimpleQueryParameter implements QueryParameter, Cloneable, Serializable {

	protected Map<String, Boolean> excludes;
	protected Restrictions restrictions;
	protected Orders orders;
	protected Integer start;
	protected Integer limit;

	/**
	 * constructor
	 */
	protected SimpleQueryParameter() {
	}

	/**
	 * constructor
	 * 
	 * @param qp queryParameter
	 */
	protected SimpleQueryParameter(SimpleQueryParameter qp) {
		orders = qp.orders;
		restrictions = qp.restrictions;
		excludes = qp.excludes;
		start = qp.start;
		limit = qp.limit;
	}

	/**
	 * @return the excludes
	 */
	public Map<String, Boolean> getExcludes() {
		if (excludes == null) {
			excludes = new HashMap<String, Boolean>();
		}
		return excludes;
	}

	/**
	 * @param excludes the excludes to set
	 */
	public void setExcludes(Map<String, Boolean> excludes) {
		this.excludes = excludes;
	}

	/**
	 * @return true if the excludes is not empty
	 */
	public boolean isHasExcludes() {
		return excludes != null && !excludes.isEmpty();
	}

	/**
	 * @param column exclude column
	 * @return this
	 */
	public SimpleQueryParameter addExclude(String column) {
		getExcludes().put(column, true);
		return this;
	}

	/**
	 * @param column exclude column
	 * @return this
	 */
	public SimpleQueryParameter removeExclude(String column) {
		getExcludes().remove(column);
		return this;
	}

	/**
	 * clearExcludes
	 * 
	 * @return this
	 */
	public SimpleQueryParameter clearExcludes() {
		getExcludes().clear();
		return this;
	}

	/**
	 * @return restrictions
	 */
	public Restrictions getRestrictions() {
		return restrictions;
	}

	/**
	 * @param restrictions the restrictions to set
	 */
	public void setRestrictions(Restrictions restrictions) {
		this.restrictions = restrictions;
	}

	/**
	 * @return orders
	 */
	public Orders getOrders() {
		return orders;
	}

	/**
	 * @param orders the orders to set
	 */
	public void setOrders(Orders orders) {
		this.orders = orders;
	}

	/**
	 * @return the start
	 */
	public Integer getStart() {
		return start;
	}

	/**
	 * @param start the start to set
	 */
	public void setStart(Integer start) {
		this.start = start;
	}

	/**
	 * @return the limit
	 */
	public Integer getLimit() {
		return limit;
	}

	/**
	 * @param limit the limit to set
	 */
	public void setLimit(Integer limit) {
		this.limit = limit;
	}

	/**
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((excludes == null) ? 0 : excludes.hashCode());
		result = prime * result
				+ ((restrictions == null) ? 0 : restrictions.hashCode());
		result = prime * result + ((orders == null) ? 0 : orders.hashCode());
		result = prime * result + ((start == null) ? 0 : start.hashCode());
		result = prime * result + ((limit == null) ? 0 : limit.hashCode());
		return result;
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		SimpleQueryParameter other = (SimpleQueryParameter) obj;
		if (excludes == null) {
			if (other.excludes != null)
				return false;
		} else if (!excludes.equals(other.excludes))
			return false;
		if (restrictions == null) {
			if (other.restrictions != null)
				return false;
		} else if (!restrictions.equals(other.restrictions))
			return false;
		if (orders == null) {
			if (other.orders != null)
				return false;
		} else if (!orders.equals(other.orders))
			return false;
		if (start == null) {
			if (other.start != null)
				return false;
		} else if (!start.equals(other.start))
			return false;
		if (limit == null) {
			if (other.limit != null)
				return false;
		} else if (!limit.equals(other.limit))
			return false;
		return true;
	}

	/**
	 * Clone
	 * 
	 * @throws CloneNotSupportedException if clone not supported
	 * @return Clone Object
	 */
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}

	/**
	 * @return a string representation of the object.
	 */
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();

		sb.append("{ ");
		sb.append("excludes: { ").append(excludes).append(" }");
		sb.append(", ");
		sb.append("restrictions: ").append(restrictions);
		sb.append(", ");
		sb.append("orders: ").append(orders);
		sb.append(", ");
		sb.append("start: ").append(start);
		sb.append(", ");
		sb.append("limit: ").append(limit);
		sb.append(" }");

		return sb.toString();
	}

}