/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.orm.dao;

import java.util.List;
import java.util.Map;


/**
 * @param <T> entity type
 * @param <E> entity example type
 */
public interface EntityDAO<T, E> {
	/**
	 * create model example
	 * @return model example
	 */
	E createExample();
	
	/**
	 * exists
	 * 
	 * @param key T
	 * @return T
	 * @throws DataAccessException if a data access error occurs
	 */ 
	boolean exists(T key) throws DataAccessException;
	
	/**
	 * countByExample
	 * 
	 * @param example E
	 * @return count
	 * @throws DataAccessException if a data access error occurs
	 */ 
	int countByExample(E example) throws DataAccessException;

	/**
	 * selectByPrimaryKey
	 * 
	 * @param key T
	 * @return T
	 * @throws DataAccessException if a data access error occurs
	 */ 
	T selectByPrimaryKey(T key) throws DataAccessException;

	/**
	 * selectByExampleWithDataHandler
	 * 
	 * @param example E
	 * @param dataHandler data handler
	 * @return data count
	 * @throws DataAccessException if a data access error occurs
	 */ 
	int selectByExampleWithDataHandler(E example, DataHandler<T> dataHandler) throws DataAccessException;

	/**
	 * selectByExample
	 * 
	 * @param example E
	 * @return list of T 
	 * @throws DataAccessException if a data access error occurs
	 */ 
	List<T> selectByExample(E example) throws DataAccessException;

	/**
	 * selectOneByExample
	 * 
	 * @param example E
	 * @return T 
	 * @throws DataAccessException if a data access error occurs
	 */ 
	T selectOneByExample(E example) throws DataAccessException;

	/**
	 * selectAll
	 * 
	 * @return list of T 
	 * @throws DataAccessException if a data access error occurs
	 */ 
	List<T> selectAll() throws DataAccessException;

	/**
	 * selectMapByExample
	 * 
	 * @param example E
	 * @param keyProp The property to be used as the key in the Map.
	 * @return map of T
	 * @throws DataAccessException if a data access error occurs
	 */ 
	Map<?, T> selectMapByExample(E example, String keyProp) throws DataAccessException;

	/**
	 * selectMapAll
	 * 
	 * @param keyProp The property to be used as the key in the Map.
	 * @return map of T
	 * @throws DataAccessException if a data access error occurs
	 */ 
	Map<?, T> selectMapAll(String keyProp) throws DataAccessException;

	/**
	 * selectMapByExample
	 * 
	 * @param example E
	 * @param keyProp The property to be used as the key in the Map.
	 * @param valProp The property to be used as the value in the Map.
	 * @return map
	 * @throws DataAccessException if a data access error occurs
	 */ 
	Map<?, ?> selectMapByExample(E example, String keyProp, String valProp) throws DataAccessException;

	/**
	 * selectMapAll
	 * 
	 * @param keyProp The property to be used as the key in the Map.
	 * @param valProp The property to be used as the value in the Map.
	 * @return map
	 * @throws DataAccessException if a data access error occurs
	 */ 
	Map<?, ?> selectMapAll(String keyProp, String valProp) throws DataAccessException;

	/**
	 * if (exists(data)) { updateByPrimaryKey(data); } else { insert(data}; }
	 * 
	 * @param data T
	 * @throws DataAccessException if a data access error occurs
	 */ 
	void save(T data) throws DataAccessException;
	
	/**
	 * insert
	 * 
	 * @param data T
	 * @throws DataAccessException if a data access error occurs
	 */ 
	void insert(T data) throws DataAccessException;

	/**
	 * deleteByPrimaryKey
	 * 
	 * @param key T
	 * @return count of deleted records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	int deleteByPrimaryKey(T key) throws DataAccessException;

	/**
	 * deleteByExample
	 * 
	 * @param example E
	 * @return count of deleted records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	int deleteByExample(E example) throws DataAccessException;

	/**
	 * deleteAll
	 * 
	 * @return count of deleted records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	int deleteAll() throws DataAccessException;

	/**
	 * updateByPrimaryKey
	 * 
	 * @param data T
	 * @return count of updated records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	int updateByPrimaryKey(T data) throws DataAccessException;

	/**
	 * updateByPrimaryKeySelective (ignore null properties)
	 * 
	 * @param data T
	 * @return count of updated records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	int updateByPrimaryKeySelective(T data) throws DataAccessException;

	/**
	 * updateByExample
	 * 
	 * @param data T
	 * @param example E
	 * @return count of updated records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	int updateByExample(T data, E example) throws DataAccessException;

	/**
	 * updateByExampleSelective (ignore null properties)
	 * 
	 * @param data T
	 * @param example E
	 * @return count of updated records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	int updateByExampleSelective(T data, E example) throws DataAccessException;
}
