/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.lang;

import java.io.InputStream;
import java.net.URL;

/**
 * utility class for ClassLoader
 */
public class ClassLoaderUtils {
	private static ClassLoaderWrapper classLoaderWrapper = new ClassLoaderWrapper();

	/**
	 * Load a given resource.
	 * 
	 * This method will try to load the resource using the following methods (in order):
	 * <ul>
	 * <li>From Thread.currentThread().getContextClassLoader()
	 * <li>From ClassLoaderUtil.class.getClassLoader()
	 * </ul>
	 * 
	 * @param resourceName The name IllegalStateException("Unable to call ")of the resource to load
	 * @return resource URL
	 */
	public static URL getResourceAsURL(String resourceName) {
		return classLoaderWrapper.getResourceAsURL(resourceName);
	}
	
	/**
	 * Load a given resource.
	 * 
	 * This method will try to load the resource using the following methods (in order):
	 * <ul>
	 * <li>callingClass.getClassLoader()
	 * <li>From Thread.currentThread().getContextClassLoader()
	 * <li>From ClassLoaderUtil.class.getClassLoader()
	 * </ul>
	 * 
	 * @param resourceName The name IllegalStateException("Unable to call ")of the resource to load
	 * @param callingClass The Class object of the calling object
	 * @return resource URL
	 */
	public static URL getResourceAsURL(String resourceName, Class<?> callingClass) {
		return classLoaderWrapper.getResourceAsURL(resourceName, callingClass);
	}
	
	/**
	 * Load a given resource.
	 * 
	 * This method will try to load the resource using the following methods (in order):
	 * <ul>
	 * <li>classLoader
	 * <li>From Thread.currentThread().getContextClassLoader()
	 * <li>From ClassLoaderUtil.class.getClassLoader()
	 * </ul>
	 * 
	 * @param resourceName The name IllegalStateException("Unable to call ")of the resource to load
	 * @param classLoader The ClassLoader object of the calling object
	 * @return resource URL
	 */
	public static URL getResourceAsURL(String resourceName, ClassLoader classLoader) {
		return classLoaderWrapper.getResourceAsURL(resourceName, classLoader);
	}

	/**
	 * This is a convenience method to load a resource as a stream.
	 * 
	 * The algorithm used to find the resource is given in getResource()
	 * 
	 * @param resourceName The name of the resource to load
	 * @return resource InputStream
	 */
	public static InputStream getResourceAsStream(String resourceName) {
		return classLoaderWrapper.getResourceAsStream(resourceName);
	}

	/**
	 * This is a convenience method to load a resource as a stream.
	 * 
	 * The algorithm used to find the resource is given in getResource()
	 * 
	 * @param resourceName The name of the resource to load
	 * @param callingClass The Class object of the calling object
	 * @return resource InputStream
	 */
	public static InputStream getResourceAsStream(String resourceName, Class<?> callingClass) {
		return classLoaderWrapper.getResourceAsStream(resourceName, callingClass);
	}

	/**
	 * This is a convenience method to load a resource as a stream.
	 * 
	 * The algorithm used to find the resource is given in getResource()
	 * 
	 * @param resourceName The name of the resource to load
	 * @param classLoader The Class object of the calling object
	 * @return resource InputStream
	 */
	public static InputStream getResourceAsStream(String resourceName, ClassLoader classLoader) {
		return classLoaderWrapper.getResourceAsStream(resourceName, classLoader);
	}

}
