/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.beans;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * MapBeanHandler
 */
@SuppressWarnings("unchecked")
public class MapBeanHandler<T extends Map> extends AbstractJavaBeanHandler<T> {
	/**
	 * Constructor
	 * @param factory bean handler factory
	 * @param type bean type
	 */
	public MapBeanHandler(BeanHandlerFactory factory, Class<T> type) {
		super(factory, type);
	}
	
	/**
	 * get read property names
	 * @param map bean object (can be null)
	 * @return property names
	 */
	public String[] getReadPropertyNames(T map) {
		if (map == null) {
			return new String[0];
		}

		List<String> ns = new ArrayList<String>();
		for (Object o : map.keySet()) {
			if (o != null) {
				ns.add(o.toString());
			}
		}
		return ns.toArray(new String[ns.size()]);
	}

	/**
	 * get write property names
	 * @param map bean object (can be null)
	 * @return property names
	 */
	public String[] getWritePropertyNames(T map) {
		return getReadPropertyNames(map);
	}

	@Override
	public Class getPropertyType(String propertyName) {
		return Object.class;
	}
	
	public Class getPropertyType(T map, String propertyName) {
		if (map == null) {
			return Object.class;
		}
		Object val = getPropertyValue(map, propertyName);
		return val == null ? Object.class : val.getClass();
	}
	
	/**
	 * get bean type
	 * @param beanName property name
	 * @return bean type
	 */
	@Override
	public Class getBeanType(String beanName) {
		return Object.class;
	}

	/**
	 * get bean type
	 * @param map bean object (can be null)
	 * @param beanName bean name
	 * @return bean type
	 */
	@Override
	public Class getBeanType(T map, String beanName) {
		if (map != null) {
			Object val = getBeanValue(map, beanName);
			if (val != null) {
				return val.getClass();
			}
			return super.getBeanType(map, beanName);
		}
		return Object.class;
	}

	public Object getPropertyValue(T map, String propertyName) {
		return map.get(propertyName);
	}
	
	public void setPropertyValue(T map, String propertyName, Object propertyValue) {
		map.put(propertyName, propertyValue);
	}

	/**
	 * get property value 
	 * @param map bean object
	 * @param beanName bean name
	 * @return bean value
	 */
	@Override
	public Object getBeanValue(T map, String beanName) {
		if (map.containsKey(beanName)) {
			return map.get(beanName);
		}
		return super.getBeanValue(map, beanName);
	}

	/**
	 * set property value 
	 * @param map bean object
	 * @param beanName bean name
	 * @param value value
	 */
	@Override
	public void setBeanValue(T map, String beanName, Object value) {
		if (map.containsKey(beanName)) {
			map.put(beanName, value);
			return;
		}

		super.setBeanValue(map, beanName, value);
	}
}
