/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.algorithm.sort;

import java.util.Arrays;
import java.util.List;

import nuts.core.collections.CollectionUtils;


/**
 */
public abstract class Sorter {
	protected boolean ascend;
	
	public Sorter() {
		this(true);
	}
	
	public Sorter(boolean ascend) {
		this.ascend = ascend;
	}
	
	/**
	 * @return the ascend
	 */
	public boolean isAscend() {
		return ascend;
	}

	/**
	 * @param ascend the ascend to set
	 */
	public void setAscend(boolean ascend) {
		this.ascend = ascend;
	}

	/**
	 * Sorts the list
	 * @param list the list to be sorted
     * @param from the index of the first element (inclusive) to be sorted
     * @param to the index of the last element (exclusive) to be sorted
	 */
	public abstract <T extends Comparable<T>> void sort(List<T> list, int from, int to);

	/**
	 * Sorts the array
	 * @param array the array to be sorted
     * @param from the index of the first element (inclusive) to be sorted
     * @param to the index of the last element (exclusive) to be sorted
	 */
	public <T extends Comparable<T>> void sort(T[] array, int from, int to) {
		sort(Arrays.asList(array), from, to);
	}

	public <T extends Comparable<T>> void sort(List<T> list) {
		sort(list, 0, list.size());
	}

	public <T extends Comparable<T>> void sort(T[] array) {
		sort(Arrays.asList(array), 0, array.length);
	}
	
	public <T extends Comparable<T>> int compare(List<T> list, int x, int y) {
		T a = list.get(x);
		T b = list.get(y);
		return ascend ? a.compareTo(b) : b.compareTo(a);
	}
	
	public <T extends Comparable<T>> void swap(List<T> list, int x, int y) {
		CollectionUtils.swap(list, x, y);
	}
	
	protected <T extends Comparable<T>> boolean verify(T a, T b, boolean ascend) {
		if (ascend) {
			if (a.compareTo(b) > 0) {
				return false;
			}
		}
		else {
			if (a.compareTo(b) < 0) {
				return false;
			}
		}
		return true;
	}

	public <T extends Comparable<T>> boolean verify(List<T> list) {
		return verify(list, 0, list.size());
	}

	public <T extends Comparable<T>> boolean verify(List<T> list, int from, int to) {
		for (int i = from; i < to; i++) {
			for (int j = from; j < to; j++) {
				if (i != j) {
					int c = compare(list, i, j);
					if ((i < j && c > 0) || (i > j && c < 0)) {
						return false;
					}
				}
			}
		}
		return true;
	}

	public <T extends Comparable<T>> boolean verify(T[] array) {
		return verify(array, 0, array.length);
	}
	
	public <T extends Comparable<T>> boolean verify(T[] array, int from, int to) {
		return verify(Arrays.asList(array), from, to);
	}
}
