/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.aws.dao;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import nuts.core.io.IOUtils;
import nuts.core.lang.ClassUtils;

import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.services.simpledb.AmazonSimpleDBClient;

/**
 */
public class AwsDataAccessClientFactory {

	private static Map<String, AwsDataAccessClient> dataAccessClients = new HashMap<String, AwsDataAccessClient>();

	private static final String DEFAULT = "default";
	
	/**
	 * build dataAccessClient object
	 * 
	 * @param name  the configuration file resource name.
	 * @return GaeDataAccessClient instance.
	 */
	public static AwsDataAccessClient buildDataAccessClient(String name) {
		InputStream is = null;
		try {
			Properties p = new Properties();
			if (name != null) {
				is = ClassUtils.getClassLoader().getResourceAsStream(name);
				p.load(is);
			}
			return buildDataAccessClient(name, p);
		}
		catch (IOException e) {
			throw new RuntimeException("Can not load " + name, e);
		}
		finally {
			IOUtils.closeQuietly(is);
		}
	}

	/**
	 * build dataAccessClient object
	 * 
	 * @param props properties
	 * @return GaeDataAccessClient instance.
	 */
	public static AwsDataAccessClient buildDataAccessClient(Properties props) {
		return buildDataAccessClient(null, props);
	}
	
	/**
	 * build dataAccessClient object
	 * 
	 * @param name  the dataAccessClient name
	 * @param props properties
	 * @return GaeDataAccessClient instance.
	 */
	public static AwsDataAccessClient buildDataAccessClient(String name, Properties props) {
		String accessKey = props.getProperty("accessKey");
		String secretKey = props.getProperty("secretKey");
		
		BasicAWSCredentials credential = new BasicAWSCredentials(accessKey, secretKey);
		
		AwsDataAccessClient dac = new AwsDataAccessClient(new AmazonSimpleDBClient(credential), props);
		dac.setName(name == null ? DEFAULT : name);
		dataAccessClients.put(dac.getName(), dac);
		return dac;
	}

	/**
	 * getDataAccessClient
	 * @param name  the gae-datastore.properties configuration file resource name.
	 * @return GaeDataAccessClient instance.
	 */
	public static AwsDataAccessClient getDataAccessClient(String name) {
		AwsDataAccessClient smc = dataAccessClients.get(name == null ? DEFAULT : name);
		if (smc == null) {
			return buildDataAccessClient(name);
		}
		return smc;
	}
}
