/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.aems.action;

import java.util.Arrays;
import java.util.Calendar;

import nuts.aems.constant.GC;
import nuts.aems.constant.RC;
import nuts.aems.model.bean.CommonBean;
import nuts.aems.model.bean.User;
import nuts.aems.util.BaseActionConsts;
import nuts.aems.util.BaseActionUtils;
import nuts.core.orm.dao.QueryParameter;
import nuts.exts.struts2.actions.ModelDrivenAction;

/**
 * @param <T> data type
 * @param <E> example type
 */
public abstract class AbstractModelDrivenAction<T extends CommonBean, E extends QueryParameter> 
	extends ModelDrivenAction<T, E> {

	/**
	 * Constructor
	 */
	public AbstractModelDrivenAction() {
	}

	/**
	 * @return the utils
	 */
	protected BaseActionUtils utils() {
		return (BaseActionUtils)super.getUtils();
	}

	/**
	 * @return the consts
	 */
	protected BaseActionConsts consts() {
		return (BaseActionConsts)super.getConsts();
	}

	/**
	 * hasPermission
	 * @param namespace namespace
	 * @param action action
	 * @return true if action has access permit
	 */
	@Override
	public boolean hasPermission(String namespace, String action) {
		return utils().hasPermission(namespace, action);
	}

	/**
	 * checkUpdated
	 * @param data data
	 * @param srcData source data
	 * @return true if check successfully
	 * @throws Exception if an error occurs
	 */
	@Override
	protected boolean checkUpdated(T data, T srcData) throws Exception {
		if (srcData.beUpdated(data)) {
			data.setUusid(srcData.getUusid());
			data.setUtime(srcData.getUtime());
			addActionWarning(getText(RC.ERROR_DATA_IS_UPDATED,
					Arrays.asList(new Object[] { srcData.getUtime(), srcData.getUusid() })));
			return false;
		}
		return true;
	}

	/**
	 * startInsert
	 * @param data data
	 * @throws Exception if an error occurs
	 */
	@Override
	protected void startInsert(T data) throws Exception {
		super.startInsert(data);

		User user = utils().getLoginUser();

		if (data.getInvalid() == null) {
			data.setInvalid(false);
		}
		data.setCusid(user == null ? GC.UNKNOWN_USID : user.getId());
		data.setCtime(Calendar.getInstance().getTime());
		data.setUusid(data.getCusid());
		data.setUtime(data.getCtime());
	}

	/**
	 * startUpdate
	 * @param data data
	 * @param srcData srcData
	 * @throws Exception if an error occurs
	 */
	@Override
	protected void startUpdate(T data, T srcData) throws Exception {
		super.startUpdate(data, srcData);
		
		User user = utils().getLoginUser();

		if (data.getInvalid() == null) {
			data.setInvalid(false);
		}
		data.setUusid(user == null ? GC.UNKNOWN_USID : user.getId());
		data.setUtime(Calendar.getInstance().getTime());
	}
}
