package org.mineap.NNDD.libraryManager
{
	import flash.events.EventDispatcher;
	import flash.events.FileListEvent;
	import flash.filesystem.File;
	
	import org.mineap.NNDD.FileIO;
	import org.mineap.NNDD.LogManager;
	import org.mineap.NNDD.TagManager;
	import org.mineap.NNDD.model.NNDDVideo;
	import org.mineap.NNDD.model.VideoType;
	import org.mineap.NNDD.util.LibraryUtil;
	import org.mineap.NNDD.event.LibraryLoadEvent;
	
	[Event(name="libraryLoadComplete", type="LibraryLoadEvent")]
	[Event(name="libraryLoading", type="LibraryLoadEvent")]
	
	/**
	 * 
	 * 動画を管理するためのクラスです。
	 * 
	 * @author shiraminekeisuke (MineAP)
	 * 
	 */
	public class LibraryManager extends EventDispatcher
	{
		
		/**
		 * ロガー
		 */
		private var _logger:LogManager = LogManager.instance;
		
		/**
		 * タグ管理
		 */
		private var _tagManager:TagManager = TagManager.instance;
		
		/**
		 * NNDDVideoオブジェクト用Map
		 */
		private var _libraryMap:Object = new Object();
		
		/**
		 * ライブラリ更新時に一時的に利用するTempオブジェクトです。
		 */
		private var _tempLibraryMap:Object = null;
		
		/**
		 * LibraryManagerの唯一のインスタンス
		 */
		private static const _libraryManager:LibraryManager = new LibraryManager();
		
		/**
		 * ライブラリファイルの名前です
		 */
		public static const LIBRARY_FILE_NAME:String = "library.xml";
		
		/**
		 * 
		 */
		public static const DELETED_IMAGE_URL:String = "http://www.nicovideo.jp/img/common/delete.jpg";
		
		/**
		 * NNDDのライブラリファイルの保存先ディレクトリです
		 */
		private var _libraryDir:File = null;
		
		/**
		 * 
		 */
		private var _totalVideoCount:int = 0;
		
		/**
		 * 
		 */
		private var _videoCount:int = 0;
		
		/**
		 * 
		 * 
		 */
		public function LibraryManager()
		{
			if(_libraryManager != null){
				throw new ArgumentError("LibraryManagerはインスタンス化出来ません。");
			}
		}
		
		/**
		 * 
		 * @return 
		 * 
		 */
		public static function get instance():LibraryManager{
			return _libraryManager;
		}
		
		/**
		 *
		 * @param loadingShowTarget
		 * @param tagProvider 
		 */
		public static function initialize(tagProvider:Array):void{
			instance._tagManager.initialize(tagProvider);
		}
		
		/**
		 * 
		 * @return 
		 * 
		 */
		public function get tagManager():TagManager{
			return this._tagManager;
		}
		
		
		/**
		 * ライブラリファイルの場所を返します。
		 * @return 
		 * 
		 */
		public function get libraryFile():File{
			return new File(this.systemFileDir.url + "/" + LibraryManager.LIBRARY_FILE_NAME);
		}
		
		/**
		 * 現在のライブラリディレクトリを返します。
		 * @return 
		 * 
		 */
		public function get libraryDir():File{
			return new File(this._libraryDir.url);
		}
		
		/**
		 * NNDDのシステムディレクトリを返します。
		 * @return 
		 * 
		 */
		public function get systemFileDir():File{
			var systemDir:File = new File(libraryDir.url + "/system/");
			return systemDir;
		}
		
		/**
		 * NNDDの一時ファイル保存ディレクトリを返します。
		 * @return 
		 * 
		 */
		public function get tempDir():File{
			var tempDir:File = new File(systemFileDir.url + "/temp/");
			return tempDir;
		}
		
		/**
		 * ライブラリファイルの保存場所を更新します。
		 * 
		 * @param libraryDir
		 * @return 
		 */
		public function changeLibraryDir(libraryDir:File, isSave:Boolean = true):Boolean{
			if(libraryDir.isDirectory){
				this._libraryDir = libraryDir;
				if(isSave){
					this.saveLibraryFile();
				}
				return true;
			}else{
				return false;
			}
		}
		
		
		
		/**
		 * 指定されたディレクトリに対してライブラリファイルを保存します。
		 * 
		 * @param saveDir 保存先ディレクトリ。指定されていない場合は既定の場所に保存する。
		 * @return 
		 * 
		 */
		public function saveLibraryFile(saveDir:File = null):Boolean{
			
			try{
				
				if(saveDir == null){
					saveDir = this.systemFileDir;
				}
				
				if(!saveDir.exists){
					return false;
				}
				
				var xml:XML = new LibraryXMLHelper().convert(this._libraryMap);
				
				var fileIO:FileIO = new FileIO(_logger);
				fileIO.saveXMLSync(this.libraryFile, xml);
				
				this._logger.addLog("ライブラリを保存:" + new File(saveDir.url + "/" + LIBRARY_FILE_NAME).nativePath);
				fileIO.closeFileStream();
				
				return true;
				
			}catch(error:Error){
				_logger.addLog("ライブラリの保存に失敗:" + error + ":" + error.getStackTrace());
			}
			return false;
		}
		
		/**
		 * ライブラリファイルの読み込みを行います。
		 * 
		 * @param libraryDir ライブラリ
		 * @return ライブラリファイルの読み込みに成功すればtrue、失敗すればfalse。
		 */
		public function loadLibraryFile(libraryDir:File = null):Boolean{
			if(libraryDir == null){
				if(this.libraryFile.exists){
					return loadLibrary();
				}else{
					return false;
				}
			}else{
				if(libraryDir.isDirectory){
					this._libraryDir = libraryDir;
					return loadLibrary();
				}else{
					return false;
				}
			}
		}
		
		/**
		 * ライブラリを読み込みます。
		 * 
		 * @return 正常に読み込みが完了したかどうか。成功していればtrueを返す。
		 * 
		 */
		private function loadLibrary():Boolean{
			var fileIO:FileIO = new FileIO(_logger);
			
			try{
				
				var libraryXML:XML = fileIO.loadXMLSync(this.libraryFile.url, true);
				
				if(libraryXML != null){
					
					this._libraryMap = new LibraryXMLHelper().perseXML(libraryXML);
					
					this._tagManager.loadTag();
					
					return true;
				}
				
			}catch(error:Error){
				trace(error.getStackTrace());
			}
			
			return false;
			
		}
		
		/**
		 * ライブラリを更新します。
		 * 
		 * @param libraryDir 更新先ディレクトリ
		 * @param renewSubDir サブディレクトリも更新するかどうか。trueの場合は更新する。
		 */
		public function renewLibrary(libraryDir:File, renewSubDir:Boolean):void{
			
			_totalVideoCount = 0;
			_videoCount = 0;
			
			var videoList:Array = renewDir(libraryDir, renewSubDir);
			
			_tempLibraryMap = new Object();
			
			_totalVideoCount = videoList.length;
			
			trace(_totalVideoCount);
			
			videoList.forEach(infoLoadFunction);
			
		}
		
		/**
		 * 指定されたディレクトリを更新します。
		 * 
		 * @param dir 更新対象ディレクトリ
		 * @param renewSubDir サブディレクトリを更新するかどうか
		 * 
		 */
		private function renewDir(dir:File, renewSubDir:Boolean):Array{
			
			if(!dir.isDirectory){
				return new Array();
			}
			
			var fileList:Array = dir.getDirectoryListing();
			
			var videoList:Array = new Array();
			
			for(var index:uint = 0; index<fileList.length;index++){
				if(renewSubDir && fileList[index].isDirectory){
					var array:Array = renewDir((fileList[index] as File), true);
					
					for(var obj:Object in array){
						videoList.push(array);
					}
					
				}else if(!fileList[index].isDirectory){
					
					var extension:String = (fileList[index] as File).extension.toUpperCase();
					if(extension == VideoType.FLV_L || extension == VideoType.MP4_L){
						
						videoList.push(fileList[index].url);
						
					}else if(extension == VideoType.SWF_L){
						if((fileList[index] as File).nativePath.indexOf(VideoType.NICOWARI) == -1){
							videoList.push(fileList[index].url);
						}
						
					}
				}
			}
			
//			_tempLibraryMap = new Object();
//			
//			_totalVideoCount += videoList.length;
			
//			trace(_totalVideoCount);
			
			return videoList;
		}
		
		/**
		 * ビデオ一覧生成済みのArrayから呼ばれるコールバック関数です。
		 * 
		 * @param item コールバックもと配列の当該インデックスの要素
		 * @param index コールバック元の配列のインデックス
		 * @param array コールバック元の配列
		 * 
		 */
		private function infoLoadFunction(item:*, index:int, array:Array):void{

			_videoCount++;
			
			var file:File = null;
			var fileName:String = null;
			
			if(item is String){
				file = new File(item);
				fileName = file.nativePath;
			}
			
			try{
				
				var loader:LocalVideoInfoLoader = new LocalVideoInfoLoader();
				var nnddVideo:NNDDVideo = loader.loadInfo(file.url);
				
				var key:String = LibraryUtil.getVideoKey(file.nativePath);
				
				_tempLibraryMap[key] = nnddVideo;
				
				if(index%10 == 0){
					dispatchEvent(new LibraryLoadEvent(LibraryLoadEvent.LIBRARY_LOADING, false, false, _totalVideoCount, _videoCount, file));
					trace(_videoCount + ":" + file.nativePath);
				}
				
				if(_videoCount >= _totalVideoCount){
					for(var obj:Object in _tempLibraryMap){
						_libraryMap[obj] = _tempLibraryMap[obj];
					}
					_tempLibraryMap = null;
					
					this._tagManager.loadTag();
					
					dispatchEvent(new LibraryLoadEvent(LibraryLoadEvent.LIBRARY_LOAD_COMPLETE, false, false, _totalVideoCount, _videoCount))
				}
				
			}catch(error:Error){
				_logger.addLog("ファイルの読み込みに失敗:" + error + ":"+ fileName +":" + error.getStackTrace());
				
				if(_videoCount >= _totalVideoCount){
					for(var obj:Object in _tempLibraryMap){
						_libraryMap[obj] = _tempLibraryMap[obj];
					}
					_tempLibraryMap = null;
					
					this._tagManager.loadTag();
					
					dispatchEvent(new LibraryLoadEvent(LibraryLoadEvent.LIBRARY_LOAD_COMPLETE, false, false, _totalVideoCount, _videoCount))
				}
			}
			
		}
		
		/**
		 * 指定されたVideoIDをもつビデオを削除します。
		 * 
		 * @param videoId
		 * @param isSaveLibraryFile ライブラリファイルを保存するかどうか
		 * @return 
		 * 
		 */
		public function remove(videoId:String, isSaveLibraryFile:Boolean):NNDDVideo{
			
			var video:NNDDVideo = _libraryMap[videoId];
			
			delete _libraryMap[videoId];
			
			if(isSaveLibraryFile){
				saveLibraryFile();
			}
			
			return video;
		}
		
		/**
		 * 指定されたNNDDVideoでライブラリの動画を更新します。
		 * 
		 * @param video
		 * @param isSaveLibraryFile
		 * @return 
		 * 
		 */
		public function update(video:NNDDVideo, isSaveLibraryFile:Boolean):Boolean{
			
			var key:String = LibraryUtil.getVideoKey(video.getDecodeUrl());
			
			if(key != null){
				_libraryMap[key] = video;
				return true;
			}else{
				return false;
			}
		}
		
		/**
		 * 指定されたNNDDVideoをライブラリに追加します。
		 * 
		 * @param video
		 * @param isSaveLibraryFile ライブラリファイルを保存するかどうか
		 * @param isOverWrite 動画が登録済の場合に上書きするかどうか
		 * @return 
		 * 
		 */
		public function add(video:NNDDVideo, isSaveLibraryFile:Boolean, isOverWrite:Boolean = false):Boolean{
			var url:String = video.getDecodeUrl();
			
			if(!url.match(/\[Nicowari\]/)){
				var key:String = LibraryUtil.getVideoKey(video.getDecodeUrl());
				if(key != null && isExist(key) == null){
					
					_libraryMap[key] = video;
					
					if(isSaveLibraryFile){
						saveLibraryFile();
					}
					
					return true;
				}else{
					if(isOverWrite){
						_libraryMap[key] = video;
						if(isSaveLibraryFile){
							saveLibraryFile();
						}
						
						return true;
					}else{
						return false;
					}
				}
				
			}else{
				return false;
			}
		}
		
		/**
		 * ディレクトリのパスが変わったときに呼ばれます。
		 * ライブラリに登録されている項目で、oldDirUrlを含むビデオのパスを、newDirUrlに変更します。
		 * @param oldDirUrl デコード済の変更前ディレクトリURL
		 * @param newDirUrl でコード済の変更後ディレクトリURL
		 */
		public function changeDirName(oldDirUrl:String, newDirUrl:String):void{
			var oldPattern:RegExp = new RegExp(oldDirUrl);
			var isFailed:Boolean = false;
			
			var key:String = LibraryUtil.getVideoKey(oldDirUrl);
			if(key != null){
				var video:NNDDVideo = _libraryMap[key];
				if(video != null && video.getDecodeUrl().indexOf(oldDirUrl) != -1){
					var url:String = video.getDecodeUrl().replace(oldPattern, newDirUrl);
					
					var thumbUrl:String = video.thumbUrl;
					if(thumbUrl.indexOf("http") != -1){
						video.thumbUrl = thumbUrl.replace(oldPattern, newDirUrl);
					}else{
						//そのまま
					}
					
					var newVideo:NNDDVideo = new NNDDVideo(encodeURI(url), null, video.isEconomy, video.tagStrings, video.modificationDate, video.creationDate, video.thumbUrl, video.playCount);
					_libraryMap[key] = newVideo;
					
					//ライブラリ保存
					saveLibraryFile();
				}
			}
			
			_logger.addLog("ライブラリを更新:" + new File(newDirUrl).nativePath);
		}
		
		/**
		 * 指定された動画IDの動画が存在するかどうかを調べます。
		 * 
		 * @param videoId
		 * @return 
		 * 
		 */
		public function isExistByVideoId(videoId:String):NNDDVideo{
			var nnddVideo:NNDDVideo = null;
			if(videoId != null){
				nnddVideo = _libraryMap[videoId];
			}
			return nnddVideo;
		}
		
		/**
		 * 指定されたキーの動画が存在するかどうか調べます。<br />
		 * キーは {@link #getVideoKey()} で取得した値です。
		 * @param key
		 * @return 
		 * 
		 */
		public function isExist(key:String):NNDDVideo{
			var nnddVideo:NNDDVideo = null;
			if(key != null){
				nnddVideo = _libraryMap[key];
			}
			return nnddVideo;
		}
		
		
		
		/**
		 * タグ情報を取得します。
		 * 
		 * @param dir タグ情報を収集するディレクトリ。nullの場合すべてのディレクトリ。
		 * @return 
		 * 
		 */
		public function collectTag(dir:File = null):Array{
			var array:Array = new Array();
			var map:Object = new Object();
			
			for(var key:Object in _libraryMap){
				
				var video:NNDDVideo = _libraryMap[key];
				if(video == null){
					delete _libraryMap[key];
					continue;
				}
				
				if(dir != null && (decodeURIComponent(dir.url) == video.getDecodeUrl().substr(0, video.getDecodeUrl().lastIndexOf("/")))){
					for each(var tag:String in video.tagStrings){
						if(map[tag] == null){
							array.push(tag);
							map[tag] = tag;
						}
					}
				}else if(dir == null){
					for each(var tag:String in video.tagStrings){
						if(map[tag] == null){
							array.push(tag);
							map[tag] = tag;
						}
					}
				}
			}
			return array;
		}
		
		/**
		 * arrayで指定された名前のビデオがもつタグを返します。
		 * 
		 * @param array
		 * @return 
		 * 
		 */
		public function collectTagByVideoName(nameArray:Array):Array{
			var tagArray:Array = new Array();
			var tagMap:Object = new Object();
			
			for each(var videoName:String in nameArray){
				var key:String = LibraryUtil.getVideoKey(videoName);
				if(key != null){
					var video:NNDDVideo = isExist(key);
					if(video != null){
						for each(var tag:String in video.tagStrings){
							if(tagMap[tag] == null){
								tagArray.push(tag);
								tagMap[tag] = tag;
							}
						}
					}
				}
			}
			
			return tagArray;
		}
		
		/**
		 * 渡された文字列からタグを検索し、Tagビューに反映します。
		 * 
		 * @param word
		 * 
		 */
		public function searchTagAndShow(word:String):void{
			
			//wordをスペースで分割
			var pattern:RegExp = new RegExp("\\s*([^\\s]*)", "ig");
			var array:Array = word.match(pattern);
			
			_tagManager.searchTagAndShow(array);
			
		}
		
		/**
		 * saveDirで指定されたディレクトリ下に存在するビデオを返します。
		 * 
		 * @param saveDir
		 * @param isShowAll tureに設定すると、saveDir下のすべてのビデオを返します。
		 * @return 
		 * 
		 */
		public function getNNDDVideoArray(saveDir:File, isShowAll:Boolean):Vector.<NNDDVideo>{
			
			var saveUrl:String = decodeURIComponent(saveDir.url);
			var videos:Vector.<NNDDVideo> = new Vector.<NNDDVideo>();
			
			if(saveUrl.lastIndexOf("/") != saveUrl.length-1){
				saveUrl += "/";
			}
			var pattern:RegExp = new RegExp(saveUrl);
			
			for each(var video:NNDDVideo in _libraryMap){
				var index:int = video.getDecodeUrl().indexOf(saveUrl);
				if(index != -1){
					if(!isShowAll){
						var url:String = video.getDecodeUrl().replace(pattern, "");
						if(url.indexOf("/") == -1){
							videos.push(video);
						}
					}else{
						videos.push(video);
					}
				}
			}
			
			return videos;
		}
		
	}
}