# -*- coding: utf-8 -*-
#
#  Copyright (C) 2003-2013 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import logging
import os
import sys
import hashlib
import logging

from gi.repository import Gtk
from gi.repository import Gdk
from gi.repository import GdkPixbuf
import cairo

import numpy

import _gtkhack

class BaseTransparentWindow(Gtk.Window):
    __gsignals__ = {'screen-changed': 'override', }

    def __init__(self, type=Gtk.WindowType.TOPLEVEL):
        Gtk.Window.__init__(self, type=type)
        self.set_decorated(False)
        self.set_resizable(False)
        self.screen_changed()

    def screen_changed(self, old_screen=None):
        screen = self.get_screen()
        if self.is_composited():
            visual = screen.get_rgba_visual()
            self.supports_alpha = True
        else:
            visual = screen.get_system_visual()
            logging.debug('screen does NOT support alpha.\n')
            self.supports_alpha = False
        assert visual is not None
        self.set_visual(visual)


class TransparentWindow(BaseTransparentWindow):

    def __init__(self, type=Gtk.WindowType.TOPLEVEL):
        BaseTransparentWindow.__init__(self, type=type)
        self.set_app_paintable(True)
        self.set_focus_on_map(False)
        self.__position = (0, 0)
        self.__surface_position = (0, 0)
        self.__redraw = None
        self.connect_after('size_allocate', self.size_allocate)
        # create drawing area
        self.darea = Gtk.DrawingArea()
        self.darea.show()
        self.darea.connect = self.wrap_connect
        self.add(self.darea)
        self.override_background_color(Gtk.StateFlags.NORMAL,
                                       Gdk.RGBA(0, 0, 0, 0))

    def wrap_connect(self, signal, user_function, user_data=None):
        if signal == 'draw':
            self.__redraw = (user_function, user_data)
            Gtk.DrawingArea.connect(self.darea, signal, self.wrap_draw)
        else:
            if user_data is not None:
                Gtk.DrawingArea.connect(
                    self.darea, signal, user_function, user_data)
            else:
                Gtk.DrawingArea.connect(
                    self.darea, signal, user_function)

    def wrap_draw(self, darea, cr):
        if self.__redraw is None:
            return
        cr.translate(*self.get_draw_offset())
        user_function, user_data = self.__redraw
        if user_data is not None:
            user_function(darea, cr, user_data)
        else:
            user_function(darea, cr)
        ##region = Gdk.cairo_region_create_from_surface(cr.get_target())
        ##self.input_shape_combine_region(region)
        _gtkhack.gtkwindow_set_input_shape(self, cr.get_target())

    def update_size(self, w, h):
        self.get_child().set_size_request(w, h) # XXX
        self.queue_resize()

    def size_allocate(self, widget, event):
        new_x, new_y = self.__position
        Gtk.Window.move(self, new_x, new_y)

    def move(self, x, y):
        left, top, scrn_w, scrn_h = get_workarea()
        w, h = self.get_child().get_size_request() # XXX
        new_x = min(max(left, x), scrn_w - w)
        new_y = min(max(top, y), scrn_h - h)
        Gtk.Window.move(self, new_x, new_y)
        self.__position = (new_x, new_y)
        self.__surface_position = (x, y)
        self.darea.queue_draw()

    def get_draw_offset(self):
        window_x, window_y = self.__position
        surface_x, surface_y = self.__surface_position
        return surface_x - window_x, surface_y - window_y

    def winpos_to_surfacepos(self, x, y, scale):
        window_x, window_y = self.__position
        surface_x, surface_y = self.__surface_position
        new_x = int((x - (surface_x - window_x)) * 100 / scale)
        new_y = int((y - (surface_y - window_y)) * 100 / scale)
        return new_x, new_y


def get_png_size(path):
    if not path or not os.path.exists(path):
        return 0, 0
    head, tail = os.path.split(path)
    basename, ext = os.path.splitext(tail)
    ext = os.fsdecode(ext).lower()
    if ext == '.dgp':
        buf = get_DGP_IHDR(path)
    elif ext == '.ddp':
        buf = get_DDP_IHDR(path)
    else:
        buf = get_png_IHDR(path)
    assert buf[0:8] == b'\x89PNG\r\n\x1a\n' # png format
    assert buf[12:16] == b'IHDR' # name of the first chunk in a PNG datastream
    w = buf[16:20]
    h = buf[20:24]
    width = (w[0] << 24) + (w[1] << 16) + (w[2] << 8) + w[3]
    height = (h[0] << 24) + (h[1] << 16) + (h[2] << 8) + h[3]
    return width, height

def get_png_lastpix(path):
    if not path or not os.path.exists(path):
        return None
    pixbuf = __pixbuf_new_from_file(path)
    assert pixbuf.get_n_channels() in [3, 4]
    assert pixbuf.get_bits_per_sample() == 8
    color = '#{:02x}{:02x}{:02x}'.format(*pixbuf.get_pixels()[-3:])
    return color.encode('ascii')

def get_DGP_IHDR(path):
    head, tail = os.path.split(os.fsdecode(path)) # XXX
    filename = tail
    m_half = hashlib.md5(filename[:len(filename) // 2]).hexdigest()
    m_full = hashlib.md5(filename).hexdigest()
    tmp = ''.join((m_full, filename))
    key = ''
    j = 0
    for i in range(len(tmp)):
        value = ord(tmp[i]) ^ ord(m_half[j])
        if not value:
            break
        key = ''.join((key, chr(value)))
        j += 1
        if j >= len(m_half):
            j = 0
    key_length = len(key)
    if key_length == 0: # not encrypted
        logging.warning(''.join((filename, ' generates a null key.')))
        return get_png_IHDR(path)
    key = ''.join((key[1:], key[0]))
    key_pos = 0
    buf = b''
    with open(path, 'rb') as f:
        for i in range(24):
            c = f.read(1)
            buf = b''.join(
                (buf, int.to_bytes(c[0] ^ ord(key[key_pos]), 1, 'little')))
            key_pos += 1
            if key_pos >= key_length:
                key_pos = 0
    return buf

def get_DDP_IHDR(path):
    size = os.path.getsize(path)
    key = size << 2    
    buf = b''
    with open(path, 'rb') as f:
        for i in range(24):
            c = f.read(1)
            key = (key * 0x08088405 + 1) & 0xffffffff
            buf = b''.join(
                (buf, int.to_bytes((c[0] ^ key >> 24) & 0xff, 1, 'little')))
    return buf

def get_png_IHDR(path):
    with open(path, 'rb') as f:
        buf = f.read(24)
    return buf

def __pixbuf_new_from_file(path):
    path = os.fsdecode(path) # XXX
    return GdkPixbuf.Pixbuf.new_from_file(path)

def __surface_new_from_file(path):
    path = os.fsdecode(path) # XXX
    return cairo.ImageSurface.create_from_png(path)

def create_icon_pixbuf(path):
    path = os.fsdecode(path) # XXX
    try:
        pixbuf = __pixbuf_new_from_file(path)
    except: # compressed icons are not supported. :-(
        pixbuf = None
    else:
        pixbuf = pixbuf.scale_simple(16, 16, GdkPixbuf.InterpType.BILINEAR)
    return pixbuf

def create_blank_surface(width, height): ## FIXME
    surface = cairo.ImageSurface(cairo.FORMAT_ARGB32, width, height)
    return surface

def create_pixbuf_from_DGP_file(path):
    head, tail = os.path.split(os.fsdeocde(path)) # XXX
    filename = tail
    m_half = hashlib.md5(filename[:len(filename) // 2]).hexdigest()
    m_full = hashlib.md5(filename).hexdigest()
    tmp = ''.join((m_full, filename))
    key = ''
    j = 0
    for i in range(len(tmp)):
        value = ord(tmp[i]) ^ ord(m_half[j])
        if not value:
            break
        key = ''.join((key, chr(value)))
        j += 1
        if j >= len(m_half):
            j = 0
    key_length = len(key)
    if key_length == 0: # not encrypted
        logging.warning(''.join((filename, ' generates a null key.')))
        pixbuf = __pixbuf_new_from_file(filename)
        return pixbuf
    key = ''.join((key[1:], key[0]))
    key_pos = 0
    loader = GdkPixbuf.PixbufLoader.new_with_type('png')
    with open(path, 'rb') as f:
        while 1:
            c = f.read(1)
            if c == b'':
                break
            loader.write(int.to_bytes(c[0] ^ ord(key[key_pos]), 1, 'little'))
            key_pos += 1
            if key_pos >= key_length:
                key_pos = 0
    pixbuf = loader.get_pixbuf()
    loader.close()
    return pixbuf

def create_pixbuf_from_DDP_file(path):
    with open(path, 'rb') as f:
        buf = f.read()
    key = len(buf) << 2
    loader = GdkPixbuf.PixbufLoader.new_with_type('png')
    for i in range(len(buf)):
        key = (key * 0x08088405 + 1) & 0xffffffff
        loader.write(int.to_bytes((buf[i] ^ key >> 24) & 0xff, 1, 'little'))
    pixbuf = loader.get_pixbuf()
    loader.close()
    return pixbuf

def create_surface_from_file(path, is_pnr=True, use_pna=False):
    head, tail = os.path.split(path)
    basename, ext = os.path.splitext(tail)
    pixbuf = create_pixbuf_from_file(path, is_pnr, use_pna)
    surface = cairo.ImageSurface(cairo.FORMAT_ARGB32,
                                 pixbuf.get_width(), pixbuf.get_height())
    cr = cairo.Context(surface)
    Gdk.cairo_set_source_pixbuf(cr, pixbuf, 0, 0)
    cr.set_operator(cairo.OPERATOR_SOURCE)
    cr.paint()
    del cr
    ##if is_pnr:
    ##    buf = surface.get_data()
    ##    ar = numpy.frombuffer(buf, numpy.uint32)
    ##    argb = ar[0]
    ##    ar[ar == argb] = 0x00000000
    ##if use_pna:
    ##    path = os.path.join(head, b''.join((basename, b'.pna')))
    ##    if os.path.exists(path):
    ##        pna_surface = __surface_new_from_file(path)
    ##        pna_buf = pna_surface.get_data()
    ##        buf = surface.get_data()
    ##        step = cairo.ImageSurface.format_stride_for_width(
    ##            pna_surface.get_format(), 1)
    ##        pos = 0
    ##        for index in range(0, len(buf), 4):
    ##            if sys.byteorder == 'little':
    ##                buf[index + 3] = pna_buf[pos]
    ##            else:
    ##                buf[index + 0] = pna_buf[pos]
    ##            pos += step
    return surface

def create_pixbuf_from_file(path, is_pnr=True, use_pna=False):
    head, tail = os.path.split(path)
    basename, ext = os.path.splitext(tail)
    ext = os.fsdecode(ext).lower()
    if ext == '.dgp':
        pixbuf = create_pixbuf_from_DGP_file(path)
    elif ext == '.ddp':
        pixbuf = create_pixbuf_from_DDP_file(path)
    else:
        pixbuf = __pixbuf_new_from_file(path)
    if is_pnr:
        array = pixbuf.get_pixels()
        if not pixbuf.get_has_alpha():
            r = array[0]
            g = array[1]
            b = array[2]
            pixbuf = pixbuf.add_alpha(True, r, g, b)
        else:
            ar = numpy.frombuffer(
                _gtkhack.gdkpixbuf_get_pixels_array(pixbuf),
                dtype=numpy.uint32)
            rgba = ar[0]
            ar[ar == rgba] = 0x00000000
    if use_pna:
        path = os.path.join(head, b''.join((basename, b'.pna')))
        if os.path.exists(path):
            assert pixbuf.get_has_alpha()
            pna_pixbuf = __pixbuf_new_from_file(path)
            assert pna_pixbuf.get_bits_per_sample() / 8 == 1
            pixbuf_array = _gtkhack.gdkpixbuf_get_pixels_array(pixbuf)
            pna_array = _gtkhack.gdkpixbuf_get_pixels_array(pna_pixbuf)
            pixbuf_array[:,:,3] = pna_array[:,:,0]
    return pixbuf

def get_workarea():
    scrn = Gdk.Screen.get_default()
    root = scrn.get_root_window()
    left, top, width, height = root.get_geometry()
    return left, top, width, height
