# -*- coding: utf-8 -*-
#
#  satori.py - a "里々" compatible Shiori module for ninix
#  Copyright (C) 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2013 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003, 2004 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

# TODO:
# - φエスケープ： 特殊記号の無効化, replaceの無効化など
# - イベントを単語群で定義できるように
# - 文と単語群の重複回避
# - ≫
# - コミュニケート
# - 内部呼び出し: 単語の追加, sync
# - ＄トーク中のなでられ反応
# - ＄BalloonOffset0, ＄BalloonOffset1
# - マルチキャラクタ

import os
import sys
import logging
import re
import time
import random

from ninix.home import get_normalized_path

NODE_TEXT       = 1
NODE_REF        = 2
NODE_SIDE       = 3
NODE_ASSIGNMENT = 4
NODE_JUMP       = 5
NODE_SEARCH     = 6
NODE_CHOICE     = 7
NODE_CALL       = 8
NODE_SAORI      = 9
NODE_OR_EXPR    = 20
NODE_AND_EXPR   = 21
NODE_COMP_EXPR  = 22
NODE_ADD_EXPR   = 23
NODE_MUL_EXPR   = 24
NODE_POW_EXPR   = 25
NODE_UNARY_EXPR = 26

def encrypt(s):
    buf = []
    t = len(s)
    p = (t + 1) // 2
    for n in range(p):
        buf.append(s[n])
        if len(s[p:]) > n:
            buf.append(s[-n - 1])
    return buf

def decrypt(s):
    buf = []
    t = len(s)
    for n in range(0, t, 2):
        buf.append(s[n:][:1]) # XXX
    if t % 2 == 0:
        p = 1
    else:
        p = 2
    for n in range(p, t, 2):
        buf.append(s[-n:][:1]) # XXX
    return b''.join(buf)

def list_dict(top_dir):
    buf = []
    try:
        dir_list = os.listdir(top_dir)
    except OSError:
        dir_list = []
    for filename in dir_list:
        basename, ext = os.path.splitext(filename)
        ext = ext.lower()
        if filename.lower().startswith(b'dic') and \
           ext in [b'.txt', b'.sat'] or \
           filename.lower() in [b'replace.txt', b'replace_after.txt',
                                b'satori_conf.txt', b'satori_conf.sat']: # XXX
            buf.append(os.path.join(top_dir, filename))
    return buf


class Filter:

    def __init__(self, rules):
        self.rules = []
        for pat, rep in rules:
            self.rules.append((pat, rep))

    def apply(self, text):
        for pat, rep in self.rules:
            text = text.replace(pat, rep)
        return text


###   PARSER   ###

def read_tab_file(path, encrypted=0):
    lineno = 0
    buf = []
    with open(path, 'rb') as f:
        for line in f:
            lineno += 1
            if line.endswith(b'\r\n'):
                line = line[:-2]
            elif line.endswith(b'\r') or line.endswith(b'\n'):
                line = line[:-1]
            if encrypted:
                line = decrypt(decrypt(line))
            try:
                line = str(line, 'CP932')
            except UnicodeError as e:
                logging.debug('satori.py: {0} in {1} (line {2:d})'.format(e, path, lineno))
                continue
            try:
                old, new = line.split('\t')
            except ValueError:
                logging.debug('satori.py: invalid line in {0} (line {1:d})'.format(path, lineno))
                continue
            buf.append((old, new))
    return buf


class Parser:

    def __init__(self):
        self.talk = {}
        self.word = {}
        self.variable = []
        self.parenthesis = 0
        self.replace_filter = Filter([])
        self.anchor_list = []
        self.anchor_filter = Filter(self.anchor_list)
        self.is_anchor = 0
        self.saori = []
        self.separator = ['\1', ',', '，', '、', '､']
        self.count = {'Talk':        0,
                      'NoNameTalk':  0,
                      'EventTalk':   0,
                      'OtherTalk':   0,
                      'Words':       0,
                      'Word':        0,
                      'Variable':    0,
                      'Anchor':      0,
                      'Parenthesis': 0,
                      'Parentheres': 0, ## XXX
                      'Line':        0,}

    def set_saori(self, saori_list):
        self.saori = saori_list

    def get_count(self, name):
        if name in self.count:
            return self.count[name]
        else:
            return 0

    def load_replace_file(self, path):
        self.replace_filter = Filter(read_tab_file(path))

    def get_dict(self):
        return self.talk, self.word

    def read(self, path):
        basename, ext = os.path.splitext(path)
        ext = ext.lower()
        if ext == b'.sat':
            encrypted = 1
        else:
            encrypted = 0
        filename = os.path.basename(path)
        if filename.lower().startswith(b'dicanchor'): # XXX
            self.is_anchor = 1
        else:
            self.is_anchor = 0
        with open(path, 'rb') as f:
            self.read_file(f, path, encrypted)
        if self.is_anchor:
            self.anchor_filter = Filter(self.anchor_list)

    def read_file(self, f, path=None, encrypted=0):
        lineno = 0
        linelist = None
        line_buffer = None # XXX
        phi_escape = {} # key = lineno: [position]
        for line in f:
            if line_buffer is None:
                lineno += 1
                phi_escape[lineno] = []
            if line.endswith(b'\r\n'):
                line = line[:-2]
            elif line.endswith(b'\r') or line.endswith(b'\n'):
                line = line[:-1]
            if encrypted:
                line = decrypt(decrypt(line))
            try:
                line = str(line, 'CP932')
            except UnicodeError as e:
                if path is None:
                    logging.debug('satori.py: {0} (line {1:d})'.format(e, lineno))
                else:
                    logging.debug('satori.py: {0} in {1} (line {2:d})'.format(e, path, lineno))
                continue
            if line_buffer is not None:
                line = ''.join((line_buffer, line))
                line_buffer = None
            pos = 0
            while line.count('φ', pos) > 0:
                pos = line.find('φ', pos)
                if pos == len(line) - 1:
                    line_buffer = line[:-1]
                    break
                else:
                    phi_escape[lineno].append(pos) ## FIXME
                    line = ''.join((line[:pos], line[pos + 1:]))
            if line_buffer is not None:
                continue
            pos = 0
            while line.count('＃', pos) > 0:
                pos = line.find('＃', pos)
                if pos not in phi_escape[lineno]: ## FIXME
                    line = line[:pos]
                    break
            if not line:
                continue
            if line.startswith('＊') and 0 not in phi_escape[lineno]:
                if linelist:
                    parser(linelist, phi_escape)
                parser = self.parse_talk
                linelist = [lineno, line]
            elif line.startswith('＠') and 0 not in phi_escape[lineno]:
                if linelist:
                    parser(linelist, phi_escape)
                parser = self.parse_word_group
                linelist = [lineno, line]
            elif linelist:
                # apply replace.txt
                line = self.replace_filter.apply(line) ## FIXME: phi_escape
                linelist.append(line)
#        for no in phi_escape:
#            if phi_escape[no]:
#                print('PHI:', no, phi_escape[no])
        if linelist:
            parser(linelist, phi_escape)
        self.count['Line'] = self.count['Line'] + lineno
        talk = 0
        eventtalk = 0
        for key, value in self.talk.items():
            number = len(value)
            talk += number
            if key.startswith('On'):
                eventtalk += number
        self.count['Talk'] = talk
        self.count['EventTalk'] = eventtalk
        if '' in self.talk:
            self.count['NoNameTalk'] = len(self.talk[''])
        self.count['OtherTalk'] = self.count['Talk'] \
                                  - self.count['NoNameTalk'] \
                                  - self.count['EventTalk']
        self.count['Words'] = len(self.word)
        word = 0
        for value in self.word.values():
            word += len(value)
        self.count['Word'] = word
        self.count['Anchor'] = len(self.anchor_list)
        self.count['Variable'] = len(self.variable)
        self.count['Parenthesis'] = self.parenthesis
        self.count['Parentheres'] = self.parenthesis

    def parse_talk(self, linelist, phi_escape):
        lineno = linelist[0]
        buf = []
        line = linelist[1]
        assert line.startswith('＊')
        name = line[1:]
        while len(linelist) > 3 and not linelist[-1]:
            del linelist[-1]
        prev = ''
        num_open = 0
        num_close = 0
        for n in range(2, len(linelist)):
            line = linelist[n]
            num_open += line.count('（') ### FIXME: φ
            num_close += line.count('）') ### FIXME: φ
            if num_open > 0 and num_open != num_close:
                if n == len(linelist) - 1:
                    logging.debug(
                        'satori.py: syntax error (unbalanced parens)')
                else:
                    prev = ''.join((prev, linelist[n]))
                    continue 
            else:
                num_open = 0
                num_close = 0
            line = ''.join((prev, linelist[n]))
            prev = ''
            current_lineno = lineno + n - 2
            if line and line[0] == '＄' and 0 not in phi_escape[current_lineno]:
                node = self.parse_assignment(line)
                if node is not None:
                    buf.append(node)
            elif line and line[0] == '＞' and 0 not in phi_escape[current_lineno]:
                node = self.parse_jump(line)
                if node is not None:
                    buf.append(node)
            elif line and line[0] == '≫' and 0 not in phi_escape[current_lineno]:
                node = self.parse_search(line)
                if node is not None:
                    buf.append(node)
            elif line and line[0] == '＿' and 0 not in phi_escape[current_lineno]:
                node = self.parse_choice(line)
                if node is not None:
                    buf.append(node)
            else:
                nodelist = self.parse_talk_word(line)
                if nodelist is not None:
                    buf.extend(nodelist)
        if buf:
            try:
                talk_list = self.talk[name]
            except KeyError:
                talk_list = self.talk[name] = []
            talk_list.append(buf)
            if self.is_anchor:
                self.anchor_list.append(
                    [name, '\\_a[{0}]{0}\\_a'.format(name)])

    def parse_word_group(self, linelist, phi_escape):
        lineno = linelist[0]
        buf = []
        line = linelist[1]
        assert line.startswith('＠')
        name = line[1:]
        prev = ''
        num_open = 0
        num_close = 0
        for n in range(2, len(linelist)):
            line = linelist[n]
            num_open += line.count('（') ### FIXME: φ
            num_close += line.count('）') ### FIXME: φ
            if num_open > 0 and num_open != num_close:
                if n == len(linelist) - 1:
                    logging.debug(
                        'satori.py: syntax error (unbalanced parens)')
                else:
                    prev = ''.join((prev, linelist[n]))
                    continue 
            else:
                num_open = 0
                num_close = 0
            line = ''.join((prev, linelist[n]))
            prev = ''
            if not line:
                continue
            word = self.parse_word(line)
            if word:
                buf.append(word)
        if buf:
            try:
                word_list = self.word[name]
            except KeyError:
                word_list = self.word[name] = []
            word_list.extend(buf)

    def parse_assignment(self, line):
        assert line[0] == '＄'
        for n in range(1, len(line)):
            if line[n] in ['\t', ' ', '　', '＝']: ### FIXME: φ
                break
        else:
            logging.debug('satori.py: syntax error (expected a tab or equal)')
            return None
        name_str = ''.join(line[1:n]) # XXX
        name = self.parse_word(line[1:n])
        if line[n] == '＝': ### FIXME: φ
            n += 1
            value = self.parse_expression(line[n:])
        else:
            sep = line[n]
            while n < len(line) and line[n] == sep:
                n += 1
            value = self.parse_word(line[n:])
        if name_str == '引数区切り削除':
            sep = ''.join(line[n:]) # XXX
            if sep in self.separator:
                self.separator.remove(sep)
            return None
        elif name_str == '引数区切り追加':
            sep = ''.join(line[n:]) # XXX
            if sep not in self.separator:
                self.separator.append(sep)
            return None
        if name not in self.variable:
            self.variable.append(name)
        return [NODE_ASSIGNMENT, name, value]

    def parse_jump(self, line):
        assert line[0] == '＞'
        for n in range(1, len(line)):
            if line[n] == '\t': ### FIXME: φ
                break
        else:
            n = len(line)
        target = self.parse_word(line[1:n])
        while n < len(line) and line[n] == '\t': ### FIXME: φ
            n += 1
        if n < len(line):
            condition = self.parse_expression(line[n:])
        else:
            condition = None
        return [NODE_JUMP, target, condition]

    def parse_search(self, line):
        return [NODE_SEARCH]

    def parse_choice(self, line):
        assert line[0] == '＿'
        for n in range(1, len(line)):
            if line[n] == '\t': ### FIXME: φ
                break
        else:
            n = len(line)
        label = self.parse_word(line[1:n])
        while n < len(line) and line[n] == '\t': ### FIXME: φ
            n += 1
        if n < len(line):
            id_ = self.parse_word(line[n:])
        else:
            id_ = None
        return [NODE_CHOICE, label, id_]

    def parse_talk_word(self, line):
        buf = self.parse_word(line)
        buf.append([NODE_TEXT, [r'\n']])
        return buf

    def parse_word(self, line):
        buffer_ = []
        text = []
        while line:
            if line[0] == '：': ### FIXME: φ
                if text:
                    buffer_.append([NODE_TEXT, text])
                    text = []
                buffer_.append([NODE_SIDE, [line[0]]])
                line = line[1:]
            elif line[0] == '（': ### FIXME: φ
                self.parenthesis += 1
                if text:
                    buffer_.append([NODE_TEXT, text])
                    text = []
                line = self.parse_parenthesis(line, buffer_)
            else:
                text.append(line[0])
                line = line[1:]
        if text:
            buffer_.append([NODE_TEXT, text])
        return buffer_

    def find_close(self, text, position):
        nest = 0
        current = position
        while text[current:].count('）') > 0:
            pos_new = text.index('）', current)
            if pos_new == 0:
                break
            nest = text[position:pos_new].count('（') - text[position:pos_new].count('）')
            if nest > 0:
                current = pos_new + 1
            else:
                current = pos_new
                break
        return current

    def split_(self, text, sep):
        buf = []
        pos_end = -1
        while 1:
            position = text.find('（', pos_end + 1)
            if position == -1:
                ll = text[pos_end + 1:].split(sep)
                if len(buf) > 0:
                    last = buf.pop(-1)
                    buf.append(''.join((last, ll[0])))
                    buf.extend(ll[1:])
                else:
                    buf.extend(ll)
                break
            else:
                ll = text[pos_end + 1:position].split(sep)
                pos_end = self.find_close(text, position + 1)
                last = ll.pop(-1)
                ll.append(''.join((last, text[position:pos_end + 1])))
                if len(buf) > 0:
                    last = buf.pop(-1)
                    buf.append(''.join((last, ll[0])))
                    buf.extend(ll[1:])
                else:
                    buf.extend(ll)
        buf = [x.strip() for x in buf]
        return buf

    def parse_parenthesis(self, line, buf):
        text_ = []
        assert line[0] == '（'
        line = line[1:] ## FIXME
        depth = 1
        count = 1
        while line:
            if line[0] == '）': ### FIXME: φ
                depth -= 1
                if len(text_) != count: # （）
                    text_.append('')
                if depth == 0:
                    line = line[1:]
                    break
                text_.append(line[0])
                line = line[1:]
            elif line[0] == '（': ### FIXME: φ
                depth += 1
                count += 1
                text_.append(line[0])
                line = line[1:]
            else:
                text_.append(line[0])
                line = line[1:]
        if text_:
            sep = None
            pos = len(text_)
            for c in self.separator: ### FIXME: φ
                if text_.count(c) == 0:
                    continue
                if text_.index(c) < pos:
                    pos = text_.index(c)
                    sep = c
            if sep is not None:
                list_ = self.split_(''.join(text_), sep) ## FIXME
            else:
                list_ = [''.join(text_)]
            if list_[0] in ['単語の追加', 'sync', 'loop', 'call',
                            'set', 'remember', 'nop', '合成単語群',
                            'when', 'times', 'while', 'for'] or list_[0].endswith('の数'):
                function = list_[0]
                args = []
                if len(list_) > 1:
                    for i in range(1, len(list_)):
                        args.append(self.parse_expression(list_[i]))
                buf.append([NODE_CALL, function, args])
                return line
            elif list_[0] in self.saori:
                function = list_[0]
                args = []
                if len(list_) > 1:
                    for i in range(1, len(list_)):
                        # XXX: parse as text
                        args.append(self.parse_word(list_[i]))
                buf.append([NODE_SAORI, function, args])
                return line
            else:
                if len(list_) > 1: # XXX
                    nodelist = [[NODE_TEXT, ['（']]]
                    nodelist.extend(self.parse_word(text_))
                    nodelist.append([NODE_TEXT, ['）']])
                    buf.append([NODE_REF, nodelist])
                    return line
                else:
                    nodelist = [[NODE_TEXT, ['（']]]
                    nodelist.extend(self.parse_word(text_))
                    nodelist.append([NODE_TEXT, ['）']])
                    buf.append([NODE_REF, nodelist])
                    return line
        buf.append([NODE_TEXT, ['']]) # XXX
        return line

    def parse_expression(self, line):
        default = [[NODE_TEXT, line[:]]]
        try:
            line, buf = self.get_or_expr(line)
        except ValueError as e:
            return default
        if line:
            return default
        return buf

    def get_or_expr(self, line):
        line, and_expr = self.get_and_expr(line)
        buf = [NODE_OR_EXPR, and_expr]
        while line and line[0] in ['|', '｜']: ### FIXME: φ
            line = line[1:]
            if line and line[0] in ['|', '｜']: ### FIXME: φ
                line = line[1:]
            else:
                raise ValueError('broken OR operator')
            line, and_expr = self.get_and_expr(line)
            buf.append(and_expr)
        if len(buf) == 2:
            return line, buf[1]
        return line, [buf]

    def get_and_expr(self, line):
        line, comp_expr = self.get_comp_expr(line)
        buf = [NODE_AND_EXPR, comp_expr]
        while line and line[0] in ['&', '＆']: ### FIXME: φ
            line = line[1:]
            if line and line[0] in ['&', '＆']: ### FIXME: φ
                line = line[1:]
            else:
                raise ValueError('broken AND operator')
            line, comp_expr = self.get_comp_expr(line)
            buf.append(comp_expr)
        if len(buf) == 2:
            return line, buf[1]
        return line, [buf]

    def get_comp_expr(self, line):
        line, buf = self.get_add_expr(line)
        if line and line[0] in ['<', '＜']: ### FIXME: φ
            line = line[1:]
            op = '<'
            if line and line[0] in ['=', '＝']: ### FIXME: φ
                line = line[1:]
                op = '<='
            line, add_expr = self.get_add_expr(line)
            return line, [[NODE_COMP_EXPR, buf, op, add_expr]]
        elif line and line[0] in ['>', '＞']: ### FIXME: φ
            line = line[1:]
            op = '>'
            if line and line[0] in ['=', '＝']: ### FIXME: φ
                line = line[1:]
                op = '>='
            line, add_expr = self.get_add_expr(line)
            return line, [[NODE_COMP_EXPR, buf, op, add_expr]]
        elif line and line[0] in ['=', '＝']: ### FIXME: φ
            line = line[1:]
            if line and line[0] in ['=', '＝']: ### FIXME: φ
                line = line[1:]
            else:
                raise ValueError('broken EQUAL operator')
            line, add_expr = self.get_add_expr(line)
            return line, [[NODE_COMP_EXPR, buf, '==', add_expr]]
        elif line and line[0] in ['!', '！']: ### FIXME: φ
            line = line[1:]
            if line and line[0] in ['=', '＝']: ### FIXME: φ
                line = line[1:]
            else:
                raise ValueError('broken NOT EQUAL operator')
            line, add_expr = self.get_add_expr(line)
            return line, [[NODE_COMP_EXPR, buf, '!=', add_expr]]
        return line, buf

    def get_add_expr(self, line):
        line, mul_expr = self.get_mul_expr(line)
        buf = [NODE_ADD_EXPR, mul_expr]
        while line and line[0] in ['+', '＋', '-', '−']: ### FIXME: φ
            if line[0] in ['+', '＋']:
                buf.append('+')
            else:
                buf.append('-')
            line = line[1:]
            line, mul_expr = self.get_mul_expr(line)
            buf.append(mul_expr)
        if len(buf) == 2:
            return line, buf[1]
        return line, [buf]

    def get_mul_expr(self, line):
        line, pow_expr = self.get_pow_expr(line)
        buf = [NODE_MUL_EXPR, pow_expr]
        while line and \
              line[0] in ['*', '＊', '×', '/', '／', '÷', '%', '％']: ### FIXME: φ
            if line[0] in ['*', '＊', '×']:
                buf.append('*')
            elif line[0] in ['/', '／', '÷']:
                buf.append('/')
            else:
                buf.append('%')
            line = line[1:]
            line, pow_expr = self.get_pow_expr(line)
            buf.append(pow_expr)
        if len(buf) == 2:
            return line, buf[1]
        return line, [buf]

    def get_pow_expr(self, line):
        line, unary_expr = self.get_unary_expr(line)
        buf = [NODE_POW_EXPR, unary_expr]
        while line and line[0] in ['^', '＾']: ### FIXME: φ
            line = line[1:]
            line, unary_expr = self.get_unary_expr(line)
            buf.append(unary_expr)
        if len(buf) == 2:
            return line, buf[1]
        return line, [buf]

    def get_unary_expr(self, line):
        if line and line[0] in ['-', '−']: ### FIXME: φ
            line = line[1:]
            line, unary_expr = self.get_unary_expr(line)
            return line, [[NODE_UNARY_EXPR, '-', unary_expr]]
        if line and line[0] in ['!', '！']: ### FIXME: φ
            line = line[1:]
            line, unary_expr = self.get_unary_expr(line)
            return line, [[NODE_UNARY_EXPR, '!', unary_expr]]
        if line and line[0] == '(': ### FIXME: φ
            line = line[1:]
            line, buf = self.get_or_expr(line)
            if line and line[0] == ')': ### FIXME: φ
                line = line[1:]
            else:
                raise ValueError('expected a close paren')
            return line, buf
        return self.get_factor(line)

    operators = [
        '|', '｜', '&', '＆', '<', '＜', '>', '＞', '=', '＝', '!', '！',
        '+', '＋', '-', '−', '*', '＊', '×', '/', '／', '÷', '%', '％',
        '^', '＾', '(', ')']

    def get_factor(self, line):
        buf = []
        while line and line[0] not in self.operators:
            if line and line[0] == '（': ### FIXME: φ
                line = self.parse_parenthesis(line, buf)
                continue
            text = []
            while line and line[0] not in self.operators and line[0] != '（': ### FIXME: φ
                text.append(line[0])
                line = line[1:]
            if text:
                buf.append([NODE_TEXT, text])
        if not buf:
            raise ValueError('expected a constant')
        return line, buf

    def print_nodelist(self, node_list, depth=0):
        for node in node_list:
            indent = '  ' * depth
            if node[0] == NODE_TEXT:
                print(''.join((indent, 'NODE_TEXT "{0}"'.format(''.join(
                                    [x.encode('utf-8') for x in node[1]])))))
            elif node[0] == NODE_REF:
                print(''.join((indent, 'NODE_REF')))
                self.print_nodelist(node[1], depth + 1)
            elif node[0] == NODE_CALL:
                print(''.join((indent, 'NODE_CALL')))
                for i in range(len(node[2])):
                    self.print_nodelist(node[2][i], depth + 1)
            elif node[0] == NODE_SAORI:
                print(''.join((indent, 'NODE_SAORI')))
                for i in range(len(node[2])):
                    self.print_nodelist(node[2][i], depth + 1)
            elif node[0] == NODE_SIDE:
                print(''.join((indent, 'NODE_SIDE')))
            elif node[0] == NODE_ASSIGNMENT:
                print(''.join((indent, 'NODE_ASSIGNMENT')))
                print(''.join((indent, 'variable')))
                self.print_nodelist(node[1], depth + 1)
                print(''.join((indent, 'value')))
                self.print_nodelist(node[2], depth + 1)
            elif node[0] == NODE_JUMP:
                print(''.join((indent, 'NODE_JUMP')))
                print(''.join((indent, 'name')))
                self.print_nodelist(node[1], depth + 1)
                if node[2] is not None:
                    print(''.join((indent, 'condition')))
                    self.print_nodelist(node[2], depth + 1)
            elif node[0] == NODE_SEARCH:
                print(''.join((indent, 'NODE_SEARCH')))
            elif node[0] == NODE_CHOICE:
                print(''.join((indent, 'NODE_CHOICE')))
                print(''.join((indent, 'label')))
                self.print_nodelist(node[1], depth + 1)
                if node[2] is not None:
                    print(''.join((indent, 'id')))
                    self.print_nodelist(node[2], depth + 1)
            elif node[0] == NODE_OR_EXPR:
                print(''.join((indent, 'NODE_OR_EXPR')))
                self.print_nodelist(node[1], depth + 1)
                for i in range(2, len(node)):
                    print(''.join((indent, 'op ||')))
                    self.print_nodelist(node[i], depth + 1)
            elif node[0] == NODE_AND_EXPR:
                print(''.join((indent, 'NODE_ADD_EXPR')))
                self.print_nodelist(node[1], depth + 1)
                for i in range(2, len(node)):
                    print(''.join((indent, 'op &&')))
                    self.print_nodelist(node[i], depth + 1)
            elif node[0] == NODE_COMP_EXPR:
                print(''.join((indent, 'NODE_COMP_EXPR')))
                self.print_nodelist(node[1], depth + 1)
                print(''.join((indent, 'op')), node[2])
                self.print_nodelist(node[3], depth + 1)
            elif node[0] == NODE_ADD_EXPR:
                print(''.join((indent, 'NODE_ADD_EXPR')))
                self.print_nodelist(node[1], depth + 1)
                for i in range(2, len(node), 2):
                    print(''.join((indent, 'op')), node[i])
                    self.print_nodelist(node[i + 1], depth + 1)
            elif node[0] == NODE_MUL_EXPR:
                print(''.join((indent, 'NODE_MUL_EXPR')))
                self.print_nodelist(node[1], depth + 1)
                for i in range(2, len(node), 2):
                    print(''.join((indent, 'op')), node[i])
                    self.print_nodelist(node[i + 1], depth + 1)
            elif node[0] == NODE_POW_EXPR:
                print(''.join((indent, 'NODE_POW_EXPR')))
                self.print_nodelist(node[1], depth + 1)
                for i in range(2, len(node)):
                    print(''.join((indent, 'op ^')))
                    self.print_nodelist(node[i], depth + 1)
            elif node[0] == NODE_UNARY_EXPR:
                print(''.join((indent, 'NODE_UNARY_EXPR')))
                print(''.join((indent, 'op')), node[1])
                self.print_nodelist(node[2], depth + 1)
            else:
                raise RuntimeError('should not reach here')

# expression := or_expr
# or_expr    := and_expr ( op_op and_expr )*
# or_op      := "｜｜"
# and_expr   := comp_expr ( and_op comp_expr )*
# and_op     := "＆＆"
# comp_expr  := add_expr ( comp_op add_expr )?
# comp_op    := "＜" | "＞" | "＜＝" | "＞＝" | "＝＝" | "！＝"
# add_expr   := mul_expr ( add_op mul_expr )*
# add_op     := "＋" | "−"
# mul_expr   := pow_expr ( mul_op pow_expr )*
# mul_op     := "×" | "÷" | "＊" | "／" | "％"
# pow_expr   := unary_expr ( pow_op unary_expr )*
# pow_op     := "＾"
# unary_expr := unary_op unary_expr | "(" or_expr ")" | factor
# unary_op   := "−" | "！"
# factor     := ( constant | reference )*


###   INTERPRETER   ###

class Satori:

    DBNAME = b'satori_savedata.txt'
    EDBNAME = b'satori_savedata.sat'

    def __init__(self, satori_dir=os.fsencode(os.curdir)):
        self.satori_dir = satori_dir
        self.dbpath = os.path.join(satori_dir, self.DBNAME)
        self.saori_function = {}
        self.parser = Parser()
        self.reset()

    def reset(self):
        self.word = {}
        self.talk = {}
        self.variable = {}
        self.replace_filter = Filter([])
        self.reset_surface = 1
        self.mouse_move_count = {}
        self.mouse_wheel_count = {}
        self.touch_threshold = 60
        self.touch_timeout = 2
        self.current_surface = [0, 10]
        self.default_surface = [0, 10]
        self.add_to_surface = [0, 0]
        self.newline = r'\n[half]'
        self.newline_script = ''
        self.save_interval = 0
        self.save_timer = 0
        self.url_list = {}
        self.boot_script = None
        self.script_history = [None] * 64
        self.wait_percent = 100
        self.random_talk = -1
        self.reserved_talk = {}
        self.silent_time = 0
        self.choice_id = None
        self.choice_label = None
        self.choice_number = None
        self.timer = {}
        self.time_start = None
        self.runtime = 0 # accumulative
        self.runcount = 1 # accumulative
        self.folder_change = 0
        self.saori_value = {}

    def load(self):
        buf = []
        for path in list_dict(self.satori_dir):
            filename = os.path.basename(path)
            if filename == b'replace.txt':
                self.parser.load_replace_file(path)
            elif filename == b'replace_after.txt':
                self.load_replace_file(path)
            elif filename in [b'satori_conf.txt', b'satori_conf.sat']:
                self.load_config_file(path)
            else:
                buf.append(path)
        for path in buf:
            self.parser.read(path)
        self.talk, self.word = self.parser.get_dict()
        self.load_database()
        self.time_start = time.time()
        self.get_event_response('OnSatoriLoad')
        self.boot_script = self.get_event_response('OnSatoriBoot')

    def load_config_file(self, path):
        parser = Parser()
        parser.read(path)
        talk, word = parser.get_dict()
        for nodelist in talk.get('初期化', []):
            self.expand(nodelist)

    def load_replace_file(self, path):
        self.replace_filter = Filter(read_tab_file(path))

    def load_database(self):
        if  self.variable.get('セーブデータ暗号化') == '有効':
            encrypted = 1
            self.dbpath = os.path.join(self.satori_dir, self.EDBNAME)
        else:
            encrypted = 0
            self.dbpath = os.path.join(self.satori_dir, self.DBNAME)
        try:
            database = read_tab_file(self.dbpath, encrypted)
        except IOError:
            database = []
        for name, value in database:
            if name.startswith('b\'') or name.startswith('b"'):
                continue # XXX: 文字コード変換に問題のあったバージョン対策
            self.assign(name, value)

    def save_database(self):
        if  self.variable.get('セーブデータ暗号化') == '有効':
            encrypted = 1
            self.dbpath = os.path.join(self.satori_dir, self.EDBNAME)
        else:
            encrypted = 0
            self.dbpath = os.path.join(self.satori_dir, self.DBNAME)
        try:
            with open(self.dbpath, 'wb') as f:
                for name, value in self.variable.items():
                    if name in ['前回終了時サーフェス0',
                                '前回終了時サーフェス1',
                                'デフォルトサーフェス0',
                                'デフォルトサーフェス1']:
                        continue
                    line = '{0}\t{1}'.format(name, value)
                    line = line.encode('CP932')
                    if encrypted:
                        line = ''.join(encrypt(encrypt(line)))
                    f.write(line)
                    f.write(b'\r\n')
                for side in [0, 1]:
                    name = 'デフォルトサーフェス{0:d}'.format(side)
                    value = self.to_zenkaku('{0:d}'.format(self.default_surface[side]))
                    line = '{0}\t{1}'.format(name, value)
                    line = line.encode('CP932')
                    if encrypted:
                        line = ''.join(encrypt(encrypt(line)))
                    f.write(line)
                    f.write(b'\r\n')
                for side in [0, 1]:
                    name = '前回終了時サーフェス{0:d}'.format(side)
                    value = self.to_zenkaku('{0:d}'.format(self.current_surface[side]))
                    line = '{0}\t{1}'.format(name, value)
                    line = line.encode('CP932')
                    if encrypted:
                        line = ''.join(encrypt(encrypt(line)))
                    f.write(line)
                    f.write(b'\r\n')
                name = '起動回数'
                value = self.to_zenkaku('{0:d}'.format(self.runcount))
                line = '{0}\t{1}'.format(name, value)
                line = line.encode('CP932')
                if encrypted:
                    line = ''.join(encrypt(encrypt(line)))
                f.write(line)
                f.write(b'\r\n')
                for name in self.timer:
                    value = self.to_zenkaku(self.timer[name])
                    line = '{0}\t{1}'.format(name, value)
                    line = line.encode('CP932')
                    if encrypted:
                        line = ''.join(encrypt(encrypt(line)))
                    f.write(line)
                    f.write(b'\r\n')
                for name in self.reserved_talk:
                    value = self.to_zenkaku(self.reserved_talk[name])
                    line = '{0}\t{1}'.format(''.join(('次から', value, '回目のトーク')), name)
                    line = line.encode('CP932')
                    if encrypted:
                        line = ''.join(encrypt(encrypt(line)))
                    f.write(line)
                    f.write(b'\r\n')
        except IOError:
            logging.debug('satori.py: cannot write {0}'.format(self.dbpath))
            return

    def finalize(self):
        self.get_event_response('OnSatoriUnload')
        accumulative_runtime = self.runtime + self.get_runtime()
        self.assign('単純累計秒', self.to_zenkaku(accumulative_runtime))
        self.save_database()

    # SHIORI/1.0 API
    def getaistringrandom(self):
        return self.get_script('')

    def getaistringfromtargetword(self, word):
        return ''

    def getdms(self):
        return ''

    def getword(self, word_type):
        return ''

    # SHIORI/2.2 API
    EVENT_MAP = {
        'OnFirstBoot':         '初回',
        'OnBoot':              '起動',
        'OnClose':             '終了',
        'OnGhostChanging':     '他のゴーストへ変更',
        'OnGhostChanged':      '他のゴーストから変更',
        'OnVanishSelecting':   '消滅指示',
        'OnVanishCancel':      '消滅撤回',
        'OnVanishSelected':    '消滅決定',
        'OnVanishButtonHold':  '消滅中断',
        }

    def get_event_response(self, event,
                           ref0=None, ref1=None, ref2=None, ref3=None,
                           ref4=None, ref5=None, ref6=None, ref7=None):
        self.event = event
        self.reference = [ref0, ref1, ref2, ref3, ref4, ref5, ref6, ref7]
        tail = '\e'
        if event == 'OnUpdateReady':
            try:
                ref0 = str(int(ref0) + 1)
                self.reference[0] = ref0
            except:
                pass
        elif event == 'OnMouseMove':
            key = (ref3, ref4) # side, part
            count, timestamp = self.mouse_move_count.get(key, (0, 0))
            if int(time.time() - timestamp) > self.touch_timeout:
                count = 0
            count += 1
            if count >= self.touch_threshold:
                event = '{0}{1}なでられ'.format(str(ref3), str(ref4))
                count = 0
            self.mouse_move_count[key] = (count, time.time())
        elif event == 'OnMouseWheel':
            key = (ref3, ref4) # side, part
            count, timestamp = self.mouse_wheel_count.get(key, (0, 0))
            if int(time.time() - timestamp) > 2:
                count = 0
            count += 1
            if count >= 2:
                event = '{0}{1}ころころ'.format(str(ref3), str(ref4))
                count = 0
            self.mouse_wheel_count[key] = (count, time.time())
        elif event == 'OnSecondChange':
            self.silent_time += 1
            if self.save_interval > 0:
               self.save_timer -= 1
               if self.save_timer <= 0:
                   self.save_database()
                   self.save_timer = self.save_interval
            if ref3 != '0': # cantalk
                # check random talk timer
                if self.random_talk == 0:
                    self.reset_random_talk_interval()
                elif self.random_talk > 0:
                    self.random_talk -= 1
                    if self.random_talk == 0:
                        event = self.get_reserved_talk()
                        if event:
                            self.reference[0] = self.to_zenkaku(1)
                        else:
                            self.reference[0] = self.to_zenkaku(0)
                        self.reference[1] = event
                        script = self.get_script('OnTalk')
                        if script:
                            self.script_history.pop(0)
                            self.script_history.append(script)
                            return script
                        self.reference[0] = ref0
                        self.reference[1] = ref1
            # check user-defined timers
            for name in self.timer.keys():
                count = self.timer[name] - 1
                if count > 0:
                    self.timer[name] = count
                elif ref3 != '0': # cantalk
                    del self.timer[name]
                    event = name[:-6]
                    break
        elif event == 'OnSurfaceChange':
            self.current_surface[0] = ref0
            self.current_surface[1] = ref1
        elif event == 'OnChoiceSelect':
            self.choice_id = ref0
            self.choice_label = ref1
            self.choice_number = ref2
            if 'OnChoiceSelect' not in self.talk:
                event = ref0
        elif event == 'OnChoiceEnter':
            self.choice_id = ref1
            self.choice_label = ref0
            self.choice_number = ref2
        elif event == 'OnAnchorSelect':
            if ref0 in self.talk:
                event = ref0
        elif event in ['sakura.recommendsites', 'sakura.portalsites',
                       'kero.recommendsites']:
            return self.get_url_list(event)
        elif event == 'OnRecommandedSiteChoice':
            script = self.get_url_script(ref0, ref1)
            if script:
                self.script_history.pop(0)
                self.script_history.append(script)
            return script
        elif event in self.EVENT_MAP:
            if event in ['OnBoot', 'OnGhostChanged']:
                if self.boot_script:
                    script = self.boot_script
                    self.boot_script = None
                    self.script_history.pop(0)
                    self.script_history.append(script)
                    return script
            if event in ['OnClose', 'OnGhostChanging']:
                if event == 'OnClose':
                    tail = r'\-\e'
                script = self.get_script('OnSatoriClose', tail=tail)
                if script:
                    self.script_history.pop(0)
                    self.script_history.append(script)
                    return script
            if event not in self.talk:
                event = self.EVENT_MAP[event]
#        print('EVENT:', event)
        script = self.get_script(event, tail=tail)
        if script:
            self.script_history.pop(0)
            self.script_history.append(script)
        return script

    # SHIORI/2.4 API
    def teach(self, word):
        name = self.variable.get('教わること')
        if name is not None:
            self.variable[name] = word
            script = self.get_script(''.join((name, 'を教えてもらった')))
            self.script_history.pop(0)
            self.script_history.append(script)
            return script
        return None

    # SHIORI/2.5 API
    def getstring(self, name):
        word = self.word.get(name)
        if word is not None:
            return self.expand(random.choice(word))
        return None

    # internal
    def get_reserved_talk(self):
        reserved = None
        for key in self.reserved_talk:
            self.reserved_talk[key] -= 1
            if self.reserved_talk[key] <= 0:
                reserved = key
        if reserved is not None:
            del self.reserved_talk[reserved]
        else:
            reserved = ''
        return reserved

    re_reservation = re.compile('次から((０|１|２|３|４|５|６|７|８|９|[0-9])+)(〜((０|１|２|３|４|５|６|７|８|９|[0-9])+))?回目のトーク')

    def assign(self, name, value):
        if name.endswith('タイマ'):
            if name[:-3] in self.talk:
                self.add_timer(name, value)
        elif name == '全タイマ解除':
            if value == '実行':
                self.delete_all_timers()
        elif name == '辞書リロード':
            if value == '実行':
                self.reload()
        elif name == '手動セーブ':
            if value == '実行':
                self.save_database()
        elif self.re_reservation.match(name):
            if not value:
                return None
            match = self.re_reservation.match(name)
            number = self.to_integer(match.group(1))
            if match.group(4) is not None:
                number = random.randint(number,
                                        self.to_integer(match.group(4)))
            while 1:
                for key in self.reserved_talk:
                    if self.reserved_talk[key] == number:
                        number += 1
                        break
                else:
                    break
            self.reserved_talk[value] = number
        elif name == '次のトーク':
            if not value:
                return None
            number = 1
            while 1:
                for key in self.reserved_talk:
                    if self.reserved_talk[key] == number:
                        number += 1
                        break
                else:
                    break
            self.reserved_talk[value] = number
        elif name == 'トーク予約のキャンセル':
            if value == '＊':
                self.reserved_talk = {}
            elif value in self.reserved_talk:
                del self.reserved_talk[value]
        elif name == '起動回数':
            self.runcount = self.to_integer(value) + 1
        elif not value:
            if name in self.variable:
                del self.variable[name]
        else:
            self.variable[name] = value
            if name in ['喋り間隔', '喋り間隔誤差']:
                self.reset_random_talk_interval()
            elif name == '教わること':
                return r'\![open,teachbox]'
            elif name == '会話時サーフェス戻し':
                if value == '有効':
                    self.reset_surface = 1
                elif value == '無効':
                    self.reset_surface = 0
            elif name == 'デフォルトサーフェス0':
                value = self.to_integer(value)
                if value is not None:
                    self.default_surface[0] = value
            elif name == 'デフォルトサーフェス1':
                value = self.to_integer(value)
                if value is not None:
                    self.default_surface[1] = value
            elif name == 'サーフェス加算値0':
                value = self.to_integer(value)
                if value is not None:
                    self.default_surface[0] = value
                    self.add_to_surface[0] = value
            elif name == 'サーフェス加算値1':
                value = self.to_integer(value)
                if value is not None:
                    self.default_surface[1] = value
                    self.add_to_surface[1] = value
            elif name == '単純累計秒':
                self.runtime = self.to_integer(value)
            elif name == 'なでられ反応回数':
                self.touch_threshold = self.to_integer(value)
            elif name == 'なでられ持続秒数':
                self.touch_timeout = self.to_integer(value)
            elif name == 'スコープ切り換え時':
                self.newline = value
            elif name == 'さくらスクリプトによるスコープ切り換え時':
                self.newline_script = value
            elif name == '自動挿入ウェイトの倍率':
                if value.endswith('%'):
                    value = value[:-1]
                elif value.endswith('％'):
                    value = value[:-1]
                value = self.to_integer(value)
                if value is not None and value >= 0 and value <= 1000:
                    self.wait_percent = value
            elif name == '自動セーブ間隔':
                self.save_interval = self.to_integer(value)
                self.save_timer = self.save_interval
            elif name == '辞書フォルダ':
                self.folder_change = 1
        return None

    def change_folder(self):
        value = self.variable.get('辞書フォルダ')
        dir_list = value.split(',')
        self.parser = Parser()
        self.parser.set_saori(self.saori_function.keys())
        for path in list_dict(self.satori_dir):
            filename = os.path.basename(path)
            if filename == b'replace.txt':
                self.parser.load_replace_file(path)
            elif filename == b'replace_after.txt':
                self.load_replace_file(path)
        buf = []
        for dir_ in dir_list:
            dir_ = get_normalized_path(dir_)
            dict_dir = os.path.join(self.satori_dir, dir_)
            if not os.path.isdir(dict_dir):
                logging.debug('satori.py: cannot read {0}'.format(dict_dir))
                continue
            for path in list_dict(dict_dir):
                filename = os.path.basename(path)
                if filename == b'replace.txt': ## XXX
                    self.parser.load_replace_file(path)
                elif filename == b'replace_after.txt': ## XXX
                    self.load_replace_file(path)
                elif filename in [b'satori_conf.txt', b'satori_conf.sat']:
                    pass
                else:
                    buf.append(path)
        for path in buf:
            self.parser.read(path)
        self.talk, self.word = self.parser.get_dict()

    def reload(self):
        self.finalize()
        self.reset()
        self.load()

    def reset_random_talk_interval(self):
        interval = self.get_integer('喋り間隔', 'ignore')
        if interval is None or interval == 0:
            self.random_talk = -1
            return
        rate = self.get_integer('喋り間隔誤差', 'ignore')
        if rate is None:
            rate = 0.1
        else:
            rate = min(max(rate, 1), 100) / 100.0
        diff = int(interval * rate)
        self.random_talk = random.randint(interval - diff, interval + diff)

    def add_timer(self, name, value):
        count = self.to_integer(value)
        if count is None or count == 0:
            if name in self.timer:
                del self.timer[name]
        else:
            self.timer[name] = count

    def delete_all_timers(self):
        self.timer = {}

    def get_script(self, name, head=r'\1', tail=r'\e'):
        if self.reset_surface:
            self.current_reset_surface = [1, 1]
        else:
            self.current_reset_surface = [0, 0]
        script = self.get(name, default=None)
        if script is not None and script and script != '\\n':
            ##logging.debug('make("{0}")'.format(script.encode('utf-8')))
            return self.make(''.join((head, script, tail)))
        return None

    def get_url_list(self, name):
        self.url_list[name] = []
        list_ = self.talk.get(name)
        if list_ is None:
            return None
        url_list = ''
        for i in range(len(list_)-1, -1, -1):
            nodelist = list_[i]
            title = ''
            j = 0
            while j < len(nodelist):
                node = nodelist[j]
                j += 1
                if node[0] == NODE_TEXT:
                    if node[1] == ['\\n']:
                        break
                    else:
                        title = ''.join((title, ''.join(node[1])))
                else:
                   pass
            if not title:
                continue
            if title == '-':
                if url_list:
                    url_list = ''.join((url_list, chr(2)))
                url_list = ''.join((url_list, title, chr(1)))
                continue
            url = ''
            while j < len(nodelist):
                node = nodelist[j]
                j += 1
                if node[0] == NODE_TEXT:
                    if node[1] == ['\\n']:
                        break
                    else:
                        url = ''.join((url, ''.join(node[1])))
                else:
                   pass
            if not url:
                continue
            bannar = ''
            while j < len(nodelist):
                node = nodelist[j]
                j += 1
                if node[0] == NODE_TEXT:
                    if node[1] == ['\\n']:
                        break
                    else:
                        bannar = ''.join((bannar, ''.join(node[1])))
                else:
                   pass
            if nodelist[j:]:
                script = nodelist[j:]
            else:
                script = None
            self.url_list[name].append([title, url, bannar, script])
            if url_list:
                url_list = ''.join((url_list, chr(2)))
            url_list = ''.join((url_list, title, chr(1), url, chr(1), bannar))
        if not url_list:
            url_list = None
        return url_list

    def get_url_script(self, title, url):
        script = None
        if self.reset_surface:
            self.current_reset_surface = [1, 1]
        else:
            self.current_reset_surface = [0, 0]
        for key in self.url_list:
            for item in self.url_list[key]:
                if item[0] == title and item[1] == url:
                    if item[3] is not None:
                        script = self.expand(item[3])
                        if script is not None and script and script != '\\n':
                            script = self.make(''.join((r'\1', script, r'\e')))
                            break
        return script

    redundant_tags = [
        (re.compile(r'(\\[01hu])+(\\[01hu])'),   lambda m: m.group(2)),
        (re.compile(r'(\\n)+(\\e|$)'),           lambda m: m.group(2)),
        (re.compile(r'(\\e)+'),                  lambda m: m.group(1)),
        ]
    re_newline = re.compile(r'((\\n)*)(\\e)')
    re_0 = re.compile(r'\\[0h]')
    re_1 = re.compile(r'\\[1u]')
    re_wait_after = re.compile(r'、|。|，|．')
    re_wait_before = re.compile(r'\\[01hunce]')
    re_tag = re.compile(r'\\[ehunjcxtqzy*v0123456789fmia!&+---]|'
                        r'\\[sb][0-9]?|\\w[0-9]|\\_[wqslvVbe+cumna]|'
                        r'\\__[ct]|\\URL')

    def make(self, script):
        if script is None:
            return None
        # make anchor
        buf = []
        i = 0
        while 1:
            match = self.re_tag.match(script, i)
            if match:
                start = match.start()
                end = match.end()
                buf.append(self.parser.anchor_filter.apply(script[i:start]))
                buf.append(script[start:end])
                i = end
            else:
                buf.append(self.parser.anchor_filter.apply(script[i:]))
                break
        script = ''.join(buf)
        # apply replace_after.txt
        script = self.replace_filter.apply(script)
        # remove redundant tags
        for pattern, replace in self.redundant_tags:
            script, count = pattern.subn(replace, script)
        # remove redundant newline tags
        match = self.re_newline.search(script)
        if match:
            tag = match.group(3)
            if tag == r'\e':
                script = ''.join((script[:match.start()],
                                  tag, script[match.end():]))
            else:
                raise RuntimeError('should not reach here')
        # insert newline
        i = 1
        while 1:
            match = self.re_0.search(script, i)
            if match:
                end = match.end()
                match = self.re_0.search(script, end)
                if match:
                    start = match.start()
                    if start < len(self.newline) or \
                       script[start - len(self.newline):start] != self.newline:
                       script = r'{0}{1}{2}'.format(
                           script[:match.end()], self.newline_script,
                           script[match.end():])
                else:
                    break
                i = end
            else:
                break
        i = 1
        while 1:
            match = self.re_1.search(script, i)
            if match:
                end = match.end()
                match = self.re_1.search(script, end)
                if match:
                    start = match.start()
                    if start < len(self.newline) or \
                       script[start - len(self.newline):start] != self.newline:
                       script = r'{0}{1}{2}'.format(
                           script[:match.end()], self.newline_script,
                           script[match.end():])
                else:
                    break
                i = end
            else:
                break
        # insert waits
        buf = []
        n = 0
        i, j = 0, len(script)
        while i < j:
            match = self.re_wait_after.match(script, i)
            if match:
                buf.append(match.group())
                buf.extend(self.make_wait(n))
                n = 0
                i = match.end()
                continue
            match = self.re_wait_before.match(script, i)
            if match:
                buf.extend(self.make_wait(n))
                buf.append(match.group())
                n = 0
                i = match.end()
                continue
            if script[i] == '[':
                pos = script.find(']', i)
                if pos > i:
                    buf.append(script[i:pos + 1])
                    i = pos + 1
                    continue
            match = self.re_tag.match(script, i)
            if match:
                buf.append(script[i:match.end()])
                i = match.end()
            else:
                buf.append(script[i])
                n += 3
                i += 1
        return ''.join(buf)

    def make_wait(self, ms):
        buf = []
        n = int((ms + 25) * self.wait_percent / 100 / 50)
        while n > 0:
            buf.append(r'\w{0:d}'.format(min(n, 9)))
            n -= 9
        return buf

    def get(self, name, default=''):
        result = self.talk.get(name)
        if result is None:
            return default
        return self.expand(random.choice(result))

    def expand(self, nodelist, caller_history=None, side=1):
        if nodelist is None:
            return ''
        buf = []
        history = []
        talk = 0
        newline = [None, None]
        for node in nodelist:
            if node[0] == NODE_REF:
                if caller_history is not None:
                    value = self.get_reference(node[1], caller_history, side)
                else:
                    value = self.get_reference(node[1], history, side)
                if value:
                    talk = 1
                    buf.append(value)
                    history.append(value)
                    if caller_history is not None:
                        caller_history.append(value)
            elif node[0] == NODE_CALL:
                function = node[1]
                args = node[2]
                value = self.call_function(
                    function, args,
                    caller_history if caller_history is not None else history,
                    side)
                if value:
                    talk = 1
                    buf.append(value)
                    history.append(value)
                    if caller_history is not None:
                        caller_history.append(value)
            elif node[0] == NODE_SAORI:
                if self.variable.get('SAORI引数の計算') == '無効':
                    expand_only = 1
                else:
                    expand_only = 0
                if caller_history is not None:
                    value = self.call_saori(
                        node[1],
                        self.calc_args(node[2], caller_history, expand_only),
                        caller_history, side)
                else:
                    value = self.call_saori(
                        node[1],
                        self.calc_args(node[2], history, expand_only),
                        history, side)
                if value:
                    talk = 1
                    buf.append(value)
                    history.append(value)
                    if caller_history is not None:
                        caller_history.append(value)
            elif node[0] == NODE_TEXT:
                buf.extend(node[1])
                talk = 1
            elif node[0] == NODE_SIDE:
                if talk:
                    newline[side] = self.newline
                else:
                    newline[side] = None
                talk = 0
                if side == 0:
                    side = 1
                else:
                    side = 0
                buf.append(r'\{0:d}'.format(side))
                if self.current_reset_surface[side]:
                    buf.append(r'\s[{0:d}]'.format(self.default_surface[side]))
                    self.current_reset_surface[side] = 0
                if newline[side] is not None:
                    buf.append(r'{0}'.format(newline[side]))
            elif node[0] == NODE_ASSIGNMENT:
                value = self.expand(node[2])
                result = self.assign(self.expand(node[1]), value)
                if result is not None:
                    buf.append(result)
            elif node[0] == NODE_JUMP:
                if node[2] is None or \
                   self.expand(node[2]) not in ['０', '0']:
                    target = self.expand(node[1])
                    if target == 'OnTalk':
                        self.reference[1] = self.get_reserved_talk()
                        if self.reference[1]:
                            self.reference[0] = self.to_zenkaku(1)
                        else:
                            self.reference[0] = self.to_zenkaku(0)
                    script = self.get(target, default=None)
                    if script is not None and script and script != '\\n':
                        buf.append(''.join((r'\1', script)))
                        break
            elif node[0] == NODE_SEARCH: ## FIXME
                buf.append('')
            elif node[0] == NODE_CHOICE:
                label = self.expand(node[1])
                if node[2] is None:
                    id_ = label
                else:
                    id_ = self.expand(node[2])
                buf.append(r'\q[{0},{1}]\n'.format(label, id_))
                talk = 1
            elif node[0] == NODE_OR_EXPR:
                for i in range(1, len(node)):
                    if self.expand(node[i]) not in ['０', '0']:
                        buf.append('１')
                        break
                else:
                    buf.append('０')
            elif node[0] == NODE_AND_EXPR:
                for i in range(1, len(node)):
                    if self.expand(node[i]) in ['０', '0']:
                        buf.append('０')
                        break
                else:
                    buf.append('１')
            elif node[0] == NODE_COMP_EXPR:
                operand1 = self.expand(node[1])
                operand2 = self.expand(node[3])
                n1 = self.to_integer(operand1)
                n2 = self.to_integer(operand2)
                if not (n1 is None or n2 is None):
                    operand1 = n1
                    operand2 = n2
                elif n1 is None and n2 is None and \
                     node[2] in ['<', '>', '<=', '>=']:
                    operand1 = len(operand1)
                    operand2 = len(operand2)
                if node[2] == '==':
                    buf.append(self.to_zenkaku(int(operand1 == operand2)))
                elif node[2] == '!=':
                    buf.append(self.to_zenkaku(int(operand1 != operand2)))
                elif node[2] == '<':
                    buf.append(self.to_zenkaku(int(operand1 < operand2)))
                elif node[2] == '>':
                    buf.append(self.to_zenkaku(int(operand1 > operand2)))
                elif node[2] == '<=':
                    buf.append(self.to_zenkaku(int(operand1 <= operand2)))
                elif node[2] == '>=':
                    buf.append(self.to_zenkaku(int(operand1 >= operand2)))
                else:
                    raise RuntimeError('should not reach here')
            elif node[0] == NODE_ADD_EXPR:
                value_str = self.expand(node[1])
                value = self.to_integer(value_str)
                for i in range(2, len(node), 2):
                    operand_str = self.expand(node[i + 1])
                    operand = self.to_integer(operand_str)
                    if node[i] == '-':
                        if value is None or operand is None:
                            value_str = value_str.replace(operand_str, '')
                            value = None
                        else:
                            value -= operand
                            value_str = self.to_zenkaku(value)
                        continue
                    if value is None:
                        value = 0
                    if operand is None:
                        operand = 0
                    if node[i] == '+':
                        value += operand
                    else:
                        raise RuntimeError('should not reach here')
                if value is None:
                    buf.append(value_str)
                else:
                    buf.append(self.to_zenkaku(value))
            elif node[0] == NODE_MUL_EXPR:
                value_str = self.expand(node[1])
                value = self.to_integer(value_str)
                for i in range(2, len(node), 2):
                    operand_str = self.expand(node[i + 1])
                    operand = self.to_integer(operand_str)
                    if node[i] == '*':
                        if value is None and operand is None:
                            value_str = ''
                            value = None
                        elif value is None:
                            value_str *= operand
                            value = None
                        elif operand is None:
                            value_str *= operand_str
                            value = None
                        else:
                            value *= operand
                        continue
                    if value is None:
                        value = 0
                    if operand is None:
                        operand = 0
                    if node[i] == '/':
                        value //= operand
                    elif node[i] == '%':
                        value %= operand
                    else:
                        raise RuntimeError('should not reach here')
                if value is None:
                    buf.append(value_str)
                else:
                    buf.append(self.to_zenkaku(value))
            elif node[0] == NODE_POW_EXPR:
                value = self.to_integer(self.expand(node[1]))
                if value is None:
                    value = 0
                for i in range(2, len(node)):
                    operand = self.to_integer(self.expand(node[i]))
                    if operand is None:
                        operand = 0
                    value **= operand
                buf.append(self.to_zenkaku(value))
            elif node[0] == NODE_UNARY_EXPR:
                value = self.expand(node[2])
                if node[1] == '-':
                    value = self.to_integer(value)
                    if value is None:
                        value = 0
                    value = -value
                elif node[1] == '!':
                    value = int(value in ['０', '0'])
                else:
                    raise RuntimeError('should not reach here')
                buf.append(self.to_zenkaku(value))
            else:
                raise RuntimeError('should not reach here')
        return ''.join(buf).strip()

    re_random = re.compile('乱数((−|＋|[-+])?(０|１|２|３|４|５|６|７|８|９|[0-9])+)〜((−|＋|[-+])?(０|１|２|３|４|５|６|７|８|９|[0-9])+)')
    re_is_empty = re.compile('(変数|文|単語群)「(.*)」の存在')
    re_n_reserved = re.compile('次から((０|１|２|３|４|５|６|７|８|９|[0-9])+)回目のトーク')
    re_is_reserved = re.compile('トーク「(.*)」の予約有無')

    def get_reference(self, nodelist, history, side):
        key = self.expand(nodelist[1:-1], history)
        if key and key[0] in ['Ｒ', 'R']:
            n = self.to_integer(key[1:])
            if n is not None and 0 <= n < len(self.reference): ## FIXME
                if self.reference[n] is None:
                    return ''
                return str(self.reference[n])
        elif key and key[0] in ['Ｓ', 'S']:
            n = self.to_integer(key[1:])
            if n is not None:
                if key in self.saori_value:
                    if self.saori_value[key] is None:
                        return ''
                    return str(self.saori_value[key])
                else:
                    return ''
        elif key and key[0] in ['Ｈ', 'H']:
            ##logging.debug(''.join(('["', '", "'.join(history), '"]')))
            n = self.to_integer(key[1:])
            if n is not None and 1 <= n < len(history) + 1: ## FIXME
                return history[n-1]
        n = self.to_integer(key)
        if n is not None:
            return r'\s[{0:d}]'.format(n + self.add_to_surface[side])
        if key in self.word:
            return self.expand(random.choice(self.word[key]), history,
                               side=side)
        elif key in self.talk:
            self.reference = [None] * 8
            return self.expand(random.choice(self.talk[key]), side=1)
        elif key in self.variable:
            return self.variable[key]
        elif key in self.timer:
            return self.to_zenkaku(self.timer[key])
        elif self.is_reserved(key):
            return self.get_reserved(key)
        elif self.re_random.match(key):
            match = self.re_random.match(key)
            i = self.to_integer(match.group(1))
            j = self.to_integer(match.group(4))
            if i < j:
                return self.to_zenkaku(random.randint(i, j))
            else:
                return self.to_zenkaku(random.randint(j, i))
        elif self.re_n_reserved.match(key):
            match = self.re_n_reserved.match(key)
            number = self.to_integer(match.group(1))
            for key in self.reserved_talk:
                if self.reserved_talk[key] == number:
                    return key
            else:
                return ''
        elif self.re_is_reserved.match(key):
            match = self.re_is_reserved.match(key)
            name = match.group(1)
            if name in self.reserved_talk:
                return self.to_zenkaku(1)
            else:
                return self.to_zenkaku(0)
        elif self.re_is_empty.match(key):
            match = self.re_is_empty.match(key)
            type_ = match.group(1)
            name = match.group(2)
            if type_ == '変数':
                if name in self.variable:
                    return self.to_zenkaku(1)
                else:
                    return self.to_zenkaku(0)
            elif type_ == '文':
                if name in self.talk:
                    return self.to_zenkaku(1)
                else:
                    return self.to_zenkaku(0)
            elif type_ == '単語群':
                if name in self.word:
                    return self.to_zenkaku(1)
                else:
                    return self.to_zenkaku(0)
        return '（{0}）'.format(key)

    def calc_args(self, args, history, expand_only=0):
        buf = []
        for i in range(len(args)):
            value = self.expand(args[i], history)
            line = value ## FIXME
            if expand_only or not line:
                buf.append(value)
            elif line[0] in ['-', '−',  '+', '＋'] and len(line) == 1: # XXX
                buf.append(value)
            elif line[0] in self.NUMBER:
                try: ## FIXME
                    line, expr = self.parser.get_add_expr(line)
                    result = str(self.to_integer(self.expand(expr, history)))
                    if result is None:
                        buf.append(value)
                    else:
                        buf.append(result)
                except:
                    buf.append(value)
            else:
                buf.append(value)
        return buf

    def call_function(self, name, args, history, side):
        if name == '単語の追加':
            pass ## FIXME
        elif name == 'call':
            ref = self.expand(args[0], history)
            args = args[1:]
            for i in range(len(args)):
                name = ''.join(('Ａ', self.to_zenkaku(i)))
                self.variable[name] = self.expand(args[i], history)
            result = self.get_reference([[NODE_TEXT, '（'], [NODE_TEXT, ref],
                                         [NODE_TEXT, '）']], history, side)
            for i in range(len(args)):
                name = ''.join(('Ａ', self.to_zenkaku(i)))
                del self.variable[name]
            return result
        elif name == 'remember':
            number = self.to_integer(self.expand(args[0], history))
            if number > 0 and number <= 64 and self.script_history[-number]:
                return self.script_history[-number] 
            else:
                return ''
        elif name == 'loop':
            ref = self.expand(args[0], history)
            if len(args) < 2:
                return ''
            elif len(args) == 2:
                start = 1
                end = self.to_integer(self.expand(args[1], history)) + 1
                step = 1
            elif len(args) == 3:
                start = self.to_integer(self.expand(args[1], history))
                end = self.to_integer(self.expand(args[2], history)) + 1
                step = 1
            elif len(args) >= 4:
                start = self.to_integer(self.expand(args[1], history))
                end = self.to_integer(self.expand(args[2], history))
                step = self.to_integer(self.expand(args[3], history))
                if step > 0:
                    end = end + 1
                elif step < 0:
                    end = end - 1
                else:
                    return '' # infinite loop
            name = ''.join((ref, 'カウンタ'))
            buf = []
            for i in range(start, end, step):
                self.variable[name] = self.to_zenkaku(i)
                buf.append(self.get_reference([[NODE_TEXT, '（'],
                                               [NODE_TEXT, ref],
                                               [NODE_TEXT, '）']],
                                              history, side))
            return ''.join(buf)
        elif name == 'sync': ## FIXME
            pass
        elif name == 'set':
            if not args:
                name = ''
            else:
                name = self.expand(args[0], history)
            if len(args) < 2:
                value = ''
            else:
                value = self.expand(args[1], history)
            if name:
                if not value:
                    if name in self.variable:
                        del self.variable[name]
                else:
                    self.variable[name] = value
            return ''
        elif name == 'nop':
            for i in range(len(args)):
                self.expand(args[i], history)
            pass
        elif name == '合成単語群': ## FIXME: not tested
            words = []
            for i in range(len(args)): ## FIXME
                name = self.expand(args[i], history)
                word = self.word.get(name)
                if word is not None:
                    words.extend(word)
            if words:
                return self.expand(random.choice(words))
        elif name.endswith('の数'):
            if name[0] in ['R', 'Ｒ']:
                return len(self.reference)
            elif name[0] in ['A', 'Ａ']: # len(args)
                pass ## FIXME
            elif name[0] in ['S', 'Ｓ']:
                ##return len(self.saori_value)
                pass ## FIXME
            #elif name[0] in ['H', 'Ｈ']:
            #    return len(history)
            #elif name[0] in ['C', 'Ｃ']:
            #    return len(count)
            else: ## FIXME
                pass
        elif name == 'when':
            assert len(args) > 1
            condition = self.expand(args[0], history)
            if condition in ['０', '0']:
                if len(args) > 2:
                    return self.expand(args[2], history)
                else:
                    return ''
            else:
                return self.expand(args[1], history)
        elif name == 'times': ## FIXME
            print('TIMES:', len(args), args)
            pass
        elif name == 'while': ## FIXME
            print('WHILE:', len(args), args)
            pass
        elif name == 'for': ## FIXME
            print('FOR:', len(args), args)
            pass
        else:
            raise RuntimeError('should not reach here')
        return ''

    def call_saori(self, name, args, history, side):
        return ''

    def get_runtime(self):
        return int(time.time() - self.time_start)

    def get_integer(self, name, error='strict'):
        value = self.variable.get(name)
        if value is None:
            return None
        return self.to_integer(value, error)

    NUMBER = {
        '０': '0', '0': '0',
        '１': '1', '1': '1',
        '２': '2', '2': '2',
        '３': '3', '3': '3',
        '４': '4', '4': '4',
        '５': '5', '5': '5',
        '６': '6', '6': '6',
        '７': '7', '7': '7',
        '８': '8', '8': '8',
        '９': '9', '9': '9',
        '＋': '+', '+': '+',
        '−': '-', '-': '-',
        }

    def to_integer(self, line, error='strict'):
        buf = []
        for char in line:
            try:
                buf.append(self.NUMBER[char])
            except KeyError:
                if char in ['．', '.']: # XXX
                    buf.append('.')
                elif error == 'strict':
                    return None
        try:
            return int(''.join(buf))
        except ValueError:
            if '.' in buf: # XXX
                return int(float(''.join(buf)))
            return None

    def is_number(self, line):
        return self.to_integer(line) is not None

    ZENKAKU = {
        '0': '０', '1': '１', '2': '２', '3': '３', '4': '４',
        '5': '５', '6': '６', '7': '７', '8': '８', '9': '９',
        '+': '＋', '-': '−',
        }

    def to_zenkaku(self, s):
        buf = list(str(s))
        for i in range(len(buf)):
            buf[i] = self.ZENKAKU.get(buf[i], buf[i])
        return ''.join(buf)

    RESERVED = [
        '現在年', '現在月', '現在日', '現在曜日',
        '現在時', '現在分', '現在秒',
        '起動時', '起動分', '起動秒',
        '累計時', '累計分', '累計秒',
        'ＯＳ起動時', 'ＯＳ起動分', 'ＯＳ起動秒',
        '単純起動分', '単純起動秒',
        '単純累計分', '単純累計秒',
        '単純ＯＳ起動分', '単純ＯＳ起動秒',
        '最終トークからの経過秒',
        'サーフェス0', 'サーフェス1',
        '選択ＩＤ', '選択ラベル', '選択番号',
        '予約トーク数',
        '起動回数',
        'Sender', 'Event', 'Charset',
        'Reference0', 'Reference1', 'Reference2', 'Reference3',
        'Reference4', 'Reference5', 'Reference6', 'Reference7',
        'countTalk', 'countNoNameTalk', 'countEventTalk', 'countOtherTalk',
        'countWords', 'countWord', 'countVariable', 'countAnchor',
        'countParenthesis', 'countParentheres', 'countLine',
        ]

    def is_reserved(self, s):
        return s in self.RESERVED

    DAYOFWEEK = ['月', '火', '水', '木', '金', '土', '日']

    def get_reserved(self, s):
        now = time.localtime(time.time())
        if s == '現在年':
            return self.to_zenkaku(now[0])
        elif s == '現在月':
            return self.to_zenkaku(now[1])
        elif s == '現在日':
            return self.to_zenkaku(now[2])
        elif s == '現在時':
            return self.to_zenkaku(now[3])
        elif s == '現在分':
            return self.to_zenkaku(now[4])
        elif s == '現在秒':
            return self.to_zenkaku(now[5])
        elif s == '現在曜日':
            return self.DAYOFWEEK[now[6]]
        runtime = self.get_runtime()
        if s == '起動時':
            return self.to_zenkaku(runtime // 3600)
        elif s == '起動分':
            return self.to_zenkaku(runtime // 60 % 60)
        elif s == '起動秒':
            return self.to_zenkaku(runtime % 60)
        elif s == '単純起動分':
            return self.to_zenkaku(runtime // 60)
        elif s == '単純起動秒':
            return self.to_zenkaku(runtime)
        accumulative_runtime = self.runtime + self.get_runtime()
        if s == '累計時':
            return self.to_zenkaku(accumulative_runtime // 3600)
        elif s == '累計分':
            return self.to_zenkaku(accumulative_runtime // 60 % 60)
        elif s == '累計秒':
            return self.to_zenkaku(accumulative_runtime % 60)
        elif s == '単純累計分':
            return self.to_zenkaku(accumulative_runtime // 60)
        elif s == '単純累計秒':
            return self.to_zenkaku(accumulative_runtime)
        elif s == '起動回数':
            return self.to_zenkaku(self.runcount)
        elif s == '最終トークからの経過秒':
            return self.to_zenkaku(self.silent_time)
        elif s == 'サーフェス0':
            return str(self.current_surface[0])
        elif s == 'サーフェス1':
            return str(self.current_surface[1])
        elif s == '選択ＩＤ':
            if self.choice_id is None:
                return ''
            else:
                return self.choice_id
        elif s == '選択ラベル':
            if self.choice_label is None:
                return ''
            else:
                return self.choice_label
        elif s == '選択番号':
            if self.choice_number is None:
                return ''
            else:
                return self.to_zenkaku(self.choice_number)
        elif s == '予約トーク数':
            return self.to_zenkaku(len(self.reserved_talk))
        elif s == 'Sender':
            return 'ninix'
        elif s == 'Event':
            return self.event
        elif s == 'Charset':
            return 'UTF-8'
        elif s.startswith('Reference'):
            n = int(s[9:])
            if self.reference[n] is None:
                return ''
            return str(self.reference[n])
        elif s.startswith('count'):
            return self.to_zenkaku(self.parser.get_count(s[5:]))
        return str('？', 'utf-8')


class Shiori(Satori):

    def __init__(self, dll_name):
        self.dll_name = dll_name
        self.saori = None
        self.saori_function = {}

    def use_saori(self, saori):
        self.saori = saori

    def load(self, satori_dir):
        Satori.__init__(self, satori_dir)
        self.saori_library = SatoriSaoriLibrary(self.saori, self)
        Satori.load(self)
        return 1

    def load_config_file(self, path):
        parser = Parser()
        parser.read(path)
        talk, word = parser.get_dict()
        for nodelist in talk.get('初期化', []):
            self.expand(nodelist)
        self.saori_function = {}
        for nodelist in word.get('SAORI', []):
            if nodelist[0][0] == NODE_TEXT:
                list_ = ''.join(nodelist[0][1]).split(',')
                if len(list_) >= 2 and list_[0] and list_[1]:
                    head, tail = os.path.split(os.fsencode(list_[1]))
                    saori_dir = os.path.join(self.satori_dir, head)
                    result =  self.saori_library.load(list_[1], saori_dir)
                    if result:
                        self.saori_function[list_[0]] = list_[1:]
                    else:
                        self.saori_function[list_[0]] = None
                        ##logging.error('satori.py: cannot load {0}'.format(list_[1]))
        self.parser.set_saori(self.saori_function.keys())

    def reload(self):
        self.finalize()
        self.reset()
        self.load(self.satori_dir)

    def unload(self):
        Satori.finalize(self)
        self.saori_library.unload()

    def find(self, top_dir, dll_name):
        result = 0
        if list_dict(top_dir):
            result = 100
        return result

    def show_description(self):
        logging.info(
            'Shiori: SATORI compatible module for ninix\n'
            '        Copyright (C) 2002 by Tamito KAJIYAMA\n'
            '        Copyright (C) 2002, 2003 by MATSUMURA Namihiko\n'
            '        Copyright (C) 2002-2013 by Shyouzou Sugitani\n'
            '        Copyright (C) 2003, 2004 by Shun-ichi TAHARA')

    def request(self, req_string):
        if self.folder_change:
            self.change_folder()
            self.folder_change = 0
        header = str(req_string, 'UTF-8').splitlines()
        req_header = {}
        line = header.pop(0)
        if line:
            line = line.strip()
            req_list = line.split()
            if len(req_list) >= 2:
                command = req_list[0].strip()
                protocol = req_list[1].strip()
            for line in header:
                line = line.strip()
                if not line:
                    continue
                if ':' not in line:
                    continue
                key, value = [x.strip() for x in line.split(':', 1)]
                try:
                    value = int(value)
                except:
                    value = str(value)
                req_header[key] = value
        result = ''
        to = None
        if 'ID' in req_header:
            if req_header['ID'] == 'dms':
                result = self.getdms()
            elif req_header['ID'] == 'OnAITalk':
                result = self.getaistringrandom()
            elif req_header['ID'] in ['\\ms', '\\mz', '\\ml', '\\mc', '\\mh', \
                                      '\\mt', '\\me', '\\mp', '\\m?']:
                result = self.getword(req_header['ID'])
            elif req_header['ID'] == 'otherghostname': ## FIXME
                ##otherghost = []
                ##for n in range(128):
                ##    if ''.join(('Reference', str(n))) in req_header:
                ##        otherghost.append(req_header[''.join(('Reference',
                ##                                              str(n)])))
                ##result = self.otherghostname(otherghost)
                pass
            elif req_header['ID'] == 'OnTeach':
                if 'Reference0' in req_header:
                    self.teach(req_header['Reference0'])
            else:
                result = self.getstring(req_header['ID'])
                if result is None:
                    ref = []
                    for n in range(8):
                        if ''.join(('Reference', str(n))) in req_header:
                            ref.append(req_header[
                                ''.join(('Reference', str(n)))])
                        else:
                            ref.append(None)
                    ref0, ref1, ref2, ref3, ref4, ref5, ref6, ref7 = ref
                    result = self.get_event_response(
                        req_header['ID'], ref0, ref1, ref2, ref3, ref4,
                        ref5, ref6, ref7)
            if result is None:
                result = ''
            to = None ##self.communicate_to() ## FIXME
        if result != '':
            self.silent_time = 0
        result = 'SHIORI/3.0 200 OK\r\n' \
                 'Sender: Satori\r\n' \
                 'Charset: UTF-8\r\n' \
                 'Value: {0}\r\n'.format(result)
        if to is not None:
            result = ''.join((result, 'Reference0: {0}\r\n'.format(to)))
        result = ''.join((result, '\r\n'))
        return result.encode('UTF-8')

    def call_saori(self, name, args, history, side):
        if name not in self.saori_function or \
           self.saori_function[name] is None:
            return ''
        saori_statuscode = ''
        saori_header = []
        saori_value = {}
        saori_protocol = ''
        req = 'EXECUTE SAORI/1.0\r\n' \
              'Sender: Satori\r\n' \
              'SecurityLevel: local\r\n' \
              'Charset: Shift_JIS\r\n' ## XXX
        default_args = self.saori_function[name][1:]
        n = len(default_args)
        for i in range(len(default_args)):
              req = ''.join((req, 'Argument{0}: {1}\r\n'.format(i, default_args[i])))
        for i in range(len(args)):
              argument = args[i]
              if argument:
                  req = ''.join((req,
                                 'Argument{0}: {1}\r\n'.format(i + n, argument)))
        req = ''.join((req, '\r\n'))
        response = self.saori_library.request(
            self.saori_function[name][0],
            req.encode('CP932', 'ignore'))
        header = response.splitlines()
        if header:
            line = header.pop(0)
            line = str(line, 'CP932').strip()
            if ' ' in line:
                saori_protocol, saori_statuscode = [x.strip() for x in line.split(' ', 1)]
            for line in header:
                line = str(line, 'CP932').strip()
                if not line:
                    continue
                if ':' not in line:
                    continue
                key, value = line.split(':', 1)
                key = key.strip()
                if key:
                    saori_header.append(key)
                    saori_value[key] = value.strip()
        for key in saori_value:
            if not key.startswith('Value'):
                continue
            try:
                i = int(key[5:])
            except:
                continue
            name = ''.join(('Ｓ', self.to_zenkaku(i)))
            self.saori_value[name] = saori_value[key] # overwrite
        if 'Result' in saori_value:
            return saori_value['Result']
        else:
            return ''


class SatoriSaoriLibrary:

    def __init__(self, saori, satori):
        self.saori_list = {}
        self.saori = saori
        self.satori = satori

    def load(self, name, top_dir):
        result = 0
        if self.saori and name not in self.saori_list:
            module = self.saori.request(name)
            if module:
                self.saori_list[name] = module
        if name in self.saori_list:
            result = self.saori_list[name].load(top_dir)
        return result

    def unload(self):
        for key in self.saori_list.keys():
            self.saori_list[key].unload()
        return None

    def request(self, name, req):
        result = '' # FIXME
        if name and name in self.saori_list:
            result = self.saori_list[name].request(req)
        return result


def satori_open(top_dir):
    satori = Satori(top_dir)
    satori.load()
    return satori

###   TEST   ###

def test():
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG) # XXX
    if sys.argv[1] == 'ini':
        test_ini(sys.argv[2])
    elif sys.argv[1] == 'parser':
        test_parser(sys.argv[2])
    elif sys.argv[1] == 'interp':
        test_interp(sys.argv[2])

def test_ini(top_dir):
    dic_list = list_dict(top_dir)
    print('number of files =', len(dic_list))
    for dic in dic_list:
        print(dic)

def test_parser(path):
    parser = Parser()
    parser.read(path)
    for name, talk in parser.talk.items():
        for node_list in talk:
            print('＊', name)
            parser.print_nodelist(node_list)
            print()
    for name, word in parser.word.items():
        print('＠', name)
        for node_list in word:
            print('>>>', test_expand(node_list))
            parser.print_nodelist(node_list, 2)
        print()

def test_expand(node_list):
    buf = []
    for node in node_list:
        if node[0] == NODE_TEXT:
            buf.extend(node[1])
        elif node[0] == NODE_REF:
            buf.extend(test_expand(node[1]))
        else:
            raise RuntimeError('should not reach here')
    return ''.join(buf)

def test_interp(top_dir):
    satori = Satori(top_dir)
    satori.load()
    while 1:
        try:
            name = str(raw_input('>>> '), 'utf-8', 'ignore') ## FIXME
        except:
            break
        if name.startswith('＠'):
            print(satori.getstring(name[2:]))
        elif name.startswith('＊'):
            print(satori.get_event_response(name[2:]))
        else:
            print(satori.get_event_response(name))
        print(satori.get_script(name))

if __name__ == '__main__':
    test()
