# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2012 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003-2005 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import os
import sys
import socket
import multiprocessing
import io


class BasePlugin(multiprocessing.Process):

    def __init__(self, port, directory, args, ninix_home, caller,
                 queue, data):
        multiprocessing.Process.__init__(self)
        self.queue = queue
        self.__cache = data
        self.sstp_port = port
        self.directory = directory
        self.args = args
        self.ninix_home = ninix_home
        self.caller = caller
        if os.name == 'nt' and directory is not None:
            sys.path.insert(0, directory) # XXX

    def set_variable(self, name, value):
        self.set_variables({name: value})

    def set_variables(self, v_dict):
        assert isinstance(v_dict, dict)
        for name, value in v_dict.items():
            assert isinstance(name, str) and ':' not in name
            assert isinstance(value, str) or value is None
        self.queue.put(v_dict)
        self.queue.join()
        self.__cache.update(v_dict)

    def get_variable(self, name):
        if name in self.__cache:
            return self.__cache[name]
        else:
            return None

    def _connect(self):
        address = ('localhost', self.sstp_port)
        sstp = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        try:
            sstp.connect(address)
        except socket.error:
            raise SystemExit('cannot connect to the SSTP server.')
        return sstp

    def send_sstp(self, sender, entry=[], script_odict=None):
        if self.sstp_port is None:
            return
        sstp = self._connect()
        assert script_odict
        if any(script_odict.keys()):
            version = 1.4
        elif entry:
            version = 1.2
        else:
            version = 1.1
        message = ''.join(('SEND SSTP/{0}\r\n'.format(version),
                           'Charset: UTF-8\r\n',
                           'Sender: ', sender, '\r\n'))
        for value in entry:
            message = ''.join((message, 'Entry: ', value, '\r\n'))
        if script_odict is not None:
            for if_ghost, script in script_odict.items():
                if if_ghost:
                    message = ''.join((message, 'IfGhost: ', if_ghost, '\r\n'))
                message = ''.join((message, 'Script: ', script, '\r\n'))
        message = ''.join((message, '\r\n'))
        sstp.send(message.encode('utf-8'))
        buffer = []
        while 1:
            buffer.append(sstp.recv(1024))
            if not buffer[-1]:
                break
        sstp.close()
        data = b''.join(buffer)
        if not data:
            return ''
        file = io.BytesIO(data)
        status = file.readline()
        if status[:12] != 'SSTP/{0} 200'.format(version): ## FIXME
            return ''
        for line in file:
            line = line.strip('\r\n\r\n')
            if not line:
                continue
            return str(line, 'utf-8')
        return ''

    def notify_sstp(self, sender, event, ref=[], entry=[], script_odict=None):
        if self.sstp_port is None:
            return
        sstp = self._connect()
        message = ''.join(('NOTIFY SSTP/1.1\r\n',
                           'Charset: UTF-8\r\n',
                           'Sender: ', sender, '\r\n',
                           'Event: ', event, '\r\n'))
        assert len(ref) <= 8
        for i, value in enumerate(ref):
            message = ''.join((message, 'Reference{0}: '.format(i), value, '\r\n'))
        for value in entry:
            message = ''.join((message, 'Entry: ', value, '\r\n'))
        if script_odict is not None:
            for if_ghost, script in script_odict.items():
                if if_ghost:
                    message = ''.join((message, 'IfGhost: ', if_ghost, '\r\n'))
                message = ''.join((message, 'Script: ', script, '\r\n'))
        message = ''.join((message, '\r\n'))
        sstp.send(message.encode('utf-8'))
        buffer = []
        while 1:
            buffer.append(sstp.recv(1024))
            if not buffer[-1]:
                break
        sstp.close()
        data = b''.join(buffer)
        if not data:
            return ''
        file = io.BytesIO(data)
        status = file.readline()
        if status[:12] != 'SSTP/1.1 200': ## FIXME
            return ''
        for line in file:
            line = line.strip('\r\n\r\n')
            if not line:
                continue
            return str(line, 'utf-8')
        return ''

    def send_script(self, sender, script, if_ghost=None):
        self.send_sstp(sender, script_odict={if_ghost: script})
