/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import static org.testng.Assert.*;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import jp.ossc.nimbus.core.Service.State;
import jp.ossc.nimbus.service.log.DefaultLoggerService;
import jp.ossc.nimbus.service.log.Logger;
import jp.ossc.nimbus.service.message.DefaultMessageRecordFactoryService;
import jp.ossc.nimbus.service.message.MessageRecord;
import jp.ossc.nimbus.service.message.MessageRecordFactory;

import org.testng.annotations.Test;
import org.testng.annotations.DataProvider;

/**
 * {@link ServiceBase}eXgP[XB<p>
 * 
 * @author M.Takata
 */
public class ServiceBaseTest{
    
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#ServiceBase()",
                    "Normal"})
    public void testServiceBase(){
        MyService1 service = new MyService1();
        assertNotNull(service.getLogger());
        assertNotNull(service.getMessageRecordFactory());
        assertNull(service.getServiceManager());
        assertNull(service.getServiceManagerName());
        assertNull(service.getServiceName());
        assertNull(service.getServiceNameObject());
        assertEquals(service.getState(), Service.State.DESTROYED);
        assertNull(service.getSystemLoggerServiceName());
        assertNull(service.getSystemMessageRecordFactoryServiceName());
        assertEquals(service.getTarget(), service);
    }
    
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#ServiceBase(jp.ossc.nimbus.core.ServiceBaseSupport)",
                    "Normal"})
    public void testServiceBase2(){
        MyServiceBaseSupport support = new MyServiceBaseSupport();
        MyService2 service = new MyService2(support);
        assertNotNull(service.getLogger());
        assertNotNull(service.getMessageRecordFactory());
        assertNull(service.getServiceManager());
        assertNull(service.getServiceManagerName());
        assertNull(service.getServiceName());
        assertNull(service.getServiceNameObject());
        assertEquals(service.getState(), Service.State.DESTROYED);
        assertNull(service.getSystemLoggerServiceName());
        assertNull(service.getSystemMessageRecordFactoryServiceName());
        assertEquals(service.getTarget(), support);
    }

    @DataProvider(name = "SetServiceManager")
    public Object[][] dataSetServiceManager(){
        DefaultServiceManagerService sm1 = new DefaultServiceManagerService();
        DefaultServiceManagerService sm2 = new DefaultServiceManagerService();
        sm2.setServiceName("manager1");
        return new Object[][]{
            {null},
            {sm1},
            {sm2}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#setServiceManager(jp.ossc.nimbus.core.ServiceManager)",
                    "jp.ossc.nimbus.core.ServiceBase#getServiceManager()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)"},
          dataProvider = "SetServiceManager")
    public void testSetServiceManager(ServiceManager manager){
        MyService1 service = new MyService1();
        service.setServiceManager(manager);
        assertEquals(service.getServiceManager(), manager);
        assertEquals(service.getServiceManagerName(), manager == null ? null : manager.getServiceName());
        assertEquals(((LoggerWrapper)service.getLogger()).getDefaultLogger(), ServiceManagerFactory.getLogger());
        assertEquals(((LoggerWrapper)service.getLogger()).getLogger(), manager == null ? ServiceManagerFactory.getLogger() : manager.getLogger());
        assertEquals(((MessageRecordFactoryWrapper)service.getMessageRecordFactory()).getDefaultMessageRecordFactory(), ServiceManagerFactory.getMessageRecordFactory());
        assertEquals(((MessageRecordFactoryWrapper)service.getMessageRecordFactory()).getMessageRecordFactory(), manager == null ? ServiceManagerFactory.getMessageRecordFactory() : manager.getMessageRecordFactory());
        assertNull(service.getServiceName());
        assertNull(service.getServiceNameObject());
        assertEquals(service.getState(), Service.State.DESTROYED);
        assertNull(service.getSystemLoggerServiceName());
        assertNull(service.getSystemMessageRecordFactoryServiceName());
        assertEquals(service.getTarget(), service);
    }

    @DataProvider(name = "SetServiceName")
    public Object[][] dataSetServiceName(){
        DefaultServiceManagerService sm = new DefaultServiceManagerService();
        sm.setServiceName("manager1");
        return new Object[][]{
            {null, null, null},
            {sm, null, null},
            {sm, null, "Service1"},
            {null, null, "Service1"},
            {null,"manager1", "Service1"},
            {sm,"manager1", "Service1"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#setServiceName(java.lang.String)",
                    "jp.ossc.nimbus.core.ServiceBase#getServiceName()",
                    "jp.ossc.nimbus.core.ServiceBase#setServiceManagerName(java.lang.String)",
                    "jp.ossc.nimbus.core.ServiceBase#getServiceManagerName()",
                    "jp.ossc.nimbus.core.ServiceBase#getServiceNameObject()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#getServiceManager\\(\\)"},
          dataProvider = "SetServiceName")
    public void testSetServiceName(ServiceManager manager, String managerName, String serviceName){
        try{
            if(manager != null){
                ServiceManagerFactory.registerManager(manager.getServiceName(), manager);
            }
            MyService1 service = new MyService1();
            service.setServiceManagerName(managerName);
            service.setServiceName(serviceName);
            assertEquals(service.getServiceManagerName(), managerName);
            assertEquals(service.getServiceName(), serviceName);
            if(serviceName != null && managerName != null){
                assertEquals(service.getServiceNameObject(), new ServiceName(managerName, serviceName));
            }else{
                assertNull(service.getServiceNameObject());
            }
            if(managerName != null && manager != null){
                assertEquals(service.getServiceManager(), manager);
            }else{
                assertNull(service.getServiceManager());
            }
        }finally{
            if(manager != null){
                ServiceManagerFactory.unregisterManager(manager.getServiceName());
            }
        }
    }

    @DataProvider(name = "SetSystemLoggerServiceName")
    public Object[][] dataSetSystemLoggerServiceName(){
        return new Object[][]{
            {null, false, null, null, null},
            {new ServiceName("test", "Logger"), false, null, null, null},
            {new ServiceName("test", "Logger"), false, new DefaultLoggerService(), null, Service.State.CREATED},
            {new ServiceName("test", "Logger"), false, new DefaultLoggerService(), null, Service.State.STARTED},
            {new ServiceName("test", "Logger"), true, new DefaultLoggerService(), Service.State.CREATED, null},
            {new ServiceName("test", "Logger"), true, new DefaultLoggerService(), Service.State.STARTED, null},
            {new ServiceName("test", "Logger"), true, new DefaultLoggerService(), null, Service.State.CREATED},
            {new ServiceName("test", "Logger"), true, new DefaultLoggerService(), null, Service.State.STARTED},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#setSystemLoggerServiceName(jp.ossc.nimbus.core.ServiceName)",
                    "jp.ossc.nimbus.core.ServiceBase#getSystemLoggerServiceName()",
                    "jp.ossc.nimbus.core.ServiceBase#getLogger()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)"},
          dataProvider = "SetSystemLoggerServiceName")
    public void testSetSystemLoggerServiceName(
        ServiceName serviceName,
        boolean isRegisterMananer,
        Service loggerService,
        Service.State preState,
        Service.State postState
    ) throws Exception{
        try{
            if(isRegisterMananer){
                ServiceManagerFactory.registerManager("test");
                if(loggerService != null){
                    ServiceManagerFactory.registerService("test", "Logger", loggerService);
                }
            }
            if(preState != null){
                ServiceManager manager = ServiceManagerFactory.findManager("test");
                switch(preState){
                case CREATED: 
                    manager.createService("Logger");
                    break;
                case STARTED: 
                    manager.createService("Logger");
                    manager.startService("Logger");
                    break;
                }
            }
            MyService1 service = new MyService1();
            service.setSystemLoggerServiceName(serviceName);
            if(preState == Service.State.STARTED){
                assertEquals(service.getSystemLoggerServiceName(), serviceName);
                assertEquals(((LoggerWrapper)service.getLogger()).getLogger(), loggerService);
            }else{
                assertNull(service.getSystemLoggerServiceName());
                assertEquals(
                    ((LoggerWrapper)service.getLogger()).getLogger(),
                    ServiceManagerFactory.getLogger()
                );
            }
            if(postState != null){
                if(!isRegisterMananer && loggerService != null){
                    ServiceManagerFactory.registerManager("test");
                    ServiceManagerFactory.registerService("test", "Logger", loggerService);
                }
                ServiceManager manager = ServiceManagerFactory.findManager("test");
                switch(postState){
                case CREATED: 
                    manager.createService("Logger");
                    break;
                case STARTED: 
                    manager.createService("Logger");
                    manager.startService("Logger");
                    break;
                }
            }
            if(postState == Service.State.STARTED){
                assertEquals(service.getSystemLoggerServiceName(), serviceName);
                assertEquals(((LoggerWrapper)service.getLogger()).getLogger(), loggerService);
            }else if(postState != null){
                assertNull(service.getSystemLoggerServiceName());
                assertEquals(
                    ((LoggerWrapper)service.getLogger()).getLogger(),
                    ServiceManagerFactory.getLogger()
                );
            }
        }finally{
            if(loggerService != null){
                loggerService.stop();
            }
            ServiceManagerFactory.unregisterService("test", "Logger");
            ServiceManagerFactory.unregisterManager("test");
        }
    }

    @DataProvider(name = "SetLogger")
    public Object[][] dataSetLogger(){
        return new Object[][]{
            {null, false, null, null, null},
            {new ServiceName("test", "Logger"), false, null, null, null},
            {new ServiceName("test", "Logger"), false, new DefaultLoggerService(), null, Service.State.CREATED},
            {new ServiceName("test", "Logger"), false, new DefaultLoggerService(), null, Service.State.STARTED},
            {new ServiceName("test", "Logger"), true, new DefaultLoggerService(), Service.State.CREATED, null},
            {new ServiceName("test", "Logger"), true, new DefaultLoggerService(), Service.State.STARTED, null},
            {new ServiceName("test", "Logger"), true, new DefaultLoggerService(), null, Service.State.CREATED},
            {new ServiceName("test", "Logger"), true, new DefaultLoggerService(), null, Service.State.STARTED},
            {new ServiceName("test", "Logger"), false, new MyLogger(), null, Service.State.CREATED},
            {new ServiceName("test", "Logger"), false, new MyLogger(), null, Service.State.STARTED},
            {new ServiceName("test", "Logger"), true, new MyLogger(), Service.State.CREATED, null},
            {new ServiceName("test", "Logger"), true, new MyLogger(), Service.State.STARTED, null},
            {new ServiceName("test", "Logger"), true, new MyLogger(), null, Service.State.CREATED},
            {new ServiceName("test", "Logger"), true, new MyLogger(), null, Service.State.STARTED},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#setLogger(jp.ossc.nimbus.service.log.Logger)",
                    "jp.ossc.nimbus.core.ServiceBase#getLogger()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#setSystemLoggerServiceName\\(jp\\.ossc\\.nimbus\\.core\\.ServiceName\\)"},
          dataProvider = "SetLogger")
    public void testSetLogger(
        ServiceName serviceName,
        boolean isRegisterMananer,
        Logger logger,
        Service.State preState,
        Service.State postState
    ) throws Exception{
        try{
            if(serviceName != null && logger != null && logger instanceof Service){
                ((Service)logger).setServiceManagerName(serviceName.getServiceManagerName());
                ((Service)logger).setServiceName(serviceName.getServiceName());
            }
            if(isRegisterMananer){
                ServiceManagerFactory.registerManager("test");
                if(logger != null){
                    ServiceManagerFactory.registerService("test", "Logger", logger);
                }
            }
            if(preState != null){
                ServiceManager manager = ServiceManagerFactory.findManager("test");
                switch(preState){
                case CREATED: 
                    manager.createService("Logger");
                    break;
                case STARTED: 
                    manager.createService("Logger");
                    manager.startService("Logger");
                    break;
                }
            }
            MyService1 service = new MyService1();
            service.setLogger(logger);
            if(preState == Service.State.STARTED || !(logger instanceof Service)){
                if(logger instanceof Service){
                    assertEquals(service.getSystemLoggerServiceName(), serviceName);
                }else{
                    assertNull(service.getSystemLoggerServiceName());
                }
                assertEquals(((LoggerWrapper)service.getLogger()).getLogger(), logger == null ? ServiceManagerFactory.getLogger() : logger);
            }else{
                assertNull(service.getSystemLoggerServiceName());
                assertEquals(
                    ((LoggerWrapper)service.getLogger()).getLogger(),
                    ServiceManagerFactory.getLogger()
                );
            }
            if(postState != null){
                if(!isRegisterMananer && logger != null){
                    ServiceManagerFactory.registerManager("test");
                    ServiceManagerFactory.registerService("test", "Logger", logger);
                }
                ServiceManager manager = ServiceManagerFactory.findManager("test");
                switch(postState){
                case CREATED: 
                    manager.createService("Logger");
                    break;
                case STARTED: 
                    manager.createService("Logger");
                    manager.startService("Logger");
                    break;
                }
            }
            if(postState == Service.State.STARTED || !(logger instanceof Service)){
                if(logger instanceof Service){
                    assertEquals(service.getSystemLoggerServiceName(), serviceName);
                }else{
                    assertNull(service.getSystemLoggerServiceName());
                }
                assertEquals(((LoggerWrapper)service.getLogger()).getLogger(), logger == null ? ServiceManagerFactory.getLogger() : logger);
            }else if(postState != null){
                assertNull(service.getSystemLoggerServiceName());
                assertEquals(
                    ((LoggerWrapper)service.getLogger()).getLogger(),
                    ServiceManagerFactory.getLogger()
                );
            }
        }finally{
            if(logger != null && logger instanceof Service){
                ((Service)logger).stop();
            }
            ServiceManagerFactory.unregisterService("test", "Logger");
            ServiceManagerFactory.unregisterManager("test");
        }
    }

    @DataProvider(name = "SetSystemMessageRecordFactoryServiceName")
    public Object[][] dataSetSystemMessageRecordFactoryServiceName(){
        return new Object[][]{
            {null, false, null, null, null},
            {new ServiceName("test", "Message"), false, null, null, null},
            {new ServiceName("test", "Message"), false, new DefaultMessageRecordFactoryService(), null, Service.State.CREATED},
            {new ServiceName("test", "Message"), false, new DefaultMessageRecordFactoryService(), null, Service.State.STARTED},
            {new ServiceName("test", "Message"), true, new DefaultMessageRecordFactoryService(), Service.State.CREATED, null},
            {new ServiceName("test", "Message"), true, new DefaultMessageRecordFactoryService(), Service.State.STARTED, null},
            {new ServiceName("test", "Message"), true, new DefaultMessageRecordFactoryService(), null, Service.State.CREATED},
            {new ServiceName("test", "Message"), true, new DefaultMessageRecordFactoryService(), null, Service.State.STARTED},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#setSystemMessageRecordFactoryServiceName(jp.ossc.nimbus.core.ServiceName)",
                    "jp.ossc.nimbus.core.ServiceBase#getSystemMessageRecordFactoryServiceName()",
                    "jp.ossc.nimbus.core.ServiceBase#getMessageRecordFactory()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)"},
          dataProvider = "SetSystemMessageRecordFactoryServiceName")
    public void testSetSystemMessageRecordFactoryServiceName(
        ServiceName serviceName,
        boolean isRegisterMananer,
        Service messageService,
        Service.State preState,
        Service.State postState
    ) throws Exception{
        try{
            if(isRegisterMananer){
                ServiceManagerFactory.registerManager("test");
                if(messageService != null){
                    ServiceManagerFactory.registerService("test", "Message", messageService);
                }
            }
            if(preState != null){
                ServiceManager manager = ServiceManagerFactory.findManager("test");
                switch(preState){
                case CREATED: 
                    manager.createService("Message");
                    break;
                case STARTED: 
                    manager.createService("Message");
                    manager.startService("Message");
                    break;
                }
            }
            MyService1 service = new MyService1();
            service.setSystemMessageRecordFactoryServiceName(serviceName);
            if(preState == Service.State.STARTED){
                assertEquals(service.getSystemMessageRecordFactoryServiceName(), serviceName);
                assertEquals(((MessageRecordFactoryWrapper)service.getMessageRecordFactory()).getMessageRecordFactory(), messageService);
            }else{
                assertNull(service.getSystemMessageRecordFactoryServiceName());
                assertEquals(
                    ((MessageRecordFactoryWrapper)service.getMessageRecordFactory()).getMessageRecordFactory(),
                    ServiceManagerFactory.getMessageRecordFactory()
                );
            }
            if(postState != null){
                if(!isRegisterMananer && messageService != null){
                    ServiceManagerFactory.registerManager("test");
                    ServiceManagerFactory.registerService("test", "Message", messageService);
                }
                ServiceManager manager = ServiceManagerFactory.findManager("test");
                switch(postState){
                case CREATED: 
                    manager.createService("Message");
                    break;
                case STARTED: 
                    manager.createService("Message");
                    manager.startService("Message");
                    break;
                }
            }
            if(postState == Service.State.STARTED){
                assertEquals(service.getSystemMessageRecordFactoryServiceName(), serviceName);
                assertEquals(((MessageRecordFactoryWrapper)service.getMessageRecordFactory()).getMessageRecordFactory(), messageService);
            }else if(postState != null){
                assertNull(service.getSystemMessageRecordFactoryServiceName());
                assertEquals(
                    ((MessageRecordFactoryWrapper)service.getMessageRecordFactory()).getMessageRecordFactory(),
                    ServiceManagerFactory.getMessageRecordFactory()
                );
            }
        }finally{
            if(messageService != null){
                messageService.stop();
            }
            ServiceManagerFactory.unregisterService("test", "Message");
            ServiceManagerFactory.unregisterManager("test");
        }
    }

    @DataProvider(name = "SetMessageRecordFactory")
    public Object[][] dataSetMessageRecordFactory(){
        return new Object[][]{
            {null, false, null, null, null},
            {new ServiceName("test", "Message"), false, null, null, null},
            {new ServiceName("test", "Message"), false, new DefaultMessageRecordFactoryService(), null, Service.State.CREATED},
            {new ServiceName("test", "Message"), false, new DefaultMessageRecordFactoryService(), null, Service.State.STARTED},
            {new ServiceName("test", "Message"), true, new DefaultMessageRecordFactoryService(), Service.State.CREATED, null},
            {new ServiceName("test", "Message"), true, new DefaultMessageRecordFactoryService(), Service.State.STARTED, null},
            {new ServiceName("test", "Message"), true, new DefaultMessageRecordFactoryService(), null, Service.State.CREATED},
            {new ServiceName("test", "Message"), true, new DefaultMessageRecordFactoryService(), null, Service.State.STARTED},
            {new ServiceName("test", "Message"), false, new MyMessageRecordFactory(), null, Service.State.CREATED},
            {new ServiceName("test", "Message"), false, new MyMessageRecordFactory(), null, Service.State.STARTED},
            {new ServiceName("test", "Message"), true, new MyMessageRecordFactory(), Service.State.CREATED, null},
            {new ServiceName("test", "Message"), true, new MyMessageRecordFactory(), Service.State.STARTED, null},
            {new ServiceName("test", "Message"), true, new MyMessageRecordFactory(), null, Service.State.CREATED},
            {new ServiceName("test", "Message"), true, new MyMessageRecordFactory(), null, Service.State.STARTED},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#setMessageRecordFactory(jp.ossc.nimbus.service.message.MessageRecordFactory)",
                    "jp.ossc.nimbus.core.ServiceBase#getMessageRecordFactory()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#setSystemMessageRecordFactoryServiceName\\(jp\\.ossc\\.nimbus\\.core\\.ServiceName\\)"},
          dataProvider = "SetMessageRecordFactory")
    public void testSetMessageRecordFactory(
        ServiceName serviceName,
        boolean isRegisterMananer,
        MessageRecordFactory message,
        Service.State preState,
        Service.State postState
    ) throws Exception{
        try{
            if(serviceName != null && message != null && message instanceof Service){
                ((Service)message).setServiceManagerName(serviceName.getServiceManagerName());
                ((Service)message).setServiceName(serviceName.getServiceName());
            }
            if(isRegisterMananer){
                ServiceManagerFactory.registerManager("test");
                if(message != null){
                    ServiceManagerFactory.registerService("test", "Message", message);
                }
            }
            if(preState != null){
                ServiceManager manager = ServiceManagerFactory.findManager("test");
                switch(preState){
                case CREATED: 
                    manager.createService("Message");
                    break;
                case STARTED: 
                    manager.createService("Message");
                    manager.startService("Message");
                    break;
                }
            }
            MyService1 service = new MyService1();
            service.setMessageRecordFactory(message);
            if(preState == Service.State.STARTED || !(message instanceof Service)){
                if(message instanceof Service){
                    assertEquals(service.getSystemMessageRecordFactoryServiceName(), serviceName);
                }else{
                    assertNull(service.getSystemMessageRecordFactoryServiceName());
                }
                assertEquals(((MessageRecordFactoryWrapper)service.getMessageRecordFactory()).getMessageRecordFactory(), message == null ? ServiceManagerFactory.getMessageRecordFactory() : message);
            }else{
                assertNull(service.getSystemMessageRecordFactoryServiceName());
                assertEquals(
                    ((MessageRecordFactoryWrapper)service.getMessageRecordFactory()).getMessageRecordFactory(),
                    ServiceManagerFactory.getMessageRecordFactory()
                );
            }
            if(postState != null){
                if(!isRegisterMananer && message != null){
                    ServiceManagerFactory.registerManager("test");
                    ServiceManagerFactory.registerService("test", "Message", message);
                }
                ServiceManager manager = ServiceManagerFactory.findManager("test");
                switch(postState){
                case CREATED: 
                    manager.createService("Message");
                    break;
                case STARTED: 
                    manager.createService("Message");
                    manager.startService("Message");
                    break;
                }
            }
            if(postState == Service.State.STARTED || !(message instanceof Service)){
                if(message instanceof Service){
                    assertEquals(service.getSystemMessageRecordFactoryServiceName(), serviceName);
                }else{
                    assertNull(service.getSystemMessageRecordFactoryServiceName());
                }
                assertEquals(((MessageRecordFactoryWrapper)service.getMessageRecordFactory()).getMessageRecordFactory(), message == null ? ServiceManagerFactory.getMessageRecordFactory() : message);
            }else if(postState != null){
                assertNull(service.getSystemMessageRecordFactoryServiceName());
                assertEquals(
                    ((MessageRecordFactoryWrapper)service.getMessageRecordFactory()).getMessageRecordFactory(),
                    ServiceManagerFactory.getMessageRecordFactory()
                );
            }
        }finally{
            if(message != null && message instanceof Service){
                ((Service)message).stop();
            }
            ServiceManagerFactory.unregisterService("test", "Message");
            ServiceManagerFactory.unregisterManager("test");
        }
    }

    @DataProvider(name = "Create")
    public Object[][] dataCreate(){
        return new Object[][]{
            {false, false, false},
            {true, false, false},
            {true, false, true},
            {true, true, false},
            {true, true, true},
            {false, true, false},
            {false, true, true},
            {false, false, true},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#create()",
                    "jp.ossc.nimbus.core.ServiceBase#createService()",
                    "jp.ossc.nimbus.core.ServiceBase#preCreateService()",
                    "jp.ossc.nimbus.core.ServiceBase#postCreateService()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)"},
          dataProvider = "Create")
    public void testCreate(boolean existManager, boolean existService, boolean registerService) throws Exception{
        try{
            if(existManager){
                ServiceManagerFactory.registerManager("test");
            }
            MyService1 oldService = null;
            if(existService){
                oldService = new MyService1();
                ServiceManagerFactory.registerService("test", "Service1", oldService);
            }
            MyService1 service = new MyService1();
            if(registerService){
                service.setServiceManagerName("test");
            }
            service.setServiceName("Service1");
            service.create();
            assertTrue(service.isCallPreCreateService);
            assertTrue(service.isCallCreateService);
            assertTrue(service.isCallPostCreateService);
            if(existManager){
                if(registerService){
                    assertTrue(ServiceManagerFactory.isRegisteredService("test", "Service1"));
                    assertEquals(ServiceManagerFactory.getService("test", "Service1"), service);
                }else if(!existService){
                    assertFalse(ServiceManagerFactory.isRegisteredService("test", "Service1"));
                }else{
                    assertTrue(ServiceManagerFactory.isRegisteredService("test", "Service1"));
                    assertEquals(ServiceManagerFactory.getService("test", "Service1"), oldService);
                }
            }else{
                assertFalse(ServiceManagerFactory.isRegisteredService("test", "Service1"));
            }
            assertFalse(service.isCallPreStartService);
            assertFalse(service.isCallStartService);
            assertFalse(service.isCallPostStartService);
            assertFalse(service.isCallPreStopService);
            assertFalse(service.isCallStopService);
            assertFalse(service.isCallPostStopService);
            assertFalse(service.isCallPreDestroyService);
            assertFalse(service.isCallDestroyService);
            assertFalse(service.isCallPostDestroyService);
            assertEquals(service.getState(), Service.State.CREATED);
        }finally{
            ServiceManagerFactory.unregisterManager("test");
        }
    }

    @DataProvider(name = "CreateError")
    public Object[][] dataCreateError(){
        return new Object[][]{
            {0, new Exception("test")},
            {1, new Exception("test")},
            {2, new Exception("test")},
            {0, new Error("test")},
            {1, new Error("test")},
            {2, new Error("test")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#create()",
                    "jp.ossc.nimbus.core.ServiceBase#createService()",
                    "jp.ossc.nimbus.core.ServiceBase#preCreateService()",
                    "jp.ossc.nimbus.core.ServiceBase#postCreateService()",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)"},
          expectedExceptions = {Exception.class, Error.class},
          dataProvider = "CreateError")
    public void testCreateError(int throwPoint, Throwable th) throws Exception{
        MyService1 service = new MyService1();
        service.throwable = th;
        switch(throwPoint){
        case 0:
            service.isThrowPreCreateService = true;
            break;
        case 1:
            service.isThrowCreateService = true;
            break;
        case 2:
            service.isThrowPostCreateService = true;
            break;
        }
        try{
            service.create();
        }catch(Exception e){
            switch(throwPoint){
            case 0:
                assertTrue(service.isCallPreCreateService);
                assertFalse(service.isCallCreateService);
                assertFalse(service.isCallPostCreateService);
                break;
            case 1:
                assertTrue(service.isCallPreCreateService);
                assertTrue(service.isCallCreateService);
                assertFalse(service.isCallPostCreateService);
                break;
            case 2:
                assertTrue(service.isCallPreCreateService);
                assertTrue(service.isCallCreateService);
                assertTrue(service.isCallPostCreateService);
                break;
            }
            assertFalse(service.isCallPreStartService);
            assertFalse(service.isCallStartService);
            assertFalse(service.isCallPostStartService);
            assertFalse(service.isCallPreStopService);
            assertFalse(service.isCallStopService);
            assertFalse(service.isCallPostStopService);
            assertFalse(service.isCallPreDestroyService);
            assertFalse(service.isCallDestroyService);
            assertFalse(service.isCallPostDestroyService);
            assertEquals(service.getState(), Service.State.FAILED);
            throw e;
        }
    }

    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#start()",
                    "jp.ossc.nimbus.core.ServiceBase#startService()",
                    "jp.ossc.nimbus.core.ServiceBase#preStartService()",
                    "jp.ossc.nimbus.core.ServiceBase#postStartService()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#create\\(\\)"})
    public void testStart() throws Exception{
        MyService1 service = new MyService1();
        service.create();
        service.start();
        assertTrue(service.isCallPreCreateService);
        assertTrue(service.isCallCreateService);
        assertTrue(service.isCallPostCreateService);
        assertTrue(service.isCallPreStartService);
        assertTrue(service.isCallStartService);
        assertTrue(service.isCallPostStartService);
        assertFalse(service.isCallPreStopService);
        assertFalse(service.isCallStopService);
        assertFalse(service.isCallPostStopService);
        assertFalse(service.isCallPreDestroyService);
        assertFalse(service.isCallDestroyService);
        assertFalse(service.isCallPostDestroyService);
        assertEquals(service.getState(), Service.State.STARTED);
    }

    @DataProvider(name = "StartError")
    public Object[][] dataStartError(){
        return new Object[][]{
            {0, new Exception("test")},
            {1, new Exception("test")},
            {2, new Exception("test")},
            {0, new Error("test")},
            {1, new Error("test")},
            {2, new Error("test")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#start()",
                    "jp.ossc.nimbus.core.ServiceBase#startService()",
                    "jp.ossc.nimbus.core.ServiceBase#preStartService()",
                    "jp.ossc.nimbus.core.ServiceBase#postStartService()",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#create\\(\\)"},
          expectedExceptions = {Exception.class, Error.class},
          dataProvider = "StartError")
    public void testStartError(int throwPoint, Throwable th) throws Exception{
        MyService1 service = new MyService1();
        service.throwable = th;
        switch(throwPoint){
        case 0:
            service.isThrowPreStartService = true;
            break;
        case 1:
            service.isThrowStartService = true;
            break;
        case 2:
            service.isThrowPostStartService = true;
            break;
        }
        try{
            service.create();
            service.start();
        }catch(Exception e){
            switch(throwPoint){
            case 0:
                assertTrue(service.isCallPreCreateService);
                assertTrue(service.isCallCreateService);
                assertTrue(service.isCallPostCreateService);
                assertTrue(service.isCallPreStartService);
                assertFalse(service.isCallStartService);
                assertFalse(service.isCallPostStartService);
                break;
            case 1:
                assertTrue(service.isCallPreCreateService);
                assertTrue(service.isCallCreateService);
                assertTrue(service.isCallPostCreateService);
                assertTrue(service.isCallPreStartService);
                assertTrue(service.isCallStartService);
                assertFalse(service.isCallPostStartService);
                break;
            case 2:
                assertTrue(service.isCallPreCreateService);
                assertTrue(service.isCallCreateService);
                assertTrue(service.isCallPostCreateService);
                assertTrue(service.isCallPreStartService);
                assertTrue(service.isCallStartService);
                assertTrue(service.isCallPostStartService);
                break;
            }
            assertFalse(service.isCallPreStopService);
            assertFalse(service.isCallStopService);
            assertFalse(service.isCallPostStopService);
            assertFalse(service.isCallPreDestroyService);
            assertFalse(service.isCallDestroyService);
            assertFalse(service.isCallPostDestroyService);
            assertEquals(service.getState(), Service.State.FAILED);
            throw e;
        }
    }

    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#stop()",
                    "jp.ossc.nimbus.core.ServiceBase#stopService()",
                    "jp.ossc.nimbus.core.ServiceBase#preStopService()",
                    "jp.ossc.nimbus.core.ServiceBase#postStopService()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#create\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#start\\(\\)"})
    public void testStop() throws Exception{
        MyService1 service = new MyService1();
        service.create();
        service.start();
        service.stop();
        assertTrue(service.isCallPreCreateService);
        assertTrue(service.isCallCreateService);
        assertTrue(service.isCallPostCreateService);
        assertTrue(service.isCallPreStartService);
        assertTrue(service.isCallStartService);
        assertTrue(service.isCallPostStartService);
        assertTrue(service.isCallPreStopService);
        assertTrue(service.isCallStopService);
        assertTrue(service.isCallPostStopService);
        assertFalse(service.isCallPreDestroyService);
        assertFalse(service.isCallDestroyService);
        assertFalse(service.isCallPostDestroyService);
        assertEquals(service.getState(), Service.State.STOPPED);
    }

    @DataProvider(name = "StopError")
    public Object[][] dataStopError(){
        return new Object[][]{
            {0, new Exception("test")},
            {1, new Exception("test")},
            {2, new Exception("test")},
            {0, new Error("test")},
            {1, new Error("test")},
            {2, new Error("test")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#stop()",
                    "jp.ossc.nimbus.core.ServiceBase#stopService()",
                    "jp.ossc.nimbus.core.ServiceBase#preStopService()",
                    "jp.ossc.nimbus.core.ServiceBase#postStopService()",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#create\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#start\\(\\)"},
          dataProvider = "StopError")
    public void testStopError(int throwPoint, Throwable th) throws Exception{
        MyService1 service = new MyService1();
        service.throwable = th;
        switch(throwPoint){
        case 0:
            service.isThrowPreStopService = true;
            break;
        case 1:
            service.isThrowStopService = true;
            break;
        case 2:
            service.isThrowPostStopService = true;
            break;
        }
        service.create();
        service.start();
        service.stop();
        switch(throwPoint){
        case 0:
            assertTrue(service.isCallPreCreateService);
            assertTrue(service.isCallCreateService);
            assertTrue(service.isCallPostCreateService);
            assertTrue(service.isCallPreStartService);
            assertTrue(service.isCallStartService);
            assertTrue(service.isCallPostStartService);
            assertTrue(service.isCallPreStopService);
            assertFalse(service.isCallStopService);
            assertFalse(service.isCallPostStopService);
            break;
        case 1:
            assertTrue(service.isCallPreCreateService);
            assertTrue(service.isCallCreateService);
            assertTrue(service.isCallPostCreateService);
            assertTrue(service.isCallPreStartService);
            assertTrue(service.isCallStartService);
            assertTrue(service.isCallPostStartService);
            assertTrue(service.isCallPreStopService);
            assertTrue(service.isCallStopService);
            assertFalse(service.isCallPostStopService);
            break;
        case 2:
            assertTrue(service.isCallPreCreateService);
            assertTrue(service.isCallCreateService);
            assertTrue(service.isCallPostCreateService);
            assertTrue(service.isCallPreStartService);
            assertTrue(service.isCallStartService);
            assertTrue(service.isCallPostStartService);
            assertTrue(service.isCallPreStopService);
            assertTrue(service.isCallStopService);
            assertTrue(service.isCallPostStopService);
            break;
        }
        assertFalse(service.isCallPreDestroyService);
        assertFalse(service.isCallDestroyService);
        assertFalse(service.isCallPostDestroyService);
        assertEquals(service.getState(), Service.State.FAILED);
    }

    @DataProvider(name = "Destroy")
    public Object[][] dataDestroy(){
        return new Object[][]{
            {false, false},
            {false, false},
            {true, false},
            {true, true},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#destroy()",
                    "jp.ossc.nimbus.core.ServiceBase#destroyService()",
                    "jp.ossc.nimbus.core.ServiceBase#preDestroyService()",
                    "jp.ossc.nimbus.core.ServiceBase#postDestroyService()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#create\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#start\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#stop\\(\\)"},
          dataProvider = "Destroy")
    public void testDestroy(boolean existManager, boolean registerService) throws Exception{
        try{
            if(existManager){
                ServiceManagerFactory.registerManager("test");
            }
            MyService1 service = new MyService1();
            if(registerService){
                service.setServiceManagerName("test");
            }
            service.setServiceName("Service1");
            service.create();
            service.start();
            service.stop();
            service.destroy();
            assertTrue(service.isCallPreCreateService);
            assertTrue(service.isCallCreateService);
            assertTrue(service.isCallPostCreateService);
            assertTrue(service.isCallPreStartService);
            assertTrue(service.isCallStartService);
            assertTrue(service.isCallPostStartService);
            assertTrue(service.isCallPreStopService);
            assertTrue(service.isCallStopService);
            assertTrue(service.isCallPostStopService);
            assertTrue(service.isCallPreDestroyService);
            assertTrue(service.isCallDestroyService);
            assertTrue(service.isCallPostDestroyService);
            assertEquals(service.getState(), Service.State.DESTROYED);
            assertFalse(ServiceManagerFactory.isRegisteredService("test", "Service1"));
        }finally{
            ServiceManagerFactory.unregisterManager("test");
        }
    }

    @DataProvider(name = "DestroyError")
    public Object[][] dataDestroyError(){
        return new Object[][]{
            {0, new Exception("test")},
            {1, new Exception("test")},
            {2, new Exception("test")},
            {0, new Error("test")},
            {1, new Error("test")},
            {2, new Error("test")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#destroy()",
                    "jp.ossc.nimbus.core.ServiceBase#destroyService()",
                    "jp.ossc.nimbus.core.ServiceBase#preDestroyService()",
                    "jp.ossc.nimbus.core.ServiceBase#postDestroyService()",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#create\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#start\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#stop\\(\\)"},
          dataProvider = "DestroyError")
    public void testDestroyError(int throwPoint, Throwable th) throws Exception{
        MyService1 service = new MyService1();
        service.throwable = th;
        switch(throwPoint){
        case 0:
            service.isThrowPreDestroyService = true;
            break;
        case 1:
            service.isThrowDestroyService = true;
            break;
        case 2:
            service.isThrowPostDestroyService = true;
            break;
        }
        service.create();
        service.start();
        service.stop();
        service.destroy();
        switch(throwPoint){
        case 0:
            assertTrue(service.isCallPreCreateService);
            assertTrue(service.isCallCreateService);
            assertTrue(service.isCallPostCreateService);
            assertTrue(service.isCallPreStartService);
            assertTrue(service.isCallStartService);
            assertTrue(service.isCallPostStartService);
            assertTrue(service.isCallPreStopService);
            assertTrue(service.isCallStopService);
            assertTrue(service.isCallPostStopService);
            assertTrue(service.isCallPreDestroyService);
            assertFalse(service.isCallDestroyService);
            assertFalse(service.isCallPostDestroyService);
            break;
        case 1:
            assertTrue(service.isCallPreCreateService);
            assertTrue(service.isCallCreateService);
            assertTrue(service.isCallPostCreateService);
            assertTrue(service.isCallPreStartService);
            assertTrue(service.isCallStartService);
            assertTrue(service.isCallPostStartService);
            assertTrue(service.isCallPreStopService);
            assertTrue(service.isCallStopService);
            assertTrue(service.isCallPostStopService);
            assertTrue(service.isCallPreDestroyService);
            assertTrue(service.isCallDestroyService);
            assertFalse(service.isCallPostDestroyService);
            break;
        case 2:
            assertTrue(service.isCallPreCreateService);
            assertTrue(service.isCallCreateService);
            assertTrue(service.isCallPostCreateService);
            assertTrue(service.isCallPreStartService);
            assertTrue(service.isCallStartService);
            assertTrue(service.isCallPostStartService);
            assertTrue(service.isCallPreStopService);
            assertTrue(service.isCallStopService);
            assertTrue(service.isCallPostStopService);
            assertTrue(service.isCallPreDestroyService);
            assertTrue(service.isCallDestroyService);
            assertTrue(service.isCallPostDestroyService);
            break;
        }
        assertEquals(service.getState(), Service.State.FAILED);
    }

    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#create()",
                    "jp.ossc.nimbus.core.ServiceBase#createService()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(jp\\.ossc\\.nimbus\\.core\\.ServiceBaseSupport\\)"})
    public void testCreate2() throws Exception{
        MyServiceBaseSupport support = new MyServiceBaseSupport();
        MyService2 service = new MyService2(support);
        assertTrue(support.isCallSetServiceBase);
        service.create();
        assertTrue(support.isCallCreateService);
        assertFalse(support.isCallStartService);
        assertFalse(support.isCallStopService);
        assertFalse(support.isCallDestroyService);
        assertEquals(service.getState(), Service.State.CREATED);
    }

    @DataProvider(name = "Create2Error")
    public Object[][] dataCreate2Error(){
        return new Object[][]{
            {new Exception("test")},
            {new Error("test")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#create()",
                    "jp.ossc.nimbus.core.ServiceBase#createService()",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(jp\\.ossc\\.nimbus\\.core\\.ServiceBaseSupport\\)"},
          expectedExceptions = {Exception.class, Error.class},
          dataProvider = "Create2Error")
    public void testCreate2Error(Throwable th) throws Exception{
        MyServiceBaseSupport support = new MyServiceBaseSupport();
        MyService2 service = new MyService2(support);
        assertTrue(support.isCallSetServiceBase);
        support.throwable = th;
        support.isThrowCreateService = true;
        try{
            service.create();
        }catch(Exception e){
            assertTrue(support.isCallCreateService);
            assertFalse(support.isCallStartService);
            assertFalse(support.isCallStopService);
            assertFalse(support.isCallDestroyService);
            assertEquals(service.getState(), Service.State.FAILED);
            throw e;
        }
    }

    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#start()",
                    "jp.ossc.nimbus.core.ServiceBase#startService()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(jp\\.ossc\\.nimbus\\.core\\.ServiceBaseSupport\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#create\\(\\)"})
    public void testStart2() throws Exception{
        MyServiceBaseSupport support = new MyServiceBaseSupport();
        MyService2 service = new MyService2(support);
        assertTrue(support.isCallSetServiceBase);
        service.create();
        service.start();
        assertTrue(support.isCallCreateService);
        assertTrue(support.isCallStartService);
        assertFalse(support.isCallStopService);
        assertFalse(support.isCallDestroyService);
        assertEquals(service.getState(), Service.State.STARTED);
    }

    @DataProvider(name = "Start2Error")
    public Object[][] dataStart2Error(){
        return new Object[][]{
            {new Exception("test")},
            {new Error("test")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#start()",
                    "jp.ossc.nimbus.core.ServiceBase#startService()",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(jp\\.ossc\\.nimbus\\.core\\.ServiceBaseSupport\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#create\\(\\)"},
          expectedExceptions = {Exception.class, Error.class},
          dataProvider = "Start2Error")
    public void testStart2Error(Throwable th) throws Exception{
        MyServiceBaseSupport support = new MyServiceBaseSupport();
        MyService2 service = new MyService2(support);
        assertTrue(support.isCallSetServiceBase);
        support.throwable = th;
        support.isThrowStartService = true;
        try{
            service.create();
            service.start();
        }catch(Exception e){
            assertTrue(support.isCallCreateService);
            assertTrue(support.isCallStartService);
            assertFalse(support.isCallStopService);
            assertFalse(support.isCallDestroyService);
            assertEquals(service.getState(), Service.State.FAILED);
            throw e;
        }
    }

    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#stop()",
                    "jp.ossc.nimbus.core.ServiceBase#stopService()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(jp\\.ossc\\.nimbus\\.core\\.ServiceBaseSupport\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#create\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#start\\(\\)"})
    public void testStop2() throws Exception{
        MyServiceBaseSupport support = new MyServiceBaseSupport();
        MyService2 service = new MyService2(support);
        assertTrue(support.isCallSetServiceBase);
        service.create();
        service.start();
        service.stop();
        assertTrue(support.isCallCreateService);
        assertTrue(support.isCallStartService);
        assertTrue(support.isCallStopService);
        assertFalse(support.isCallDestroyService);
        assertEquals(service.getState(), Service.State.STOPPED);
    }

    @DataProvider(name = "Stop2Error")
    public Object[][] dataStop2Error(){
        return new Object[][]{
            {new Exception("test")},
            {new Error("test")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#stop()",
                    "jp.ossc.nimbus.core.ServiceBase#stopService()",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(jp\\.ossc\\.nimbus\\.core\\.ServiceBaseSupport\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#create\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#start\\(\\)"},
          dataProvider = "Stop2Error")
    public void testStop2Error(Throwable th) throws Exception{
        MyServiceBaseSupport support = new MyServiceBaseSupport();
        MyService2 service = new MyService2(support);
        assertTrue(support.isCallSetServiceBase);
        support.throwable = th;
        support.isThrowStopService = true;
        service.create();
        service.start();
        service.stop();
        assertTrue(support.isCallCreateService);
        assertTrue(support.isCallStartService);
        assertTrue(support.isCallStopService);
        assertFalse(support.isCallDestroyService);
        assertEquals(service.getState(), Service.State.FAILED);
    }

    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#destroy()",
                    "jp.ossc.nimbus.core.ServiceBase#destroyService()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(jp\\.ossc\\.nimbus\\.core\\.ServiceBaseSupport\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#create\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#start\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#stop\\(\\)"})
    public void testDestroy2() throws Exception{
        MyServiceBaseSupport support = new MyServiceBaseSupport();
        MyService2 service = new MyService2(support);
        assertTrue(support.isCallSetServiceBase);
        service.create();
        service.start();
        service.stop();
        service.destroy();
        assertTrue(support.isCallCreateService);
        assertTrue(support.isCallStartService);
        assertTrue(support.isCallStopService);
        assertTrue(support.isCallDestroyService);
        assertEquals(service.getState(), Service.State.DESTROYED);
    }

    @DataProvider(name = "Destroy2Error")
    public Object[][] dataDestroy2Error(){
        return new Object[][]{
            {new Exception("test")},
            {new Error("test")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#destroy()",
                    "jp.ossc.nimbus.core.ServiceBase#destroyService()",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(jp\\.ossc\\.nimbus\\.core\\.ServiceBaseSupport\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#create\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#start\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#stop\\(\\)"},
          dataProvider = "Destroy2Error")
    public void testDestroy2Error(Throwable th) throws Exception{
        MyServiceBaseSupport support = new MyServiceBaseSupport();
        MyService2 service = new MyService2(support);
        assertTrue(support.isCallSetServiceBase);
        support.throwable = th;
        support.isThrowDestroyService = true;
        service.create();
        service.start();
        service.stop();
        service.destroy();
        assertTrue(support.isCallCreateService);
        assertTrue(support.isCallStartService);
        assertTrue(support.isCallStopService);
        assertTrue(support.isCallDestroyService);
        assertEquals(service.getState(), Service.State.FAILED);
    }

    @DataProvider(name = "Create3")
    public Object[][] dataCreate3(){
        return new Object[][]{
            {Service.State.CREATING, Service.State.CREATING, false},
            {Service.State.CREATED, Service.State.CREATED, false},
            {Service.State.STARTING, Service.State.STARTING, false},
            {Service.State.STARTED, Service.State.STARTED, false},
            {Service.State.STOPPING, Service.State.CREATED, true},
            {Service.State.STOPPED, Service.State.CREATED, true},
            {Service.State.DESTROYING, Service.State.CREATED, true},
            {Service.State.DESTROYED, Service.State.CREATED, true},
            {Service.State.FAILED, Service.State.CREATED, true},
            {Service.State.UNKNOWN, Service.State.CREATED, true},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#create()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)"},
          dataProvider = "Create3")
    public void testCreate3(Service.State preState, Service.State expected, boolean isCalled) throws Exception{
        MyService1 service = new MyService1();
        service.state = preState;
        service.create();
        assertEquals(service.getState(), expected);
        assertEquals(service.isCallPreCreateService, isCalled);
        assertEquals(service.isCallCreateService, isCalled);
        assertEquals(service.isCallPostCreateService, isCalled);
    }

    @DataProvider(name = "Start3")
    public Object[][] dataStart3(){
        return new Object[][]{
            {Service.State.CREATED, Service.State.STARTED, true},
            {Service.State.STARTING, Service.State.STARTING, false},
            {Service.State.STARTED, Service.State.STARTED, false},
            {Service.State.STOPPED, Service.State.STARTED, true},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#start()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)"},
          dataProvider = "Start3")
    public void testStart3(Service.State preState, Service.State expected, boolean isCalled) throws Exception{
        MyService1 service = new MyService1();
        service.state = preState;
        service.start();
        assertEquals(service.getState(), expected);
        assertEquals(service.isCallPreStartService, isCalled);
        assertEquals(service.isCallStartService, isCalled);
        assertEquals(service.isCallPostStartService, isCalled);
    }

    @DataProvider(name = "Start3Error")
    public Object[][] dataStart3Error(){
        return new Object[][]{
            {Service.State.CREATING},
            {Service.State.STOPPING},
            {Service.State.DESTROYING},
            {Service.State.DESTROYED},
            {Service.State.FAILED},
            {Service.State.UNKNOWN},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#start()",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)"},
          expectedExceptions = IllegalStateException.class,
          dataProvider = "Start3Error")
    public void testStart3Error(Service.State preState) throws Exception{
        MyService1 service = new MyService1();
        service.state = preState;
        try{
            service.start();
        }finally{
            assertEquals(service.getState(), Service.State.FAILED);
            assertFalse(service.isCallPreStartService);
            assertFalse(service.isCallStartService);
            assertFalse(service.isCallPostStartService);
        }
    }

    @DataProvider(name = "Stop3")
    public Object[][] dataStop3(){
        return new Object[][]{
            {Service.State.STARTED, Service.State.STOPPED, true},
            {Service.State.CREATED, Service.State.CREATED, false},
            {Service.State.STOPPING, Service.State.STOPPING, false},
            {Service.State.STOPPED, Service.State.STOPPED, false},
            {Service.State.DESTROYED, Service.State.DESTROYED, false},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#stop()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)"},
          dataProvider = "Stop3")
    public void testStop3(Service.State preState, Service.State expected, boolean isCalled) throws Exception{
        MyService1 service = new MyService1();
        service.state = preState;
        service.stop();
        assertEquals(service.getState(), expected);
        assertEquals(service.isCallPreStopService, isCalled);
        assertEquals(service.isCallStopService, isCalled);
        assertEquals(service.isCallPostStopService, isCalled);
    }

    @DataProvider(name = "Stop3Error")
    public Object[][] dataStop3Error(){
        return new Object[][]{
            {Service.State.CREATING},
            {Service.State.STARTING},
            {Service.State.DESTROYING},
            {Service.State.FAILED},
            {Service.State.UNKNOWN},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#stop()",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)"},
          dataProvider = "Stop3Error")
    public void testStop3Error(Service.State preState) throws Exception{
        MyService1 service = new MyService1();
        service.state = preState;
        service.stop();
        assertEquals(service.getState(), Service.State.FAILED);
        assertFalse(service.isCallPreStopService);
        assertFalse(service.isCallStopService);
        assertFalse(service.isCallPostStopService);
    }

    @DataProvider(name = "Destroy3")
    public Object[][] dataDestroy3(){
        return new Object[][]{
            {Service.State.CREATED, Service.State.DESTROYED, true},
            {Service.State.STARTED, Service.State.DESTROYED, true},
            {Service.State.STOPPED, Service.State.DESTROYED, true},
            {Service.State.FAILED, Service.State.DESTROYED, true},
            {Service.State.DESTROYING, Service.State.DESTROYING, false},
            {Service.State.DESTROYED, Service.State.DESTROYED, false},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#destroy()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)"},
          dataProvider = "Destroy3")
    public void testDestroy3(Service.State preState, Service.State expected, boolean isCalled) throws Exception{
        MyService1 service = new MyService1();
        service.state = preState;
        service.destroy();
        assertEquals(service.getState(), expected);
        assertEquals(service.isCallPreDestroyService, isCalled);
        assertEquals(service.isCallDestroyService, isCalled);
        assertEquals(service.isCallPostDestroyService, isCalled);
    }

    @DataProvider(name = "Destroy3Error")
    public Object[][] dataDestroy3Error(){
        return new Object[][]{
            {Service.State.CREATING},
            {Service.State.STARTING},
            {Service.State.STOPPING},
            {Service.State.UNKNOWN},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#destroy()",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)"},
          dataProvider = "Destroy3Error")
    public void testDestroy3Error(Service.State preState) throws Exception{
        MyService1 service = new MyService1();
        service.state = preState;
        service.destroy();
        assertEquals(service.getState(), Service.State.FAILED);
        assertFalse(service.isCallPreDestroyService);
        assertFalse(service.isCallDestroyService);
        assertFalse(service.isCallPostDestroyService);
    }
    
    @DataProvider(name = "AddServiceStateListener")
    public Object[][] dataAddServiceStateListener(){
        return new Object[][]{
            {-1, null, new Service.State[0]},
            {-1, null, new Service.State[]{Service.State.CREATING}},
            {-1, null, new Service.State[]{Service.State.CREATED}},
            {-1, null, new Service.State[]{Service.State.STARTING}},
            {-1, null, new Service.State[]{Service.State.STARTED}},
            {-1, null, new Service.State[]{Service.State.STOPPING}},
            {-1, null, new Service.State[]{Service.State.STOPPED}},
            {-1, null, new Service.State[]{Service.State.DESTROYING}},
            {-1, null, new Service.State[]{Service.State.DESTROYED}},
            {-1, null, new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED}},
            {0, new Exception("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {1, new Exception("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {2, new Exception("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {3, new Exception("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {4, new Exception("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {5, new Exception("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {6, new Exception("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {7, new Exception("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {8, new Exception("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {9, new Exception("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {10, new Exception("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {11, new Exception("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {0, new Error("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {1, new Error("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {2, new Error("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {3, new Error("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {4, new Error("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {5, new Error("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {6, new Error("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {7, new Error("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {8, new Error("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {9, new Error("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {10, new Error("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
            {11, new Error("test"), new Service.State[]{Service.State.CREATING, Service.State.CREATED, Service.State.STARTING, Service.State.STARTED, Service.State.STOPPING, Service.State.STOPPED, Service.State.DESTROYING, Service.State.DESTROYED, Service.State.FAILED}},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceBase#addServiceStateListener(jp.ossc.nimbus.core.ServiceStateListener)",
                    "jp.ossc.nimbus.core.ServiceBase#removeServiceStateListener(jp.ossc.nimbus.core.ServiceStateListener)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#ServiceBase\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#create\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#start\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#stop\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceBase#destroy\\(\\)"},
          dataProvider = "AddServiceStateListener")
    public void testAddServiceStateListener(int failedPoint, Throwable th, Service.State[] enableStates) throws Throwable{
        MyServiceStateListener listener = new MyServiceStateListener();
        listener.enableStates = enableStates;
        Set<Service.State> enableStateSet = new HashSet<Service.State>();
        Collections.addAll(enableStateSet, enableStates);
        MyService1 service = new MyService1();
        service.throwable = th;
        switch(failedPoint){
        case 0:
            service.isThrowPreCreateService = true;
            break;
        case 1:
            service.isThrowCreateService = true;
            break;
        case 2:
            service.isThrowPostCreateService = true;
            break;
        case 3:
            service.isThrowPreStartService = true;
            break;
        case 4:
            service.isThrowStartService = true;
            break;
        case 5:
            service.isThrowPostStartService = true;
            break;
        case 6:
            service.isThrowPreStopService = true;
            break;
        case 7:
            service.isThrowStopService = true;
            break;
        case 8:
            service.isThrowPostStopService = true;
            break;
        case 9:
            service.isThrowPreDestroyService = true;
            break;
        case 10:
            service.isThrowDestroyService = true;
            break;
        case 11:
            service.isThrowPostDestroyService = true;
            break;
        }
        service.addServiceStateListener(listener);
        try{
            service.create();
        }catch(Throwable e){
            ServiceStateChangeEvent event = null;
            Service.State state = null;
            if(enableStateSet.contains(Service.State.CREATING)){
                assertTrue(listener.events.size() != 0);
                event = listener.events.remove(0);
                state = listener.eventStates.remove(0);
                assertEquals(event.getSource(), service);
                assertEquals(event.getService(), service);
                assertEquals(state, Service.State.CREATING);
            }
            switch(failedPoint){
            case 0:
            case 1:
                if(enableStateSet.contains(Service.State.FAILED)){
                    assertTrue(listener.events.size() != 0);
                    event = listener.events.remove(0);
                    state = listener.eventStates.remove(0);
                    assertEquals(event.getSource(), service);
                    assertEquals(event.getService(), service);
                    assertEquals(event.getThrowable(), th);
                    assertEquals(state, Service.State.FAILED);
                }
                break;
            case 2:
                if(enableStateSet.contains(Service.State.CREATED)){
                    assertTrue(listener.events.size() != 0);
                    event = listener.events.remove(0);
                    state = listener.eventStates.remove(0);
                    assertEquals(event.getSource(), service);
                    assertEquals(event.getService(), service);
                    assertEquals(state, Service.State.CREATED);
                }
                if(enableStateSet.contains(Service.State.FAILED)){
                    assertTrue(listener.events.size() != 0);
                    event = listener.events.remove(0);
                    state = listener.eventStates.remove(0);
                    assertEquals(event.getSource(), service);
                    assertEquals(event.getService(), service);
                    assertEquals(event.getThrowable(), th);
                    assertEquals(state, Service.State.FAILED);
                }
                break;
            }
            assertTrue(listener.events.size() == 0);
            return;
        }
        if(enableStateSet.contains(Service.State.CREATING)){
            assertTrue(listener.events.size() != 0);
            ServiceStateChangeEvent event = listener.events.remove(0);
            Service.State state = listener.eventStates.remove(0);
            assertEquals(event.getSource(), service);
            assertEquals(event.getService(), service);
            assertEquals(state, Service.State.CREATING);
        }
        if(enableStateSet.contains(Service.State.CREATED)){
            assertTrue(listener.events.size() != 0);
            ServiceStateChangeEvent event = listener.events.remove(0);
            Service.State state = listener.eventStates.remove(0);
            assertEquals(event.getSource(), service);
            assertEquals(event.getService(), service);
            assertEquals(state, Service.State.CREATED);
        }
        assertTrue(listener.events.size() == 0);
        try{
            service.start();
        }catch(Throwable e){
            ServiceStateChangeEvent event = null;
            Service.State state = null;
            if(enableStateSet.contains(Service.State.STARTING)){
                assertTrue(listener.events.size() != 0);
                event = listener.events.remove(0);
                state = listener.eventStates.remove(0);
                assertEquals(event.getSource(), service);
                assertEquals(event.getService(), service);
                assertEquals(state, Service.State.STARTING);
            }
            switch(failedPoint){
            case 3:
            case 4:
                if(enableStateSet.contains(Service.State.FAILED)){
                    assertTrue(listener.events.size() != 0);
                    event = listener.events.remove(0);
                    state = listener.eventStates.remove(0);
                    assertEquals(event.getSource(), service);
                    assertEquals(event.getService(), service);
                    assertEquals(event.getThrowable(), th);
                    assertEquals(state, Service.State.FAILED);
                }
                break;
            case 5:
                if(enableStateSet.contains(Service.State.STARTED)){
                    assertTrue(listener.events.size() != 0);
                    event = listener.events.remove(0);
                    state = listener.eventStates.remove(0);
                    assertEquals(event.getSource(), service);
                    assertEquals(event.getService(), service);
                    assertEquals(state, Service.State.STARTED);
                }
                if(enableStateSet.contains(Service.State.FAILED)){
                    assertTrue(listener.events.size() != 0);
                    event = listener.events.remove(0);
                    state = listener.eventStates.remove(0);
                    assertEquals(event.getSource(), service);
                    assertEquals(event.getService(), service);
                    assertEquals(event.getThrowable(), th);
                    assertEquals(state, Service.State.FAILED);
                }
                break;
            }
            assertTrue(listener.events.size() == 0);
            return;
        }
        if(enableStateSet.contains(Service.State.STARTING)){
            assertTrue(listener.events.size() != 0);
            ServiceStateChangeEvent event = listener.events.remove(0);
            Service.State state = listener.eventStates.remove(0);
            assertEquals(event.getSource(), service);
            assertEquals(event.getService(), service);
            assertEquals(state, Service.State.STARTING);
        }
        if(enableStateSet.contains(Service.State.STARTED)){
            assertTrue(listener.events.size() != 0);
            ServiceStateChangeEvent event = listener.events.remove(0);
            Service.State state = listener.eventStates.remove(0);
            assertEquals(event.getSource(), service);
            assertEquals(event.getService(), service);
            assertEquals(state, Service.State.STARTED);
        }
        assertTrue(listener.events.size() == 0);
        service.stop();
        if(5 < failedPoint && failedPoint < 9){
            ServiceStateChangeEvent event = null;
            Service.State state = null;
            if(enableStateSet.contains(Service.State.STOPPING)){
                assertTrue(listener.events.size() != 0);
                event = listener.events.remove(0);
                state = listener.eventStates.remove(0);
                assertEquals(event.getSource(), service);
                assertEquals(event.getService(), service);
                assertEquals(state, Service.State.STOPPING);
            }
            switch(failedPoint){
            case 6:
            case 7:
                if(enableStateSet.contains(Service.State.FAILED)){
                    assertTrue(listener.events.size() != 0);
                    event = listener.events.remove(0);
                    state = listener.eventStates.remove(0);
                    assertEquals(event.getSource(), service);
                    assertEquals(event.getService(), service);
                    assertEquals(event.getThrowable(), th);
                    assertEquals(state, Service.State.FAILED);
                }
                break;
            case 8:
                if(enableStateSet.contains(Service.State.STOPPED)){
                    assertTrue(listener.events.size() != 0);
                    event = listener.events.remove(0);
                    state = listener.eventStates.remove(0);
                    assertEquals(event.getSource(), service);
                    assertEquals(event.getService(), service);
                    assertEquals(state, Service.State.STOPPED);
                }
                if(enableStateSet.contains(Service.State.FAILED)){
                    assertTrue(listener.events.size() != 0);
                    event = listener.events.remove(0);
                    state = listener.eventStates.remove(0);
                    assertEquals(event.getSource(), service);
                    assertEquals(event.getService(), service);
                    assertEquals(event.getThrowable(), th);
                    assertEquals(state, Service.State.FAILED);
                }
                break;
            }
            assertTrue(listener.events.size() == 0);
        }else{
            if(enableStateSet.contains(Service.State.STOPPING)){
                assertTrue(listener.events.size() != 0);
                ServiceStateChangeEvent event = listener.events.remove(0);
                Service.State state = listener.eventStates.remove(0);
                assertEquals(event.getSource(), service);
                assertEquals(event.getService(), service);
                assertEquals(state, Service.State.STOPPING);
            }
            if(enableStateSet.contains(Service.State.STOPPED)){
                assertTrue(listener.events.size() != 0);
                ServiceStateChangeEvent event = listener.events.remove(0);
                Service.State state = listener.eventStates.remove(0);
                assertEquals(event.getSource(), service);
                assertEquals(event.getService(), service);
                assertEquals(state, Service.State.STOPPED);
            }
            assertTrue(listener.events.size() == 0);
        }
        service.destroy();
        if(8 < failedPoint){
            ServiceStateChangeEvent event = null;
            Service.State state = null;
            if(enableStateSet.contains(Service.State.DESTROYING)){
                assertTrue(listener.events.size() != 0);
                event = listener.events.remove(0);
                state = listener.eventStates.remove(0);
                assertEquals(event.getSource(), service);
                assertEquals(event.getService(), service);
                assertEquals(state, Service.State.DESTROYING);
            }
            switch(failedPoint){
            case 9:
            case 10:
                if(enableStateSet.contains(Service.State.FAILED)){
                    assertTrue(listener.events.size() != 0);
                    event = listener.events.remove(0);
                    state = listener.eventStates.remove(0);
                    assertEquals(event.getSource(), service);
                    assertEquals(event.getService(), service);
                    assertEquals(event.getThrowable(), th);
                    assertEquals(state, Service.State.FAILED);
                }
                break;
            case 11:
                if(enableStateSet.contains(Service.State.DESTROYED)){
                    assertTrue(listener.events.size() != 0);
                    event = listener.events.remove(0);
                    state = listener.eventStates.remove(0);
                    assertEquals(event.getSource(), service);
                    assertEquals(event.getService(), service);
                    assertEquals(state, Service.State.DESTROYED);
                }
                if(enableStateSet.contains(Service.State.FAILED)){
                    assertTrue(listener.events.size() != 0);
                    event = listener.events.remove(0);
                    state = listener.eventStates.remove(0);
                    assertEquals(event.getSource(), service);
                    assertEquals(event.getService(), service);
                    assertEquals(event.getThrowable(), th);
                    assertEquals(state, Service.State.FAILED);
                }
                break;
            }
            assertTrue(listener.events.size() == 0);
        }else{
            if(enableStateSet.contains(Service.State.DESTROYING)){
                assertTrue(listener.events.size() != 0);
                ServiceStateChangeEvent event = listener.events.remove(0);
                Service.State state = listener.eventStates.remove(0);
                assertEquals(event.getSource(), service);
                assertEquals(event.getService(), service);
                assertEquals(state, Service.State.DESTROYING);
            }
            if(enableStateSet.contains(Service.State.DESTROYED)){
                assertTrue(listener.events.size() != 0);
                ServiceStateChangeEvent event = listener.events.remove(0);
                Service.State state = listener.eventStates.remove(0);
                assertEquals(event.getSource(), service);
                assertEquals(event.getService(), service);
                assertEquals(state, Service.State.DESTROYED);
            }
            assertTrue(listener.events.size() == 0);
        }
        
        service.removeServiceStateListener(listener);
        service.create();
        assertTrue(listener.events.size() == 0);
        service.start();
        assertTrue(listener.events.size() == 0);
        service.stop();
        assertTrue(listener.events.size() == 0);
        service.destroy();
        assertTrue(listener.events.size() == 0);
    }
    
    private static class MyService1 extends ServiceBase{
       private static final long serialVersionUID = 845227456635437637L;
       
       public boolean isCallPreCreateService = false;
       public boolean isCallCreateService = false;
       public boolean isCallPostCreateService = false;
       public boolean isCallPreStartService = false;
       public boolean isCallStartService = false;
       public boolean isCallPostStartService = false;
       public boolean isCallPreStopService = false;
       public boolean isCallStopService = false;
       public boolean isCallPostStopService = false;
       public boolean isCallPreDestroyService = false;
       public boolean isCallDestroyService = false;
       public boolean isCallPostDestroyService = false;

       public boolean isThrowPreCreateService = false;
       public boolean isThrowCreateService = false;
       public boolean isThrowPostCreateService = false;
       public boolean isThrowPreStartService = false;
       public boolean isThrowStartService = false;
       public boolean isThrowPostStartService = false;
       public boolean isThrowPreStopService = false;
       public boolean isThrowStopService = false;
       public boolean isThrowPostStopService = false;
       public boolean isThrowPreDestroyService = false;
       public boolean isThrowDestroyService = false;
       public boolean isThrowPostDestroyService = false;
       public Throwable throwable;
       
       @Override
       protected void preCreateService() throws Exception{
           super.preCreateService();
           isCallPreCreateService = true;
           if(isThrowPreCreateService){
               if(throwable instanceof Exception){
                   throw (Exception)throwable;
               }else{
                   throw (Error)throwable;
               }
           }
       }
       
       @Override
       public void createService() throws Exception{
           isCallCreateService = true;
           if(isThrowCreateService){
               if(throwable instanceof Exception){
                   throw (Exception)throwable;
               }else{
                   throw (Error)throwable;
               }
           }
       }
       
       @Override
       protected void postCreateService() throws Exception{
           super.postCreateService();
           isCallPostCreateService = true;
           if(isThrowPostCreateService){
               if(throwable instanceof Exception){
                   throw (Exception)throwable;
               }else{
                   throw (Error)throwable;
               }
           }
       }
       
       @Override
       protected void preStartService() throws Exception{
           super.preStartService();
           isCallPreStartService = true;
           if(isThrowPreStartService){
               if(throwable instanceof Exception){
                   throw (Exception)throwable;
               }else{
                   throw (Error)throwable;
               }
           }
       }
       
       @Override
       public void startService() throws Exception{
           isCallStartService = true;
           if(isThrowStartService){
               if(throwable instanceof Exception){
                   throw (Exception)throwable;
               }else{
                   throw (Error)throwable;
               }
           }
       }
       
       @Override
       protected void postStartService() throws Exception{
           super.postStartService();
           isCallPostStartService = true;
           if(isThrowPostStartService){
               if(throwable instanceof Exception){
                   throw (Exception)throwable;
               }else{
                   throw (Error)throwable;
               }
           }
       }
       
       @Override
       protected void preStopService() throws Exception{
           super.preStopService();
           isCallPreStopService = true;
           if(isThrowPreStopService){
               if(throwable instanceof Exception){
                   throw (Exception)throwable;
               }else{
                   throw (Error)throwable;
               }
           }
       }
       
       @Override
       public void stopService() throws Exception{
           isCallStopService = true;
           if(isThrowStopService){
               if(throwable instanceof Exception){
                   throw (Exception)throwable;
               }else{
                   throw (Error)throwable;
               }
           }
       }
       
       @Override
       protected void postStopService() throws Exception{
           super.postStopService();
           isCallPostStopService = true;
           if(isThrowPostStopService){
               if(throwable instanceof Exception){
                   throw (Exception)throwable;
               }else{
                   throw (Error)throwable;
               }
           }
       }
       
       @Override
       protected void preDestroyService() throws Exception{
           super.preDestroyService();
           isCallPreDestroyService = true;
           if(isThrowPreDestroyService){
               if(throwable instanceof Exception){
                   throw (Exception)throwable;
               }else{
                   throw (Error)throwable;
               }
           }
       }
       
       @Override
       public void destroyService() throws Exception{
           isCallDestroyService = true;
           if(isThrowDestroyService){
               if(throwable instanceof Exception){
                   throw (Exception)throwable;
               }else{
                   throw (Error)throwable;
               }
           }
       }
       
       @Override
       protected void postDestroyService() throws Exception{
           super.postDestroyService();
           isCallPostDestroyService = true;
           if(isThrowPostDestroyService){
               if(throwable instanceof Exception){
                   throw (Exception)throwable;
               }else{
                   throw (Error)throwable;
               }
           }
       }
    }
    
    private static class MyService2 extends ServiceBase{
       
        private static final long serialVersionUID = -669086786209116740L;
    
        public MyService2(ServiceBaseSupport support){
            super(support);
        }
    }
    
    private static class MyServiceBaseSupport implements ServiceBaseSupport{
        
        public boolean isCallSetServiceBase = false;
        public boolean isCallCreateService = false;
        public boolean isCallStartService = false;
        public boolean isCallStopService = false;
        public boolean isCallDestroyService = false;
        
        public boolean isThrowCreateService = false;
        public boolean isThrowStartService = false;
        public boolean isThrowStopService = false;
        public boolean isThrowDestroyService = false;
        public Throwable throwable;
        
        @Override
        public void createService() throws Exception{
            isCallCreateService = true;
            if(isThrowCreateService){
                if(throwable instanceof Exception){
                    throw (Exception)throwable;
                }else{
                    throw (Error)throwable;
                }
            }
        }

        @Override
        public void startService() throws Exception{
            isCallStartService = true;
            if(isThrowStartService){
                if(throwable instanceof Exception){
                    throw (Exception)throwable;
                }else{
                    throw (Error)throwable;
                }
            }
        }

        @Override
        public void stopService() throws Exception{
            isCallStopService = true;
            if(isThrowStopService){
                if(throwable instanceof Exception){
                    throw (Exception)throwable;
                }else{
                    throw (Error)throwable;
                }
            }
        }

        @Override
        public void destroyService() throws Exception{
            isCallDestroyService = true;
            if(isThrowDestroyService){
                if(throwable instanceof Exception){
                    throw (Exception)throwable;
                }else{
                    throw (Error)throwable;
                }
            }
        }

        @Override
        public void setServiceBase(ServiceBase service){
            isCallSetServiceBase = true;
        }
        
    }
    
    private static class MyLogger implements Logger{

        @Override
        public boolean isWrite(String messageId){
            return false;
        }

        @Override
        public void write(String messageId, Object... embed){
        }

        @Override
        public void write(String messageId, Throwable exception, Object... embed){
            
        }

        @Override
        public void write(String messageId, Locale lo, Object... embed){
        }

        @Override
        public void write(String messageId, Locale lo, Throwable exception, Object... embed){
        }
    }
    
    private static class MyMessageRecordFactory implements MessageRecordFactory{

        @Override
        public void findLocale(Locale lo){
        }

        @Override
        public String findMessage(String messageId, Object... embeds){
            return null;
        }

        @Override
        public String findMessage(String messageId, Locale lo, Object... embeds){
            return null;
        }

        @Override
        public MessageRecord findMessageRecord(String messageId){
            return null;
        }

        @Override
        public String findMessageTemplete(String messageId){
            return null;
        }

        @Override
        public String findMessageTemplete(String messageId, Locale lo){
            return null;
        }
    }
    
    private static class MyServiceStateListener implements ServiceStateListener{
        
        public Service.State[] enableStates;
        public List<ServiceStateChangeEvent> events = new ArrayList<ServiceStateChangeEvent>();
        public List<Service.State> eventStates = new ArrayList<Service.State>();
        
        @Override
        public boolean isEnabledState(State state){
            if(enableStates == null || enableStates.length == 0){
                return false;
            }
            for(Service.State enableState : enableStates){
                if(state == enableState){
                    return true;
                }
            }
            return false;
        }

        @Override
        public void stateChanged(ServiceStateChangeEvent e) throws Exception{
            events.add(e);
            eventStates.add(e.getService().getState());
        }
    }
}
