/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanInfo;
import javax.management.MBeanOperationInfo;

import static org.testng.Assert.*;
import org.testng.annotations.DataProvider;
import org.testng.annotations.Test;

import jp.ossc.nimbus.service.log.DefaultLoggerService;
import jp.ossc.nimbus.service.message.DefaultMessageRecordFactoryService;

/**
 * {@link GenericsFactoryServiceProxy}eXgP[XB<p>
 * 
 * @author M.Takata
 */
public class GenericsFactoryServiceProxyTest{
    
    @DataProvider(name = "GenericsFactoryServiceProxy")
    public Object[][] dataGenericsFactoryServiceProxy(){
        return new Object[][]{
            {POJO.class, false, false},
            {POJO.class, true, false},
            {POJOService.class, false, false},
            {POJOService.class, true, false},
            {MyServiceBaseSupport.class, false, false},
            {MyServiceBaseSupport.class, true, false},
            {MyServiceBaseSupport2.class, false, false},
            {MyServiceBaseSupport2.class, true, false},
            {MyServiceBase.class, false, false},
            {MyServiceBase.class, true, false},
            {MyServiceBase2.class, false, false},
            {MyServiceBase2.class, true, false},
            {MyService.class, false, false},
            {MyService.class, true, false},
            {POJO.class, false, true},
            {POJO.class, true, true},
            {POJOService.class, false, true},
            {POJOService.class, true, true},
            {MyServiceBaseSupport.class, false, true},
            {MyServiceBaseSupport.class, true, true},
            {MyServiceBaseSupport2.class, false, true},
            {MyServiceBaseSupport2.class, true, true},
            {MyServiceBase.class, false, true},
            {MyServiceBase.class, true, true},
            {MyServiceBase2.class, false, true},
            {MyServiceBase2.class, true, true},
            {MyService.class, false, true},
            {MyService.class, true, true}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.GenericsFactoryServiceProxy#GenericsFactoryServiceProxy(java.lang.Class,boolean)",
                    "jp.ossc.nimbus.core.GenericsFactoryServiceProxy#getMBeanInfo()",
                    "jp.ossc.nimbus.core.GenericsFactoryServiceProxy#setAttribute(javax.management.Attribute)",
                    "jp.ossc.nimbus.core.GenericsFactoryServiceProxy#getAttribute(java.lang.String)",
                    "jp.ossc.nimbus.core.GenericsFactoryServiceProxy#getAttributes(java.lang.String[])",
                    "jp.ossc.nimbus.core.GenericsFactoryServiceProxy#setAttributes(javax.management.AttributeList)",
                    "jp.ossc.nimbus.core.GenericsFactoryServiceProxy#invoke(String,java.lang.Object[],java.lang.String[])",
                    "jp.ossc.nimbus.core.GenericsFactoryServiceProxy#setServiceMetaData(jp.ossc.nimbus.core.ServiceMetaData)",
                    "jp.ossc.nimbus.core.GenericsFactoryServiceProxy#createInstance()",
                    "jp.ossc.nimbus.core.GenericsFactoryServiceProxy#createInstance(boolean)",
                    "jp.ossc.nimbus.core.GenericsFactoryServiceProxy#createTemplate()",
                    "jp.ossc.nimbus.core.GenericsFactoryServiceProxy#release(java.lang.Object)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.FactoryServiceBase#.*"},
          dataProvider = "GenericsFactoryServiceProxy")
    public <T> void testGenericsFactoryServiceProxy(Class<T> clazz, boolean isManagement, boolean isThreadLocal) throws Exception{
        GenericsFactoryServiceProxy<T> service = new GenericsFactoryServiceProxy<T>(clazz,isManagement);
        service.setThreadLocal(isThreadLocal);
        assertNotNull(service.getLogger());
        assertNotNull(service.getMessageRecordFactory());
        assertNull(service.getServiceManager());
        assertNull(service.getServiceManagerName());
        assertNull(service.getServiceName());
        assertNull(service.getServiceNameObject());
        assertEquals(service.getState(), Service.State.DESTROYED);
        assertNull(service.getSystemLoggerServiceName());
        assertNull(service.getSystemMessageRecordFactoryServiceName());
        assertEquals(service.getTarget(), service);
        
        try{
            service.setServiceManagerName("test");
            service.setServiceName("Service1");
            ServiceMetaData metaData = new ServiceMetaData("Service1", clazz.getName());
            service.setServiceMetaData(metaData);
            ServiceManagerFactory.registerManager("test");
            DefaultLoggerService logger = new DefaultLoggerService();
            logger.setServiceManagerName("test");
            logger.setServiceName("Logger");
            logger.create();
            logger.start();
            DefaultMessageRecordFactoryService message = new DefaultMessageRecordFactoryService();
            message.setServiceManagerName("test");
            message.setServiceName("Message");
            message.create();
            message.start();
            MBeanInfo info = service.getMBeanInfo();
            assertNotNull(info);
            MBeanAttributeInfo[] attributeInfos = info.getAttributes();
            assertNotNull(attributeInfos);
            if((clazz.equals(MyServiceBaseSupport2.class) || clazz.equals(MyServiceBase2.class)) && isManagement){
                assertEquals(attributeInfos.length, 12);
            }else{
                assertEquals(attributeInfos.length, 9);
            }
            for(MBeanAttributeInfo attributeInfo : attributeInfos){
                if("ServiceName".equals(attributeInfo.getName())){
                    assertEquals(attributeInfo.getType(), "java.lang.String");
                    assertTrue(attributeInfo.isReadable());
                    assertTrue(attributeInfo.isWritable());
                    assertEquals(service.getAttribute(attributeInfo.getName()), "Service1");
                    service.setAttribute(new Attribute(attributeInfo.getName(), "Service2"));
                    assertEquals(service.getAttribute(attributeInfo.getName()), "Service2");
                    service.setAttribute(new Attribute(attributeInfo.getName(), "Service1"));
                }else if("ServiceManagerName".equals(attributeInfo.getName())){
                    assertEquals(attributeInfo.getType(), "java.lang.String");
                    assertTrue(attributeInfo.isReadable());
                    assertTrue(attributeInfo.isWritable());
                    assertEquals(service.getAttribute(attributeInfo.getName()), "test");
                    service.setAttribute(new Attribute(attributeInfo.getName(), "test2"));
                    assertEquals(service.getAttribute(attributeInfo.getName()), "test2");
                    service.setAttribute(new Attribute(attributeInfo.getName(), "test"));
                }else if("ServiceNameObject".equals(attributeInfo.getName())){
                    assertEquals(attributeInfo.getType(), "jp.ossc.nimbus.core.ServiceName");
                    assertTrue(attributeInfo.isReadable());
                    assertFalse(attributeInfo.isWritable());
                    assertEquals(service.getAttribute(attributeInfo.getName()), new ServiceName("test", "Service1"));
                }else if("State".equals(attributeInfo.getName())){
                    assertEquals(attributeInfo.getType(), "jp.ossc.nimbus.core.Service$State");
                    assertTrue(attributeInfo.isReadable());
                    assertFalse(attributeInfo.isWritable());
                    assertEquals(service.getAttribute(attributeInfo.getName()), Service.State.DESTROYED);
                }else if("SystemLoggerServiceName".equals(attributeInfo.getName())){
                    assertEquals(attributeInfo.getType(), "jp.ossc.nimbus.core.ServiceName");
                    assertTrue(attributeInfo.isReadable());
                    assertTrue(attributeInfo.isWritable());
                    assertNull(service.getAttribute(attributeInfo.getName()));
                    service.setAttribute(new Attribute(attributeInfo.getName(), new ServiceName("test", "Logger")));
                    assertEquals(service.getAttribute(attributeInfo.getName()), new ServiceName("test", "Logger"));
                    service.setAttribute(new Attribute(attributeInfo.getName(), null));
                }else if("SystemMessageRecordFactoryServiceName".equals(attributeInfo.getName())){
                    assertEquals(attributeInfo.getType(), "jp.ossc.nimbus.core.ServiceName");
                    assertTrue(attributeInfo.isReadable());
                    assertTrue(attributeInfo.isWritable());
                    assertNull(service.getAttribute(attributeInfo.getName()));
                    service.setAttribute(new Attribute(attributeInfo.getName(), new ServiceName("test", "Message")));
                    assertEquals(service.getAttribute(attributeInfo.getName()), new ServiceName("test", "Message"));
                    service.setAttribute(new Attribute(attributeInfo.getName(), null));
                }else if("ThreadLocal".equals(attributeInfo.getName())){
                    assertEquals(attributeInfo.getType(), "boolean");
                    assertTrue(attributeInfo.isReadable());
                    assertTrue(attributeInfo.isWritable());
                    assertEquals(((Boolean)service.getAttribute(attributeInfo.getName())).booleanValue(), isThreadLocal);
                    service.setAttribute(new Attribute(attributeInfo.getName(), isThreadLocal ? Boolean.FALSE : Boolean.TRUE));
                    assertEquals(service.getAttribute(attributeInfo.getName()), isThreadLocal ? Boolean.FALSE : Boolean.TRUE);
                    service.setAttribute(new Attribute(attributeInfo.getName(), isThreadLocal ? Boolean.TRUE : Boolean.FALSE));
                }else if("Management".equals(attributeInfo.getName())){
                    assertEquals(attributeInfo.getType(), "boolean");
                    assertTrue(attributeInfo.isReadable());
                    assertTrue(attributeInfo.isWritable());
                    assertEquals(((Boolean)service.getAttribute(attributeInfo.getName())).booleanValue(), isManagement);
                    service.setAttribute(new Attribute(attributeInfo.getName(), isManagement ? Boolean.FALSE : Boolean.TRUE));
                    assertEquals(service.getAttribute(attributeInfo.getName()), isManagement ? Boolean.FALSE : Boolean.TRUE);
                    service.setAttribute(new Attribute(attributeInfo.getName(), isManagement ? Boolean.TRUE : Boolean.FALSE));
                }else if("CreateTemplateOnStart".equals(attributeInfo.getName())){
                    assertEquals(attributeInfo.getType(), "boolean");
                    assertTrue(attributeInfo.isReadable());
                    assertTrue(attributeInfo.isWritable());
                    assertTrue(((Boolean)service.getAttribute(attributeInfo.getName())).booleanValue());
                    service.setAttribute(new Attribute(attributeInfo.getName(), Boolean.FALSE));
                    assertEquals(service.getAttribute(attributeInfo.getName()), Boolean.FALSE);
                    service.setAttribute(new Attribute(attributeInfo.getName(), Boolean.TRUE));
                }else{
                    if(clazz.equals(MyServiceBaseSupport2.class) || clazz.equals(MyServiceBase2.class)){
                        if("Fuga".equals(attributeInfo.getName())){
                            assertEquals(attributeInfo.getType(), "int");
                            assertTrue(attributeInfo.isReadable());
                            assertEquals(service.getAttribute(attributeInfo.getName()), new Integer(100));
                        }else if("Hoge".equals(attributeInfo.getName())){
                            assertEquals(attributeInfo.getType(), "java.lang.String");
                            assertTrue(attributeInfo.isReadable());
                            assertTrue(attributeInfo.isWritable());
                            assertNull(service.getAttribute(attributeInfo.getName()));
                            service.setAttribute(new Attribute(attributeInfo.getName(), "hoge"));
                            assertEquals(service.getAttribute(attributeInfo.getName()), "hoge");
                        }else if("Bar".equals(attributeInfo.getName())){
                            assertEquals(attributeInfo.getType(), "java.lang.Object");
                            assertFalse(attributeInfo.isReadable());
                            assertTrue(attributeInfo.isWritable());
                            service.setAttribute(new Attribute(attributeInfo.getName(), "test"));
                        }else{
                            throw new Exception("Failed. unexpected attribute." + attributeInfo);
                        }
                    }else{
                        throw new Exception("Failed. unexpected attribute." + attributeInfo);
                    }
                }
            }
            AttributeList attributeList = service.getAttributes(new String[]{"ServiceName", "ServiceManagerName"});
            for(Object attribute : attributeList){
                Attribute attr = (Attribute)attribute;
                if("ServiceName".equals(attr.getName())){
                    assertEquals(attr.getValue(), "Service1");
                }else if("ServiceManagerName".equals(attr.getName())){
                    assertEquals(attr.getValue(), "test");
                }
            }
            attributeList = new AttributeList();
            attributeList.add(new Attribute("ServiceName", "Service2"));
            attributeList.add(new Attribute("ServiceManagerName", "test2"));
            attributeList = service.setAttributes(attributeList);
            for(Object attribute : attributeList){
                Attribute attr = (Attribute)attribute;
                if("ServiceName".equals(attr.getName())){
                    assertEquals(attr.getValue(), "Service2");
                    assertEquals(service.getServiceName(), "Service2");
                }else if("ServiceManagerName".equals(attr.getName())){
                    assertEquals(attr.getValue(), "test2");
                    assertEquals(service.getServiceManagerName(), "test2");
                }
            }
            service.setServiceManagerName("test");
            service.setServiceName("Service1");
            
            MBeanOperationInfo[] operationInfos = info.getOperations();
            assertNotNull(operationInfos);
            if((clazz.equals(MyServiceBaseSupport2.class) || clazz.equals(MyServiceBase2.class)) && isManagement){
                assertEquals(operationInfos.length, 9);
            }else{
                assertEquals(operationInfos.length, 8);
            }
            for(MBeanOperationInfo operationInfo : operationInfos){
                if("create".equals(operationInfo.getName())){
                    assertEquals(operationInfo.getSignature().length, 0);
                    assertEquals(operationInfo.getReturnType(), "void");
                }else if("start".equals(operationInfo.getName())){
                    assertEquals(operationInfo.getSignature().length, 0);
                    assertEquals(operationInfo.getReturnType(), "void");
                }else if("stop".equals(operationInfo.getName())){
                    assertEquals(operationInfo.getSignature().length, 0);
                    assertEquals(operationInfo.getReturnType(), "void");
                }else if("destroy".equals(operationInfo.getName())){
                    assertEquals(operationInfo.getSignature().length, 0);
                    assertEquals(operationInfo.getReturnType(), "void");
                }else if("restart".equals(operationInfo.getName())){
                    assertEquals(operationInfo.getSignature().length, 0);
                    assertEquals(operationInfo.getReturnType(), "void");
                }else if("release".equals(operationInfo.getName())){
                    if(operationInfo.getSignature().length == 1){
                        assertEquals(operationInfo.getSignature()[0].getType(), Object.class.getName());
                    }else{
                        assertEquals(operationInfo.getSignature().length, 0);
                    }
                    assertEquals(operationInfo.getReturnType(), "void");
                }else if("newInstance".equals(operationInfo.getName())){
                    assertEquals(operationInfo.getSignature().length, 0);
                    assertEquals(operationInfo.getReturnType(), Object.class.getName());
                }else{
                    if(clazz.equals(MyServiceBaseSupport2.class) || clazz.equals(MyServiceBase2.class)){
                        if("foo".equals(operationInfo.getName())){
                            assertEquals(operationInfo.getSignature().length, 1);
                            assertEquals(operationInfo.getSignature()[0].getType(), Integer.TYPE.getName());
                            assertEquals(operationInfo.getReturnType(), String.class.getName());
                        }else{
                            throw new Exception("Failed. unexpected operation." + operationInfo);
                        }
                    }else{
                        throw new Exception("Failed. unexpected operation." + operationInfo);
                    }
                }
            }
            assertNull(service.invoke("create", null, null));
            assertEquals(service.getState(), Service.State.CREATED);
            assertNull(service.invoke("start", null, null));
            assertEquals(service.getState(), Service.State.STARTED);
            
            T instance = service.newInstance();
            assertNotNull(instance);
            if(isManagement){
                if(Service.class.isAssignableFrom(clazz)){
                    assertEquals(instance.getClass(), clazz);
                }else{
                    assertEquals(instance.getClass(), GenericsServiceProxy.class);
                }
            }else{
                if(clazz.equals(MyServiceBaseSupport.class) || clazz.equals(MyServiceBaseSupport2.class)){
                    assertEquals(instance.getClass(), GenericsServiceProxy.class);
                }else{
                    assertEquals(instance.getClass(), clazz);
                }
            }
            T instance2 = service.newInstance();
            if(isThreadLocal){
                assertEquals(instance2, instance);
            }else{
                assertNotSame(instance2, instance);
            }
            if(isManagement){
                if(isThreadLocal){
                    assertEquals(service.getManagedInstanceSet().size(), 1);
                    assertEquals(service.getManagedInstanceSet().iterator().next(), instance);
                }else{
                    assertEquals(service.getManagedInstanceSet().size(), 2);
                    service.release(instance2);
                    assertEquals(service.getManagedInstanceSet().size(), 1);
                    assertEquals(service.getManagedInstanceSet().iterator().next(), instance);
                }
            }
            if(clazz.equals(POJOService.class)){
                if(isManagement){
                    assertEquals(((POJOService)((ServiceProxy)((ServiceProxy)instance).getTarget()).getTarget()).calledCreate, 1);
                    assertEquals(((POJOService)((ServiceProxy)((ServiceProxy)instance).getTarget()).getTarget()).calledStart, 1);
                    assertEquals(((POJOService)((ServiceProxy)((ServiceProxy)instance).getTarget()).getTarget()).calledStop, 0);
                    assertEquals(((POJOService)((ServiceProxy)((ServiceProxy)instance).getTarget()).getTarget()).calledDestroy, 0);
                }else{
                    assertEquals(((POJOService)instance).calledCreate, 0);
                    assertEquals(((POJOService)instance).calledStart, 0);
                    assertEquals(((POJOService)instance).calledStop, 0);
                    assertEquals(((POJOService)instance).calledDestroy, 0);
                }
            }else if(clazz.equals(MyServiceBaseSupport.class)){
                assertEquals(((MyServiceBaseSupport)((ServiceProxy)instance).getTarget()).calledCreate, 1);
                assertEquals(((MyServiceBaseSupport)((ServiceProxy)instance).getTarget()).calledStart, 1);
                assertEquals(((MyServiceBaseSupport)((ServiceProxy)instance).getTarget()).calledStop, 0);
                assertEquals(((MyServiceBaseSupport)((ServiceProxy)instance).getTarget()).calledDestroy, 0);
            }else if(clazz.equals(MyServiceBaseSupport2.class)){
                assertEquals(((MyServiceBaseSupport2)((ServiceProxy)instance).getTarget()).calledCreate, 1);
                assertEquals(((MyServiceBaseSupport2)((ServiceProxy)instance).getTarget()).calledStart, 1);
                assertEquals(((MyServiceBaseSupport2)((ServiceProxy)instance).getTarget()).calledStop, 0);
                assertEquals(((MyServiceBaseSupport2)((ServiceProxy)instance).getTarget()).calledDestroy, 0);
            }else if(clazz.equals(MyServiceBase.class)){
                assertEquals(((MyServiceBase)instance).calledCreate, 1);
                assertEquals(((MyServiceBase)instance).calledStart, 1);
                assertEquals(((MyServiceBase)instance).calledStop, 0);
                assertEquals(((MyServiceBase)instance).calledDestroy, 0);
            }else if(clazz.equals(MyService.class)){
                assertEquals(((MyService)instance).calledCreate, 1);
                assertEquals(((MyService)instance).calledStart, 1);
                assertEquals(((MyService)instance).calledStop, 0);
                assertEquals(((MyService)instance).calledDestroy, 0);
            }
            
            assertNull(service.invoke("restart", null, null));
            assertEquals(service.getState(), Service.State.STARTED);
            
            service.release(instance);
            
            assertNull(service.invoke("stop", null, null));
            assertEquals(service.getState(), Service.State.STOPPED);
            assertNull(service.invoke("destroy", null, null));
            assertEquals(service.getState(), Service.State.DESTROYED);
            
            if(clazz.equals(POJOService.class)){
                if(isManagement){
                    assertEquals(((POJOService)((ServiceProxy)((ServiceProxy)instance).getTarget()).getTarget()).calledCreate, 1);
                    assertEquals(((POJOService)((ServiceProxy)((ServiceProxy)instance).getTarget()).getTarget()).calledStart, 2);
                    assertEquals(((POJOService)((ServiceProxy)((ServiceProxy)instance).getTarget()).getTarget()).calledStop, 2);
                    assertEquals(((POJOService)((ServiceProxy)((ServiceProxy)instance).getTarget()).getTarget()).calledDestroy, 1);
                }else{
                    assertEquals(((POJOService)instance).calledCreate, 0);
                    assertEquals(((POJOService)instance).calledStart, 0);
                    assertEquals(((POJOService)instance).calledStop, 0);
                    assertEquals(((POJOService)instance).calledDestroy, 0);
                }
            }else if(clazz.equals(MyServiceBaseSupport.class)){
                if(isManagement){
                    assertEquals(((MyServiceBaseSupport)((ServiceProxy)instance).getTarget()).calledCreate, 1);
                    assertEquals(((MyServiceBaseSupport)((ServiceProxy)instance).getTarget()).calledStart, 2);
                    assertEquals(((MyServiceBaseSupport)((ServiceProxy)instance).getTarget()).calledStop, 2);
                    assertEquals(((MyServiceBaseSupport)((ServiceProxy)instance).getTarget()).calledDestroy, 1);
                }else{
                    assertEquals(((MyServiceBaseSupport)((ServiceProxy)instance).getTarget()).calledCreate, 1);
                    assertEquals(((MyServiceBaseSupport)((ServiceProxy)instance).getTarget()).calledStart, 1);
                    assertEquals(((MyServiceBaseSupport)((ServiceProxy)instance).getTarget()).calledStop, 1);
                    assertEquals(((MyServiceBaseSupport)((ServiceProxy)instance).getTarget()).calledDestroy, 1);
                }
            }else if(clazz.equals(MyServiceBaseSupport2.class)){
                if(isManagement){
                    assertEquals(((MyServiceBaseSupport2)((ServiceProxy)instance).getTarget()).calledCreate, 1);
                    assertEquals(((MyServiceBaseSupport2)((ServiceProxy)instance).getTarget()).calledStart, 2);
                    assertEquals(((MyServiceBaseSupport2)((ServiceProxy)instance).getTarget()).calledStop, 2);
                    assertEquals(((MyServiceBaseSupport2)((ServiceProxy)instance).getTarget()).calledDestroy, 1);
                }else{
                    assertEquals(((MyServiceBaseSupport2)((ServiceProxy)instance).getTarget()).calledCreate, 1);
                    assertEquals(((MyServiceBaseSupport2)((ServiceProxy)instance).getTarget()).calledStart, 1);
                    assertEquals(((MyServiceBaseSupport2)((ServiceProxy)instance).getTarget()).calledStop, 1);
                    assertEquals(((MyServiceBaseSupport2)((ServiceProxy)instance).getTarget()).calledDestroy, 1);
                }
            }else if(clazz.equals(MyServiceBase.class)){
                if(isManagement){
                    assertEquals(((MyServiceBase)instance).calledCreate, 1);
                    assertEquals(((MyServiceBase)instance).calledStart, 2);
                    assertEquals(((MyServiceBase)instance).calledStop, 2);
                    assertEquals(((MyServiceBase)instance).calledDestroy, 1);
                }else{
                    assertEquals(((MyServiceBase)instance).calledCreate, 1);
                    assertEquals(((MyServiceBase)instance).calledStart, 1);
                    assertEquals(((MyServiceBase)instance).calledStop, 1);
                    assertEquals(((MyServiceBase)instance).calledDestroy, 1);
                }
            }else if(clazz.equals(MyService.class)){
                if(isManagement){
                    assertEquals(((MyService)instance).calledCreate, 1);
                    assertEquals(((MyService)instance).calledStart, 2);
                    assertEquals(((MyService)instance).calledStop, 2);
                    assertEquals(((MyService)instance).calledDestroy, 1);
                }else{
                    assertEquals(((MyService)instance).calledCreate, 1);
                    assertEquals(((MyService)instance).calledStart, 1);
                    assertEquals(((MyService)instance).calledStop, 1);
                    assertEquals(((MyService)instance).calledDestroy, 1);
                }
            }
        }finally{
            ServiceManagerFactory.unloadManager("test");
        }
    }
    
    public static class POJO{
    }
    
    public static class POJOService{
        
        public int calledCreate;
        public int calledDestroy;
        public int calledStart;
        public int calledStop;
        
        public void create(){
            calledCreate++;
        }
        public void start(){
            calledStart++;
        }
        public void stop(){
            calledStop++;
        }
        public void destroy(){
            calledDestroy++;
        }
    }
    
    public static class MyServiceBaseSupport implements ServiceBaseSupport{
        
        public int calledSetServiceBase;
        public int calledCreate;
        public int calledDestroy;
        public int calledStart;
        public int calledStop;
        
        @Override
        public void createService(){
            calledCreate++;
        }
        @Override
        public void startService(){
            calledStart++;
        }
        @Override
        public void stopService(){
            calledStop++;
        }
        @Override
        public void destroyService(){
            calledDestroy++;
        }
        @Override
        public void setServiceBase(ServiceBase service){
            calledSetServiceBase++;
        }
    }
    
    public static class MyServiceBaseSupport2 implements ServiceBaseSupport, MyServiceBaseSupport2MBean{
        
        public int calledSetServiceBase;
        public int calledCreate;
        public int calledDestroy;
        public int calledStart;
        public int calledStop;
        
        @Override
        public void createService() throws Exception{
            calledCreate++;
        }

        @Override
        public void destroyService() throws Exception{
            calledDestroy++;
        }

        @Override
        public void startService() throws Exception{
            calledStart++;
        }

        @Override
        public void stopService() throws Exception{
            calledStop++;
        }

        @Override
        public String foo(int index){
            return "foo" + index;
        }

        @Override
        public int getFuga(){
            return 100;
        }
        
        private String hoge;

        @Override
        public String getHoge(){
            return hoge;
        }

        @Override
        public void setHoge(String val){
            hoge = val;
        }
        
        Object bar;
        
        @Override
        public void setBar(Object obj){
            bar = obj;
        }

        @Override
        public void setServiceBase(ServiceBase service){
            calledSetServiceBase++;
        }
    }
    
    private static interface MyServiceBaseSupport2MBean{
        
        public void setHoge(String val);
        public String getHoge();
        
        public int getFuga();
        
        public void setBar(Object obj);
        
        public String foo(int index);
    }
    
    public static class MyServiceBase extends ServiceBase{
        
        private static final long serialVersionUID = -4424209143434260295L;
        
        public int calledCreate;
        public int calledDestroy;
        public int calledStart;
        public int calledStop;
        
        @Override
        public void createService(){
            calledCreate++;
        }
        @Override
        public void startService(){
            calledStart++;
        }
        @Override
        public void stopService(){
            calledStop++;
        }
        @Override
        public void destroyService(){
            calledDestroy++;
        }
    }
    
    public static class MyServiceBase2 extends MyServiceBase implements MyServiceBase2MBean{

        private static final long serialVersionUID = -793256154074610770L;

        @Override
        public String foo(int index){
            return "foo" + index;
        }

        @Override
        public int getFuga(){
            return 100;
        }
        
        private String hoge;

        @Override
        public String getHoge(){
            return hoge;
        }

        @Override
        public void setHoge(String val){
            hoge = val;
        }
        
        transient Object bar;
        
        @Override
        public void setBar(Object obj){
            bar = obj;
        }
    }
    
    private static interface MyServiceBase2MBean extends ServiceBaseMBean{
        
        public void setHoge(String val);
        public String getHoge();
        
        public int getFuga();
        
        public void setBar(Object obj);
        
        public String foo(int index);
    }
    
    public static class MyService implements Service{
        
        public int calledCreate;
        public int calledDestroy;
        public int calledStart;
        public int calledStop;
        
        private String serviceManagerName;
        private String serviceName;
        private State state = State.DESTROYED;
        
        @Override
        public String getServiceManagerName(){
            return serviceManagerName;
        }
        @Override
        public void setServiceManagerName(String name) {
            serviceManagerName = name;
        }
        @Override
        public String getServiceName(){
            return serviceName;
        }
        @Override
        public void setServiceName(String name) {
            serviceName = name;
        }
        @Override
        public ServiceName getServiceNameObject() {
            return serviceManagerName != null && serviceName != null ? new ServiceName(serviceManagerName, serviceName) : null;
        }
        @Override
        public State getState(){
            return state;
        }
        @Override
        public void create(){
            state = State.CREATING;
            calledCreate++;
            state = State.CREATED;
        }
        @Override
        public void start(){
            state = State.STARTING;
            calledStart++;
            state = State.STARTED;
        }
        @Override
        public void stop(){
            state = State.STOPPING;
            calledStop++;
            state = State.STOPPED;
        }
        @Override
        public void destroy(){
            state = State.DESTROYING;
            calledDestroy++;
            state = State.DESTROYED;
        }
    }
}