/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import static org.testng.Assert.*;

import org.testng.annotations.Test;
import org.testng.annotations.DataProvider;
import org.w3c.dom.Element;

import static jp.ossc.nimbus.TestUtility.*;

/**
 * {@link ArgumentMetaData}eXgP[XB<p>
 * 
 * @author M.Takata
 */
public class ArgumentMetaDataTest{

    @DataProvider(name = "ArgumentMetaData")
    public Object[][] dataArgumentMetaData(){
        return new Object[][]{
            {null, null},
            {new InvokeMetaData(null), null},
            {null, new ObjectMetaData()},
            {new InvokeMetaData(null), new ObjectMetaData()},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ArgumentMetaData#ArgumentMetaData(jp.ossc.nimbus.core.MetaData,jp.ossc.nimbus.core.ObjectMetaData)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*"},
          dataProvider = "ArgumentMetaData")
    public void testArgumentMetaData(MetaData parent, ObjectMetaData objData){
        ArgumentMetaData metaData = new ArgumentMetaData(parent, objData);
        assertEquals(metaData.getParent(), parent);
        assertEquals(metaData.getParentObjectMetaData(), objData);
    }

    @DataProvider(name = "SetType")
    public Object[][] dataSetType(){
        return new Object[][]{
            {null},
            {"java.lang.String"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ArgumentMetaData#setType(java.lang.String)",
                    "jp.ossc.nimbus.core.ArgumentMetaData#getType()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#ArgumentMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData,jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)"},
          dataProvider = "SetType")
    public void testSetType(String type){
        ArgumentMetaData metaData = new ArgumentMetaData(null, null);
        metaData.setType(type);
        assertEquals(metaData.getType(), type);
    }

    @DataProvider(name = "SetValueType")
    public Object[][] dataSetValueType(){
        return new Object[][]{
            {null},
            {"java.lang.String"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ArgumentMetaData#setValueType(java.lang.String)",
                    "jp.ossc.nimbus.core.ArgumentMetaData#getValueType()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#ArgumentMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData,jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)"},
          dataProvider = "SetValueType")
    public void testSetValueType(String type){
        ArgumentMetaData metaData = new ArgumentMetaData(null, null);
        metaData.setValueType(type);
        assertEquals(metaData.getValueType(), type);
    }

    @DataProvider(name = "SetNullValue")
    public Object[][] dataSetNullValue(){
        return new Object[][]{
            {true},
            {false}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ArgumentMetaData#setNullValue(boolean)",
                    "jp.ossc.nimbus.core.ArgumentMetaData#isNullValue()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#ArgumentMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData,jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)"},
          dataProvider = "SetNullValue")
    public void testSetNullValue(boolean isNullValue){
        ArgumentMetaData metaData = new ArgumentMetaData(null, null);
        metaData.setNullValue(isNullValue);
        assertEquals(metaData.isNullValue(), isNullValue);
    }

    @DataProvider(name = "SetValue")
    public Object[][] dataSetValue(){
        return new Object[][]{
            {null},
            {"hoge"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ArgumentMetaData#setValue(java.lang.Object)",
                    "jp.ossc.nimbus.core.ArgumentMetaData#getValue()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#ArgumentMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData,jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)"},
          dataProvider = "SetValue")
    public void testSetValue(Object value){
        ArgumentMetaData metaData = new ArgumentMetaData(null, null);
        metaData.setValue(value);
        assertEquals(metaData.getValue(), value);
    }

    @DataProvider(name = "ImportXML")
    public Object[][] dataImportXML() throws Exception{
        return new Object[][]{
            {createElement("<argument/>"), new Object[]{null, null, false, ""}},
            {createElement("<argument></argument>"), new Object[]{null, null, false, ""}},
            {createElement("<argument type=\"java.lang.String\"></argument>"), new Object[]{"java.lang.String", null, false, ""}},
            {createElement("<argument valueType=\"java.lang.String\"></argument>"), new Object[]{null, "java.lang.String", false, ""}},
            {createElement("<argument nullValue=\"true\"></argument>"), new Object[]{null, null, true, null}},
            {createElement("<argument nullValue=\"false\"></argument>"), new Object[]{null, null, false, ""}},
            {createElement("<argument>hoge</argument>"), new Object[]{null, null, false, "hoge"}},
            {createElement("<argument type=\"org.w3c.dom.Element\"><hoge/></argument>"), new Object[]{"org.w3c.dom.Element", null, false, createElement("<hoge/>")}},
            {createElement("<argument><service-ref>hoge</service-ref></argument>"), new Object[]{null, null, false, importXML(new ServiceRefMetaData(null), "<service-ref>hoge</service-ref>")}},
            {createElement("<argument><object code=\"java.lang.String\"/></argument>"), new Object[]{null, null, false, importXML(new ObjectMetaData(), "<object code=\"java.lang.String\"/>")}},
            {createElement("<argument><static-invoke code=\"java.util.Calendar\" name=\"getInstance\"/></argument>"), new Object[]{null, null, false, importXML(new StaticInvokeMetaData(null), "<static-invoke code=\"java.util.Calendar\" name=\"getInstance\"/>")}},
            {createElement("<argument><static-field-ref code=\"java.lang.Integer\" name=\"MAX_VALUE\"/></argument>"), new Object[]{null, null, false, importXML(new StaticFieldRefMetaData(null), "<static-field-ref code=\"java.lang.Integer\" name=\"MAX_VALUE\"/>")}},
            {createElement("<argument><unknown/></argument>"), new Object[]{null, null, false, ""}},
            {createElement("<argument unknown=\"hoge\"/>"), new Object[]{null, null, false, ""}},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ArgumentMetaData#importXML(org.w3c.dom.Element)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#ArgumentMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData,jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#getType\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#getValueType\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#isNullValue\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#getValue\\(\\)"},
          dataProvider = "ImportXML")
    public void testImportXML(
        Element element,
        Object[] assertValues
    ) throws DeploymentException{
        ArgumentMetaData metaData = new ArgumentMetaData(null, null);
        metaData.importXML(element);
        assertEquals(metaData.getType(), assertValues[0]);
        assertEquals(metaData.getValueType(), assertValues[1]);
        assertEquals(metaData.isNullValue(), ((Boolean)assertValues[2]).booleanValue());
        if(metaData.getValue() instanceof Element
                && assertValues[3] instanceof Element){
            assertEquals(((Element)metaData.getValue()).getTagName(), ((Element)assertValues[3]).getTagName());
        }else if(metaData.getValue() instanceof MetaData
                && assertValues[3] instanceof MetaData){
            assertMetaDataEquals((MetaData)metaData.getValue(), (MetaData)assertValues[3]);
        }else{
            assertEquals(metaData.getValue(), assertValues[3]);
        }
    }

    @DataProvider(name = "ImportXMLError")
    public Object[][] dataImportXMLError() throws Exception{
        return new Object[][]{
            {null},
            {createElement("<hoge/>")},
            {createElement("<argument type=\"\"></argument>")},
            {createElement("<argument valueType=\"\"></argument>")},
            {createElement("<argument type=\"org.w3c.dom.Element\"><hoge/><fuga/></argument>")},
            {createElement("<argument><service-ref>hoge</service-ref><service-ref>fuga</service-ref></argument>")},
            {createElement("<argument><object code=\"java.lang.String\"/><object code=\"java.lang.String\"/></argument>")},
            {createElement("<argument><static-invoke code=\"java.util.Calendar\" name=\"getInstance\"/><static-invoke code=\"java.util.Calendar\" name=\"getInstance\"/></argument>")},
            {createElement("<argument><static-field-ref code=\"java.lang.System\" name=\"out\"/><static-field-ref code=\"java.lang.System\" name=\"out\"/></argument>")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ArgumentMetaData#importXML(org.w3c.dom.Element)",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#ArgumentMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData,jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)"},
          expectedExceptions = {NullPointerException.class, DeploymentException.class},
          dataProvider = "ImportXMLError")
    public void testImportXMLError(Element element) throws DeploymentException{
        ArgumentMetaData metaData = new ArgumentMetaData(null, null);
        metaData.importXML(element);
    }

    @DataProvider(name = "ToString")
    public Object[][] dataToString(){
        return new Object[][]{
            {null, null, "{null}"},
            {"java.lang.String", null, "{type=java.lang.String,null}"},
            {null, "hoge", "{hoge}"},
            {"java.lang.String", "hoge", "{type=java.lang.String,hoge}"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ArgumentMetaData#toString()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#ArgumentMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData,jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#setType\\(java\\.lang\\.String\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#setValue\\(java\\.lang\\.Object\\)"},
          dataProvider = "ToString")
    public void testToString(String type, Object value, String expected){
        ArgumentMetaData metaData = new ArgumentMetaData(null, null);
        metaData.setType(type);
        metaData.setValue(value);
        assertTrue(metaData.toString().endsWith(expected));
    }

    @DataProvider(name = "ToXML")
    public Object[][] dataToXML() throws Exception{
        return new Object[][]{
            {createElement("<argument/>"), "<argument></argument>"},
            {createElement("<argument></argument>"), "<argument></argument>"},
            {createElement("<argument>hoge</argument>"), "<argument>hoge</argument>"},
            {createElement("<argument type=\"java.lang.String\">hoge</argument>"), "<argument type=\"java.lang.String\">hoge</argument>"},
            {createElement("<argument valueType=\"java.lang.String\">hoge</argument>"), "<argument valueType=\"java.lang.String\">hoge</argument>"},
            {createElement("<argument nullValue=\"true\"/>"), "<argument nullValue=\"true\"/>"},
            {createElement("<argument>hoge\nfuga</argument>"), "<argument>" + System.getProperty("line.separator") + "    hoge" + System.getProperty("line.separator") + "    fuga" + System.getProperty("line.separator") + "</argument>"},
            {createElement("<argument><service-ref>hoge</service-ref></argument>"), "<argument>" + System.getProperty("line.separator") + "    <service-ref>Nimbus#hoge</service-ref>" + System.getProperty("line.separator") + "</argument>"},
            {createElement("<argument><unknown/></argument>"), "<argument></argument>"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ArgumentMetaData#toXML(java.lang.StringBuilder)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#ArgumentMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData,jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)"},
          dataProvider = "ToXML")
    public void testToXML(Element element, String expected) throws DeploymentException{
        ArgumentMetaData metaData = new ArgumentMetaData(null, null);
        metaData.importXML(element);
        StringBuilder buf = new StringBuilder();
        metaData.toXML(buf);
        assertEquals(buf.toString(), expected);
    }
}
