/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.sequence;

import java.util.*;
import java.io.*;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.io.CSVReader;
//
/**
 * 񔭔ԃT[rXB<p>
 *
 * @author H.Nakano
 */
public class StringSequenceService extends ServiceBase
 implements Sequence, StringSequenceServiceMBean {
    
    private static final long serialVersionUID = 7784010436618007574L;
    
    private static final String C_SEMICORON = ";" ;  //$NON-NLS-1$
    
    //## o[ϐ錾 ##
    
    /** V[PXԍ  */
    protected List<SequenceVariable> sequenceNo;
    
    /** tH[}bg */
    protected String format;
    
    /** Jnԍ */
    protected String initialNumber = "";
    
    /** JntO */
    protected boolean initialFlag = true;
    
    /** ReLXgT[rX */
    protected ServiceName contextServiceName;
    
    /** it@C */
    protected String persistFile;
    
    /** ԖitO */
    protected boolean isPersistEveryTime;
    
    // StringSequenceServiceMBean JavaDoc
    public void setFormat(String format){
        this.format = format;
    }
    
    // StringSequenceServiceMBean JavaDoc
    public String getFormat(){
        return format;
    }
    
    // StringSequenceServiceMBean JavaDoc
    public void setContextServiceName(ServiceName name){
        contextServiceName = name;
    }
    
    // StringSequenceServiceMBean JavaDoc
    public ServiceName getContextServiceName(){
        return contextServiceName;
    }
    
    // StringSequenceServiceMBean JavaDoc
    public void setPersistFile(String file){
        persistFile = file;
    }
    
    // StringSequenceServiceMBean JavaDoc
    public String getPersistFile(){
        return persistFile;
    }
    
    // StringSequenceServiceMBean JavaDoc
    public void setPersistEveryTime(boolean isEveryTime){
        isPersistEveryTime = isEveryTime;
    }
    
    // StringSequenceServiceMBean JavaDoc
    public boolean isPersistEveryTime(){
        return isPersistEveryTime;
    }
    
    /**
     * T[rX̐sB<p>
     * 
     * @exception Exception T[rX̐Ɏsꍇ
     */
    public void createService() throws Exception{
        sequenceNo = new ArrayList<SequenceVariable>();
    }
    
    /**
     * T[rX̊JnsB<p>
     * 
     * @exception Exception T[rX̊JnɎsꍇ
     */
    public void startService() throws Exception{
        
        initialFlag = true;
        initialNumber = "";
        
        // format؂ŕ
        List<String> parser = CSVReader.toList(
            format,
            null,
            ';',
            '\\',
            "",
            null,
            true,
            false,
            true,
            false
        );
        
        List<String> persist = null;
        if(persistFile != null){
            final File file = new File(persistFile);
            if(file.exists()){
                final FileReader fr = new FileReader(file);
                final CSVReader csvr = new CSVReader(fr);
                csvr.setSeparator(';');
                try{
                    persist = csvr.readCSVLineList();
                    if(parser.size() != persist.size()){
                        persist = null;
                    }
                }finally{
                    try{
                        fr.close();
                    }catch(IOException e){}
                    try{
                        csvr.close();
                    }catch(IOException e){}
                }
            }else if(file.getParentFile() != null
                 && !file.getParentFile().exists()){
                file.mkdirs();
            }
        }
        
        // eCX^VOăXgɊi[
        for(int i = 0, max = parser.size(); i < max; i++){
            String formatItem = parser.get(i);
            String persistItem = null;
            if(persist != null){
                persistItem = persist.get(i);
            }
            SequenceVariable item = null;
            if(formatItem.startsWith(TimeSequenceVariable.FORMAT_KEY)){
                item = new TimeSequenceVariable(formatItem);
            }else if(formatItem.indexOf(SimpleSequenceVariable.DELIMITER) != -1){
                item = new SimpleSequenceVariable(formatItem, persistItem);
            }else if(formatItem.length() > 2
                && formatItem.charAt(0)
                     == ContextSequenceVariable.DELIMITER
                && formatItem.charAt(formatItem.length() - 1)
                     == ContextSequenceVariable.DELIMITER
            ){
                item = new ContextSequenceVariable(
                    formatItem,
                    contextServiceName
                );
            }else{
                item = new ConstSequenceVariable(formatItem);
            }
            sequenceNo.add(item);
        }
    }
    
    /**
     * T[rX̒~sB<p>
     * 
     * @exception Exception T[rX̒~Ɏsꍇ
     */
    public void stopService(){
        if(persistFile != null){
            persistSequence();
        }
        sequenceNo.clear();
    }
    
    /**
     * T[rX̔jsB<p>
     * 
     * @exception Exception T[rX̔jɎsꍇ
     */
    public void destroyService(){
        sequenceNo = null;
    }
    
    // SequenceJavaDoc
    public String increment(){
        StringBuilder buf = new StringBuilder();
        synchronized(this){
            // PڂCNgJn
            for(int i = sequenceNo.size() - 1; i >= 0; --i){
                //increment
                boolean isOverFlow = sequenceNo.get(i).increment();
                // I[o[t[Ȃꍇ͂オȂ
                if(!isOverFlow){
                    break;
                }
            }
            // Jgԕ𐶐
            for(int i = 0, imax = sequenceNo.size(); i < imax; i++){
                buf.append(sequenceNo.get(i).getCurrent());
            }
            if(initialFlag){
                //Jnԍۑ
                initialNumber = buf.toString();
                initialFlag = false;
            }
        }
        if(persistFile != null && isPersistEveryTime){
            persistSequence();
        }
        return buf.toString();
    }
    
    /**
     * ݔԂĂŌ̔ԍt@CɉiB<p>
     */
    protected void persistSequence(){
        FileWriter fw = null;
        try{
            fw = new FileWriter(persistFile, false);
            final StringBuilder buf = new StringBuilder();
            for(int i = 0, imax = sequenceNo.size(); i < imax; i++){
                buf.append(sequenceNo.get(i).getCurrent());
                if(i != imax - 1){
                    buf.append(C_SEMICORON);
                }
            }
            fw.write(buf.toString(), 0, buf.length());
        }catch(IOException e){
        }finally{
            if(fw != null){
                try{
                    fw.close();
                }catch(IOException e){
                }
            }
        }
    }
    
    // SequenceJavaDoc
    public void reset(){
        synchronized(this){
            // Jgԕ𐶐
            for(int i = 0, imax = sequenceNo.size(); i < imax; i++){
                sequenceNo.get(i).clear();
            }
            //JntOtrueɂB
            initialFlag = true;
            initialNumber = "";
        }
    }
    
    // SequenceJavaDoc
    public String getInitial(){
        return initialNumber;
    }
    
    // SequenceJavaDoc
    public String getCurrent(){
        StringBuilder buf = new StringBuilder();
        synchronized(this){
            for(int i = 0, imax = sequenceNo.size(); i < imax; i++){
                buf.append(sequenceNo.get(i).getCurrent());
            }
        }
        return buf.toString();
    }
}
