/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.net;

import java.util.*;
import java.io.IOException;
import java.lang.reflect.Method;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.UndeclaredThrowableException;
import java.net.Socket;
import java.net.InetAddress;
import java.net.SocketAddress;
import java.net.SocketException;
import java.nio.channels.ServerSocketChannel;
import javax.net.ssl.SSLServerSocket;

import jp.ossc.nimbus.beans.*;

/**
 * SSLT[o\Pbgbp[B<p>
 * ̃T[o\Pbgaccept()ꂽ\PbǵA{@link #setSocketProperty(String, Object)}ŁA\ߐݒ肳ꂽvpeBݒ肳B<br>
 *
 * @author M.Takata
 * @see ServerSocketFactory
 */
public class SSLServerSocketWrapper extends SSLServerSocket{
    
    protected SSLServerSocket serverSocket;
    
    protected Map<Property, Object> socketProperties;
    
    public SSLServerSocketWrapper(SSLServerSocket serverSock) throws IOException{
        super();
        serverSocket = serverSock;
    }
    
    @Override
    public void bind(SocketAddress endpoint) throws IOException{
        serverSocket.bind(endpoint);
    }
    
    @Override
    public void bind(SocketAddress endpoint, int backlog) throws IOException{
        serverSocket.bind(endpoint, backlog);
    }
    
    @Override
    public InetAddress getInetAddress(){
        return serverSocket.getInetAddress();
    }
    
    @Override
    public int getLocalPort(){
        return serverSocket.getLocalPort();
    }
    
    @Override
    public SocketAddress getLocalSocketAddress(){
        return serverSocket.getLocalSocketAddress();
    }
    
    @Override
    public void close() throws IOException{
        serverSocket.close();
    }
    
    @Override
    public ServerSocketChannel getChannel(){
        return serverSocket.getChannel();
    }
    
    @Override
    public boolean isBound(){
        return serverSocket.isBound();
    }
    
    @Override
    public boolean isClosed(){
        return serverSocket.isClosed();
    }
    
    @Override
    public void setSoTimeout(int timeout) throws SocketException{
        serverSocket.setSoTimeout(timeout);
    }
    
    @Override
    public int getSoTimeout() throws IOException{
        return serverSocket.getSoTimeout();
    }
    
    @Override
    public void setReuseAddress(boolean on) throws SocketException{
        serverSocket.setReuseAddress(on);
    }
    
    @Override
    public boolean getReuseAddress() throws SocketException{
        return serverSocket.getReuseAddress();
    }
    
    @Override
    public void setReceiveBufferSize(int size) throws SocketException{
        serverSocket.setReceiveBufferSize(size);
    }
    
    @Override
    public int getReceiveBufferSize() throws SocketException{
        return serverSocket.getReceiveBufferSize();
    }
    
    @Override
    public void setPerformancePreferences(
        int connectionTime,
        int latency,
        int bandwidth
    ){
        try{
            Method method = serverSocket.getClass().getMethod(
                "setPerformancePreferences",
                new Class[]{Integer.TYPE, Integer.TYPE, Integer.TYPE}
            );
            method.invoke(
                serverSocket,
                new Object[]{
                    new Integer(connectionTime),
                    new Integer(latency),
                    new Integer(bandwidth)
                }
            );
        }catch(NoSuchMethodException e){
            throw new UnsupportedOperationException(e.toString());
        }catch(IllegalAccessException e){
            throw new UnsupportedOperationException(e.toString());
        }catch(InvocationTargetException e){
            Throwable target = e.getTargetException();
            if(target instanceof RuntimeException){
                throw (RuntimeException)target;
            }else if(target instanceof Error){
                throw (Error)target;
            }else{
                throw new UndeclaredThrowableException(target);
            }
        }
    }
    
    @Override
    public String toString(){
        return serverSocket.toString();
    }
    
    @Override
    public Socket accept() throws IOException{
        return applySocketProperties(serverSocket.accept());
    }
    
    @Override
    public String[] getEnabledCipherSuites(){
        return serverSocket.getEnabledCipherSuites();
    }
    
    @Override
    public void setEnabledCipherSuites(String[] suites){
        serverSocket.setEnabledCipherSuites(suites);
    }
    
    @Override
    public String[] getSupportedCipherSuites(){
        return serverSocket.getSupportedCipherSuites();
    }
    
    @Override
    public String[] getSupportedProtocols(){
        return serverSocket.getSupportedProtocols();
    }
    
    @Override
    public String[] getEnabledProtocols(){
        return serverSocket.getEnabledProtocols();
    }
    
    @Override
    public void setEnabledProtocols(String[] protocols){
        serverSocket.setEnabledProtocols(protocols);
    }
    
    @Override
    public void setNeedClientAuth(boolean need){
        serverSocket.setNeedClientAuth(need);
    }
    
    @Override
    public boolean getNeedClientAuth(){
        return serverSocket.getNeedClientAuth();
    }
    
    @Override
    public void setWantClientAuth(boolean want){
        serverSocket.setWantClientAuth(want);
    }
    
    @Override
    public boolean getWantClientAuth(){
        return serverSocket.getWantClientAuth();
    }
    
    @Override
    public void setUseClientMode(boolean mode){
        serverSocket.setUseClientMode(mode);
    }
    
    @Override
    public boolean getUseClientMode(){
        return serverSocket.getUseClientMode();
    }
    
    @Override
    public void setEnableSessionCreation(boolean flag){
        serverSocket.setEnableSessionCreation(flag);
    }
    
    @Override
    public boolean getEnableSessionCreation(){
        return serverSocket.getEnableSessionCreation();
    }
    
    /**
     * {@link java.net.Socket}ɃvpeBݒ肷B<p>
     *
     * @param props vpeB}bv
     */
    public void setSocketProperties(Map<String, Object> props){
        if(props == null || props.size() == 0){
            if(socketProperties != null){
                socketProperties = null;
            }
            return;
        }
        for(Map.Entry<String, Object> entry : props.entrySet()){
            setSocketProperty(entry.getKey(), entry.getValue());
        }
    }
    
    /**
     * {@link Socket}ɃvpeBݒ肷B<p>
     *
     * @param name vpeB
     * @param value l
     */
    public void setSocketProperty(String name, Object value){
        if(socketProperties == null){
            socketProperties = new LinkedHashMap<Property, Object>();
        }
        final Property prop = PropertyFactory.createProperty(name);
        socketProperties.put(prop, value);
    }
    
    /**
     * {@link Socket}̃vpeB擾B<p>
     *
     * @param name vpeB
     * @return l
     */
    public Object getSocketProperty(String name){
        if(socketProperties == null){
            return null;
        }
        for(Map.Entry<Property, Object> entry : socketProperties.entrySet()){
            if(entry.getKey().getPropertyName().equals(name)){
                return entry.getValue();
            }
        }
        return null;
    }
    
    protected Socket applySocketProperties(
        Socket socket
    ) throws IOException{
        try{
            if(socketProperties != null && socketProperties.size() != 0){
                for(Map.Entry<Property, Object> entry : socketProperties.entrySet()){
                    entry.getKey().setProperty(socket, entry.getValue());
                }
            }
        }catch(InvocationTargetException e){
            Throwable target = e.getTargetException();
            if(target instanceof IOException){
                throw (IOException)target;
            }else if(target instanceof RuntimeException){
                throw (RuntimeException)target;
            }else if(target instanceof Error){
                throw (Error)target;
            }else{
                throw new UndeclaredThrowableException(target);
            }
        }catch(NoSuchPropertyException e){
            throw new UndeclaredThrowableException(e);
        }
        return socket;
    }
}