/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.io;

import java.io.*;
import java.math.*;
import java.util.*;
import java.nio.*;

/**
 * CSV`ReaderNXB<p>
 * <pre>
 * import java.io.*;
 * import jp.ossc.nimbus.io.CSVReader;
 *
 * FileInputStream fis = new FileInputStream("sample.csv");
 * InputStreamReader isr = new InputStreamReader(fis, "UTF-8");
 * CSVReader reader = new CSVReader(isr);
 * try{
 *     String[] csv = null;
 *     while((csv = reader.readCSVLine()) != null){
 *           :
 *     }
 * }finally{
 *     reader.close();
 * }
 * </pre>
 * 
 * @author M.Takata
 */
public class CSVReader extends LineNumberReader{
    
    /**
     * ftHg̃Zp[^B<p>
     */
    public static final char DEFAULT_SEPARATOR = ',';
    
    /**
     * ftHg̃Zp[^̃GXP[vB<p>
     * GXP[vGXP[vꍇ́AGXP[vd˂B<br>
     */
    public static final char DEFAULT_SEPARATOR_ESCAPE = '\\';
    
    /**
     * ftHg̈͂ݕB<p>
     * ͂ݕGXP[vꍇ́A͂ݕd˂B<br>
     */
    public static final char DEFAULT_ENCLOSURE = '"';
    
    /**
     * ftHg̉sB<p>
     */
    public static final String LINE_SEPARATOR
         = System.getProperty("line.separator");
    
    protected char separator = DEFAULT_SEPARATOR;
    protected char separatorEscape = DEFAULT_SEPARATOR_ESCAPE;
    protected char enclosure = DEFAULT_ENCLOSURE;
    protected boolean isIgnoreEmptyLine;
    protected boolean isIgnoreLineEndSeparator;
    protected boolean isEnclosed;
    protected boolean isTrim;
    protected String nullValue;
    protected String commentPrefix;
    protected boolean isUnescapeLineSeparatorInEnclosure;
    
    protected CSVIterator iterator;
    
    protected ReaderWrapper readerWrapper;
    
    /**
     * ftHg̓ǂݍ݃obt@TCYڑ̃CX^X𐶐B<p>
     */
    public CSVReader(){
        super(new ReaderWrapper());
        readerWrapper = (ReaderWrapper)lock;
    }
    
    /**
     * ftHg̓ǂݍ݃obt@TCYCX^X𐶐B<p>
     *
     * @param reader ǂݍ݌Reader
     */
    public CSVReader(Reader reader){
        super(new ReaderWrapper(reader));
        readerWrapper = (ReaderWrapper)lock;
    }
    
    /**
     * w肳ꂽǂݍ݃obt@TCYڑ̃CX^X𐶐B<p>
     *
     * @param size ǂݍ݃obt@TCY
     */
    public CSVReader(int size){
        super(new ReaderWrapper(), size);
        readerWrapper = (ReaderWrapper)lock;
    }
    
    /**
     * w肳ꂽǂݍ݃obt@TCYCX^X𐶐B<p>
     *
     * @param reader ǂݍ݌Reader
     * @param size ǂݍ݃obt@TCY
     */
    public CSVReader(Reader reader, int size){
        super(new ReaderWrapper(reader), size);
        readerWrapper = (ReaderWrapper)lock;
    }
    
    /**
     * Readerݒ肷B<p>
     *
     * @param reader Reader
     * @exception IOException Readerݒ肳Ăꍇ
     */
    public void setReader(Reader reader) throws IOException{
        readerWrapper.setReader(reader);
    }
    
    /**
     * Zp[^ݒ肷B<p>
     *
     * @param separator Zp[^
     */
    public void setSeparator(char separator){
        this.separator = separator;
    }
    
    /**
     * Zp[^擾B<p>
     *
     * @return Zp[^
     */
    public char getSeparator(){
         return separator;
    }
    
    /**
     * Zp[^̃GXP[vݒ肷B<p>
     *
     * @param escape GXP[v
     */
    public void setSeparatorEscape(char escape){
        separatorEscape = escape;
    }
    
    /**
     * Zp[^̃GXP[v擾B<p>
     *
     * @return GXP[v
     */
    public char getSeparatorEscape(){
         return separatorEscape;
    }
    
    /**
     * ͂ݕݒ肷B<p>
     *
     * @param enclosure ͂ݕ
     */
    public void setEnclosure(char enclosure){
        this.enclosure = enclosure;
    }
    
    /**
     * ͂ݕ擾B<p>
     *
     * @return ͂ݕ
     */
    public char getEnclosure(){
         return enclosure;
    }
    
    /**
     * s𖳎邩ǂݒ肷B<p>
     * s𖳎悤ɐݒ肵ꍇAs͍sƂĂJEgȂB<br>
     * ftHǵAfalseŖȂB<br>
     *
     * @param isIgnore s𖳎ꍇtrue
     */
    public void setIgnoreEmptyLine(boolean isIgnore){
        isIgnoreEmptyLine = isIgnore;
    }
    
    /**
     * s𖳎邩ǂ𔻒肷B<p>
     *
     * @return truȅꍇAs𖳎
     */
    public boolean isIgnoreEmptyLine(){
         return isIgnoreEmptyLine;
    }
    
    /**
     * s̍Ō̃Zp[^𖳎邩ǂݒ肷B<p>
     * ftHǵAfalseŖȂB<br>
     *
     * @param isIgnore s̍Ō̃Zp[^𖳎ꍇtrue
     */
    public void setIgnoreLineEndSeparator(boolean isIgnore){
        isIgnoreLineEndSeparator = isIgnore;
    }
    
    /**
     * s̍Ō̃Zp[^𖳎邩ǂ𔻒肷B<p>
     *
     * @return truȅꍇAs̍Ō̃Zp[^𖳎
     */
    public boolean isIgnoreLineEndSeparator(){
         return isIgnoreLineEndSeparator;
    }
    
    /**
     * ͂ݕLǂݒ肷B<p>
     * ftHǵAfalseň͂ݕ͖ȂB<br>
     *
     * @param isEnclosed ͂ݕLȏꍇtrue
     */
    public void setEnclosed(boolean isEnclosed){
        this.isEnclosed = isEnclosed;
    }
    
    /**
     * ͂ݕLǂ𔻒肷B<p>
     *
     * @return truȅꍇA͂ݕL
     */
    public boolean isEnclosed(){
         return isEnclosed;
    }
    
    /**
     * g邩ǂݒ肷B<p>
     * ftHǵAfalseŃgȂB<br>
     *
     * @param isTrim gꍇtrue
     */
    public void setTrim(boolean isTrim){
        this.isTrim = isTrim;
    }
    
    /**
     * g邩ǂ𔻒肷B<p>
     *
     * @return truȅꍇAg
     */
    public boolean isTrim(){
         return isTrim;
    }
    
    /**
     * CSVvfǂݍ񂾏ꍇɁAnulll镶ݒ肷B<p>
     *
     * @param value 
     */
    public void setNullValue(String value){
        nullValue = value;
    }
    
    /**
     * CSVvfǂݍ񂾏ꍇɁAnulll镶擾B<p>
     *
     * @return 
     */
    public String getNullValue(){
        return nullValue;
    }
    
    /**
     * Rgs̑Ouݒ肷B<p>
     *
     * @param value Rgs̑Ou
     */
    public void setCommentPrefix(String value){
        commentPrefix = value;
    }
    
    /**
     * Rgs̑Ou擾B<p>
     *
     * @return Rgs̑Ou
     */
    public String getCommentPrefix(){
        return commentPrefix;
    }
    
    /**
     * ͂ݕň͂܂ꂽCSVvf̏ꍇɁAGXP[vꂽsAGXP[v邩ǂݒ肷B<p>
     * ftHǵAfalseŃAGXP[vȂB<br>
     * 
     * @param isUnescape AGXP[vꍇtrue
     */
    public void setUnescapeLineSeparatorInEnclosure(boolean isUnescape){
        isUnescapeLineSeparatorInEnclosure = isUnescape;
    }
    
    /**
     * ͂ݕň͂܂ꂽCSVvf̏ꍇɁAGXP[vꂽsAGXP[v邩ǂ𔻒肷B<p>
     * 
     * @return truȅꍇAAGXP[v
     */
    public boolean isUnescapeLineSeparatorInEnclosure(){
        return isUnescapeLineSeparatorInEnclosure;
    }
    
    /**
     * w肳ꂽsXLbvB<p>
     *
     * @param line XLbvs
     * @return XLbvꂽs
     * @exception IOException o̓G[ꍇ
     */
    public long skipLine(long line) throws IOException{
        int result = 0;
        for(result = 0; result < line; result++){
            if(super.readLine() == null){
                break;
            }
        }
        return result;
    }
    
    /**
     * w肳ꂽCSVsXLbvB<p>
     * {@link #isIgnoreEmptyLine()}truȅꍇ́As̓XLbvs̃JEg珜B<br>
     * CSVsŃJEg邽߁A͂ݕň͂񂾒ɉsĂA1sƂăJEgB<br>
     *
     * @param line XLbvs
     * @return XLbvꂽs
     * @exception IOException o̓G[ꍇ
     */
    public long skipCSVLine(long line) throws IOException{
        List<String> csv = null;
        int result = 0;
        for(result = 0; result < line; result++){
            csv = readCSVLineList(csv);
            if(csv == null){
                break;
            }
        }
        return result;
    }
    
    /**
     * CSVs1sǂݍށB<p>
     *
     * @return CSVvf̕z
     * @exception IOException o̓G[ꍇ
     */
    public String[] readCSVLine() throws IOException{
        final List<String> csv = readCSVLineList();
        return csv == null ? null
             : (String[])csv.toArray(new String[csv.size()]);
    }
    
    /**
     * CSVs1sǂݍށB<p>
     *
     * @return CSVvf̕񃊃Xg
     * @exception IOException o̓G[ꍇ
     */
    public List<String> readCSVLineList() throws IOException{
        return readCSVLineList(null);
    }
    
    /**
     * CSVs1sǂݍށB<p>
     * CSVvf̕i[郊Xgėp邽߂̃\bhłB<br>
     *
     * @param csv CSVvf̕i[郊Xg
     * @return CSVvf̕񃊃Xg
     * @exception IOException o̓G[ꍇ
     */
    public List<String> readCSVLineList(List<String> csv) throws IOException{
        return toList(
            this,
            csv,
            separator,
            separatorEscape,
            isEnclosed,
            enclosure,
            nullValue,
            commentPrefix,
            isIgnoreEmptyLine,
            isIgnoreLineEndSeparator,
            isTrim,
            isUnescapeLineSeparatorInEnclosure
        );
    }
    
    @Override
    public String readLine() throws IOException{
        if(readerWrapper.getReader() instanceof BufferedReader){
            return ((BufferedReader)readerWrapper.getReader()).readLine();
        }else{
            return super.readLine();
        }
    }
    
    private static List<String> toList(
        BufferedReader reader,
        List<String> csv,
        char separator,
        char separatorEscape,
        boolean isEnclosed,
        char enclosure,
        String nullValue,
        String commentPrefix,
        boolean isIgnoreEmptyLine,
        boolean isIgnoreLineEndSeparator,
        boolean isTrim,
        boolean isUnescapeLineSeparatorInEnclosure
    ) throws IOException{
        String line = null;
        do{
            line = reader.readLine();
            if(line == null){
                if(csv != null){
                    csv.clear();
                }
                return null;
            }
            if((isIgnoreEmptyLine && line.length() == 0)
                    || (commentPrefix != null && line.startsWith(commentPrefix))
            ){
                line = null;
                if(reader instanceof LineNumberReader){
                    LineNumberReader lnr = (LineNumberReader)reader;
                    lnr.setLineNumber(lnr.getLineNumber() - 1);
                }
            }
        }while(line == null);
        if(csv == null){
            csv = new ArrayList<String>();
        }else{
            csv.clear();
        }
        if(line.length() == 0){
            return csv;
        }
        final StringBuilder buf = new StringBuilder();
        boolean inEnclosure = false;
        boolean isEncElement = false;
        do{
            if(inEnclosure){
                line = reader.readLine();
                if(line == null){
                    break;
                }
                if(reader instanceof LineNumberReader){
                    LineNumberReader lnr = (LineNumberReader)reader;
                    lnr.setLineNumber(lnr.getLineNumber() - 1);
                }
                buf.append(LINE_SEPARATOR);
            }
            char c = 0;
            for(int i = 0, imax = line.length(); i < imax; i++){
                c = line.charAt(i);
                if(c == enclosure){
                    if(isEnclosed){
                        if(inEnclosure
                            && imax - 1 != i
                            && line.charAt(i + 1) == enclosure
                        ){
                            buf.append(enclosure);
                            i++;
                        }else{
                            if((i > 2
                                && line.charAt(i - 1) != separator
                                && imax - 1 != i
                                && line.charAt(i + 1) != separator)
                                || (!inEnclosure && (imax - 1 == i || buf.length() != 0))
                            ){
                                buf.append(c);
                            }else{
                                inEnclosure = !inEnclosure;
                                if(!inEnclosure){
                                    isEncElement = true;
                                }
                            }
                        }
                    }else{
                        buf.append(c);
                    }
                }else if(c == separator){
                    if(inEnclosure){
                        buf.append(c);
                    }else{
                        if(!isEncElement && isTrim){
                            trim(buf);
                        }
                        final String element = buf.toString();
                        if(nullValue == null){
                            csv.add(element);
                        }else{
                            if(nullValue.equals(element)){
                                csv.add(null);
                            }else{
                                csv.add(element);
                            }
                        }
                        buf.setLength(0);
                        isEncElement = false;
                    }
                }else if(c == separatorEscape){
                    if(imax - 1 != i){
                        final char nextChar = line.charAt(i + 1);
                        if(!inEnclosure){
                            if(nextChar == separator
                                 || nextChar == separatorEscape
                                 || nextChar == enclosure){
                                buf.append(nextChar);
                                i++;
                            }else if(nextChar == 'r'){
                                buf.append('\r');
                                i++;
                            }else if(nextChar == 'n'){
                                buf.append('\n');
                                i++;
                            }else{
                                buf.append(c);
                            }
                        }else if(isUnescapeLineSeparatorInEnclosure){
                            if(nextChar == 'r'){
                                buf.append('\r');
                                i++;
                            }else if(nextChar == 'n'){
                                buf.append('\n');
                                i++;
                            }else{
                                buf.append(c);
                            }
                        }else{
                            buf.append(c);
                        }
                    }else{
                        buf.append(c);
                    }
                }else{
                    buf.append(c);
                }
            }
            if(!inEnclosure
                 && (c != separator || !isIgnoreLineEndSeparator
                     || buf.length() != 0)){
                final String element = buf.toString();
                if(nullValue == null){
                    csv.add(element);
                }else{
                    if(nullValue.equals(element)){
                        csv.add(null);
                    }else{
                        csv.add(element);
                    }
                }
            }
        }while(inEnclosure);
        return csv;
    }
    
    /**
     * w肳ꂽgB<p>
     * ǵAw肳ꂽ̑Ő󔒕i{@link Character#isWhitespace(char)}trueƂȂ镶j폜B
     * 
     * @param buf 
     * @return gꂽ
     */
    protected static StringBuilder trim(StringBuilder buf){
        int index = 0;
        for(int i = 0, max = buf.length(); i < max; i++){
            final char c = buf.charAt(i);
            if(!Character.isWhitespace(c)){
                index = i;
                break;
            }
        }
        if(index != 0){
            buf.delete(0, index);
        }
        index = buf.length();
        for(int i = buf.length(); --i >= 0;){
            final char c = buf.charAt(i);
            if(!Character.isWhitespace(c)){
                index = i;
                break;
            }
        }
        if(index != buf.length() - 1){
            buf.delete(index, buf.length());
        }
        return buf;
    }
    
    public static String[] toArray(
        String str,
        char separator,
        char separatorEscape,
        char enclosure,
        String nullValue,
        String commentPrefix,
        boolean isIgnoreEmptyLine,
        boolean isIgnoreLineEndSeparator,
        boolean isTrim,
        boolean isUnescapeLineSeparatorInEnclosure
    ){
        if(str == null || str.length() == 0){
            return new String[0];
        }
        final StringReader sr = new StringReader(str);
        final BufferedReader br = new BufferedReader(sr);
        List<String> list = null;
        try{
            list = toList(
                br,
                null,
                separator,
                separatorEscape,
                true,
                enclosure,
                nullValue,
                commentPrefix,
                isIgnoreEmptyLine,
                isIgnoreLineEndSeparator,
                isTrim,
                isUnescapeLineSeparatorInEnclosure
            );
        }catch(IOException e){
            // NȂ͂
            return new String[0];
        }
        return list.toArray(new String[list.size()]);
    }
    
    public static String[] toArray(
        String str,
        char separator,
        char separatorEscape,
        String nullValue,
        String commentPrefix,
        boolean isIgnoreEmptyLine,
        boolean isIgnoreLineEndSeparator,
        boolean isTrim,
        boolean isUnescapeLineSeparatorInEnclosure
    ){
        if(str == null || str.length() == 0){
            return new String[0];
        }
        final StringReader sr = new StringReader(str);
        final BufferedReader br = new BufferedReader(sr);
        List<String> list = null;
        try{
            list = toList(
                br,
                null,
                separator,
                separatorEscape,
                false,
                '"',
                nullValue,
                commentPrefix,
                isIgnoreEmptyLine,
                isIgnoreLineEndSeparator,
                isTrim,
                isUnescapeLineSeparatorInEnclosure
            );
        }catch(IOException e){
            // NȂ͂
            return new String[0];
        }
        return list.toArray(new String[list.size()]);
    }
    
    public static List<String> toList(
        String str,
        List<String> result,
        char separator,
        char separatorEscape,
        char enclosure,
        String nullValue,
        String commentPrefix,
        boolean isIgnoreEmptyLine,
        boolean isIgnoreLineEndSeparator,
        boolean isTrim,
        boolean isUnescapeLineSeparatorInEnclosure
    ){
        if(result == null){
            result = new ArrayList<String>();
        }
        if(str == null || str.length() == 0){
            result.clear();
            return result;
        }
        final StringReader sr = new StringReader(str);
        final BufferedReader br = new BufferedReader(sr);
        try{
            result = toList(
                br,
                result,
                separator,
                separatorEscape,
                true,
                enclosure,
                nullValue,
                commentPrefix,
                isIgnoreEmptyLine,
                isIgnoreLineEndSeparator,
                isTrim,
                isUnescapeLineSeparatorInEnclosure
            );
        }catch(IOException e){
            // NȂ͂
            return result;
        }
        return result;
    }
    
    public static List<String> toList(
        String str,
        List<String> result,
        char separator,
        char separatorEscape,
        String nullValue,
        String commentPrefix,
        boolean isIgnoreEmptyLine,
        boolean isIgnoreLineEndSeparator,
        boolean isTrim,
        boolean isUnescapeLineSeparatorInEnclosure
    ){
        if(result == null){
            result = new ArrayList<String>();
        }
        if(str == null || str.length() == 0){
            result.clear();
            return result;
        }
        final StringReader sr = new StringReader(str);
        final BufferedReader br = new BufferedReader(sr);
        try{
            result = toList(
                br,
                result,
                separator,
                separatorEscape,
                false,
                '"',
                nullValue,
                commentPrefix,
                isIgnoreEmptyLine,
                isIgnoreLineEndSeparator,
                isTrim,
                isUnescapeLineSeparatorInEnclosure
            );
        }catch(IOException e){
            // NȂ͂
            return result;
        }
        return result;
    }
    
    /**
     * {@link CSVReader.CSVElements}̌JԂ擾B<p>
     *
     * @return CSVElementšJԂ
     */
    public CSVIterator iterator(){
        if(iterator == null){
            iterator = new CSVIterator();
        }
        return iterator;
    }
    
    /**
     * {@link CSVReader.CSVElements}̌JԂB<p>
     *
     * @author M.Takata
     */
    public class CSVIterator{
        private boolean hasNext = false;
        private CSVElements elements = new CSVElements();
        
        private CSVIterator(){}
        
        /**
         * CSVvf邩ǂ𔻒肷B<p>
         *
         * @return CSVvfꍇtrue
         * @exception IOException ǂݍ݂Ɏsꍇ
         */
        public boolean hasNext() throws IOException{
            if(hasNext){
                return hasNext;
            }
            List<String> result = readCSVLineList(elements);
            hasNext = result != null;
            return hasNext;
        }
        
        /**
         * CSVvf擾B<p>
         *
         * @return CSVvfBCSVvfȂꍇnull
         * @exception IOException ǂݍ݂Ɏsꍇ
         * @see #nextElements()
         */
        public Object next() throws IOException{
            return nextElements();
        }
        
        /**
         * CSVvf擾B<p>
         * Ŏ擾{@link CSVReader.CSVElements}́AėpB<br>
         *
         * @return CSVvfBCSVvfȂꍇnull
         * @exception IOException ǂݍ݂Ɏsꍇ
         */
        public CSVElements nextElements() throws IOException{
            if(!hasNext){
                if(!hasNext()){
                    return null;
                }
            }
            hasNext = false;
            return elements;
        }
    }
    
    /**
     * ڑ̕𐶐B<p>
     *
     * @return ڑ̕
     */
    public CSVReader cloneReader(){
        return cloneReader(new CSVReader());
    }
    
    /**
     * ڑ̕𐶐B<p>
     *
     * @param clone ڑ̃CX^X
     * @return ڑ̕
     */
    protected CSVReader cloneReader(CSVReader clone){
        clone.separator = separator;
        clone.separatorEscape = separatorEscape;
        clone.enclosure = enclosure;
        clone.isIgnoreEmptyLine = isIgnoreEmptyLine;
        clone.isIgnoreLineEndSeparator = isIgnoreLineEndSeparator;
        clone.isEnclosed = isEnclosed;
        clone.nullValue = nullValue;
        clone.commentPrefix = commentPrefix;
        return clone;
    }
    
    /**
     * CSV`f[^1s\CSVvfB<p>
     * 
     * @author M.Takata
     */
    public class CSVElements extends ArrayList<String>{
        
        private static final long serialVersionUID = 6079322185163530516L;
        
        private boolean wasNull;
        
        private CSVElements(){}
        
        /**
         * CSVvfNAB<p>
         */
        @Override
        public void clear(){
            wasNull = false;
            super.clear();
        }
        
        /**
         * 擾lnullǂ𔻒肷B<p>
         * {@link #getInt(int)}Ȃǂ́AlngetterŒl擾ꍇAlnull󕶎ꍇɁA0ԂB̎Al0̂null܂͋󕶎̂𔻒f̂ɎgpB<br>
         *
         * @return 擾lnullꍇtrue
         */
        public boolean wasNull(){
            return wasNull;
        }
        
        /**
         * w肳ꂽCfbNX̗vf擾B<p>
         *
         * @param index CfbNX
         * @return w肳ꂽCfbNX̗vf
         */
        @Override
        public String get(int index){
            String obj = super.get(index);
            wasNull = obj == null;
            return obj;
        }
        
        /**
         * w肳ꂽCfbNX̗vf擾B<p>
         *
         * @param index CfbNX
         * @return w肳ꂽCfbNX̗vf
         */
        public String getString(int index){
            String str = (String)get(index);
            wasNull = str == null;
            return str;
        }
        
        /**
         * w肳ꂽCfbNX̗vfoCg擾B<p>
         *
         * @param index CfbNX
         * @return w肳ꂽCfbNX̗vfoCg
         * @exception NumberFormatException vfoCgłȂꍇ
         */
        public byte getByte(int index) throws NumberFormatException{
            return getByte(index, 10);
        }
        
        /**
         * w肳ꂽCfbNX̗vfoCg擾B<p>
         * w肳ꂽvfnull܂͋󕶎̏ꍇ́A0ԂA{@link #wasNull()}trueԂB<br>
         *
         * @param index CfbNX
         * @param radix 
         * @return w肳ꂽCfbNX̗vfoCg
         * @exception NumberFormatException vfoCgłȂꍇ
         */
        public byte getByte(int index, int radix) throws NumberFormatException{
            final String str = getString(index);
            if(str == null || str.length() == 0){
                wasNull = true;
                return (byte)0;
            }
            return Byte.parseByte(str, radix);
        }
        
        /**
         * w肳ꂽCfbNX̗vfl擾B<p>
         * w肳ꂽvfnull܂͋󕶎̏ꍇ́A0ԂA{@link #wasNull()}trueԂB<br>
         *
         * @param index CfbNX
         * @return w肳ꂽCfbNX̗vfl
         * @exception NumberFormatException vflłȂꍇ
         */
        public short getShort(int index) throws NumberFormatException{
            final String str = getString(index);
            if(str == null || str.length() == 0){
                wasNull = true;
                return (short)0;
            }
            return Short.parseShort(str);
        }
        
        /**
         * w肳ꂽCfbNX̗vf擾B<p>
         * w肳ꂽvfnull܂͋󕶎̏ꍇ́A0ԂA{@link #wasNull()}trueԂB<br>
         * ܂Aw肳ꂽvfA琬ꍇ́A1ڂԂB<br>
         *
         * @param index CfbNX
         * @return w肳ꂽCfbNX̗vf
         */
        public char getChar(int index){
            final String str = getString(index);
            if(str == null || str.length() == 0){
                wasNull = true;
                return (char)0;
            }
            return str.charAt(0);
        }
        
        /**
         * w肳ꂽCfbNX̗vfl擾B<p>
         * w肳ꂽvfnull܂͋󕶎̏ꍇ́A0ԂA{@link #wasNull()}trueԂB<br>
         *
         * @param index CfbNX
         * @return w肳ꂽCfbNX̗vfl
         * @exception NumberFormatException vflłȂꍇ
         */
        public int getInt(int index) throws NumberFormatException{
            final String str = getString(index);
            if(str == null || str.length() == 0){
                wasNull = true;
                return (int)0;
            }
            return Integer.parseInt(str);
        }
        
        /**
         * w肳ꂽCfbNX̗vfl擾B<p>
         * w肳ꂽvfnull܂͋󕶎̏ꍇ́A0ԂA{@link #wasNull()}trueԂB<br>
         *
         * @param index CfbNX
         * @return w肳ꂽCfbNX̗vfl
         * @exception NumberFormatException vflłȂꍇ
         */
        public long getLong(int index) throws NumberFormatException{
            final String str = getString(index);
            if(str == null || str.length() == 0){
                wasNull = true;
                return 0l;
            }
            return Long.parseLong(str);
        }
        
        /**
         * w肳ꂽCfbNX̗vfl擾B<p>
         * w肳ꂽvfnull܂͋󕶎̏ꍇ́A0ԂA{@link #wasNull()}trueԂB<br>
         *
         * @param index CfbNX
         * @return w肳ꂽCfbNX̗vfl
         * @exception NumberFormatException vflłȂꍇ
         */
        public float getFloat(int index) throws NumberFormatException{
            final String str = getString(index);
            if(str == null || str.length() == 0){
                wasNull = true;
                return 0.0f;
            }
            return Float.parseFloat(str);
        }
        
        /**
         * w肳ꂽCfbNX̗vfl擾B<p>
         * w肳ꂽvfnull܂͋󕶎̏ꍇ́A0ԂA{@link #wasNull()}trueԂB<br>
         *
         * @param index CfbNX
         * @return w肳ꂽCfbNX̗vfl
         * @exception NumberFormatException vflłȂꍇ
         */
        public double getDouble(int index) throws NumberFormatException{
            final String str = getString(index);
            if(str == null || str.length() == 0){
                wasNull = true;
                return 0.0d;
            }
            return Double.parseDouble(str);
        }
        
        /**
         * w肳ꂽCfbNX̗vftO擾B<p>
         * w肳ꂽvfnull܂͋󕶎̏ꍇ́AfalseԂA{@link #wasNull()}trueԂB<br>
         *
         * @param index CfbNX
         * @return w肳ꂽCfbNX̗vftO
         */
        public boolean getBoolean(int index){
            final String str = getString(index);
            if(str == null || str.length() == 0){
                wasNull = true;
                return false;
            }
            return Boolean.valueOf(str).booleanValue();
        }
        
        /**
         * w肳ꂽCfbNX̗vfl擾B<p>
         * w肳ꂽvfnull܂͋󕶎̏ꍇ́AnullԂA{@link #wasNull()}trueԂB<br>
         *
         * @param index CfbNX
         * @return w肳ꂽCfbNX̗vfl
         * @exception NumberFormatException vflłȂꍇ
         */
        public BigInteger getBigInteger(int index) throws NumberFormatException{
            final String str = getString(index);
            if(str == null || str.length() == 0){
                wasNull = true;
                return null;
            }
            return new BigInteger(str);
        }
        
        /**
         * w肳ꂽCfbNX̗vfl擾B<p>
         * w肳ꂽvfnull܂͋󕶎̏ꍇ́AnullԂA{@link #wasNull()}trueԂB<br>
         *
         * @param index CfbNX
         * @return w肳ꂽCfbNX̗vfl
         * @exception NumberFormatException vflłȂꍇ
         */
        public BigDecimal getBigDecimal(int index) throws NumberFormatException{
            final String str = getString(index);
            if(str == null || str.length() == 0){
                wasNull = true;
                return null;
            }
            return new BigDecimal(str);
        }
    }
    
    private static class ReaderWrapper extends Reader{
        
        private Reader realReader;
        
        public ReaderWrapper(){
        }
        
        public ReaderWrapper(Reader reader){
            realReader = reader;
        }
        
        public Reader getReader(){
            return realReader;
        }
        
        public void setReader(Reader reader) throws IOException{
            if(realReader != null){
                throw new IOException("Reader is already commited.");
            }
            realReader = reader;
        }
        
        @Override
        public int read(CharBuffer target) throws IOException{
            if(realReader == null){
                return -1;
            }else{
                return realReader.read(target);
            }
        }
        
        @Override
        public int read() throws IOException{
            if(realReader == null){
                return -1;
            }else{
                return realReader.read();
            }
        }
        
        @Override
        public int read(char[] cbuf) throws IOException{
            if(realReader == null){
                return -1;
            }else{
                return realReader.read(cbuf);
            }
        }
        
        @Override
        public int read(char[] cbuf, int off, int len) throws IOException{
            if(realReader == null){
                return -1;
            }else{
                return realReader.read(cbuf, off, len);
            }
        }
        
        @Override
        public long skip(long n) throws IOException{
            if(realReader == null){
                return 0;
            }else{
                return realReader.skip(n);
            }
        }
        
        @Override
        public boolean ready() throws IOException{
            if(realReader == null){
                return false;
            }else{
                return realReader.ready();
            }
        }
        
        @Override
        public boolean markSupported(){
            if(realReader == null){
                return false;
            }else{
                return realReader.markSupported();
            }
        }
        
        @Override
        public void mark(int readAheadLimit) throws IOException{
            if(realReader == null){
                throw new IOException("Reader is null.");
            }else{
                realReader.mark(readAheadLimit);
            }
        }
        
        @Override
        public void reset() throws IOException{
            if(realReader != null){
                realReader.reset();
            }
        }
        
        @Override
        public void close() throws IOException{
            if(realReader != null){
                realReader.close();
            }
        }
    }
}
