package jp.sf.nikonikofw.validation;

import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;

import jp.sf.nikonikofw.annotation.Request;
import jp.sf.nikonikofw.exception.ValidationException;
import jp.sf.nikonikofw.util.StringUtils;

/**
 * 標準の{@link IValidator}実装クラスです。
 * {@link Request}アノテーションに対するバリデーション機能を提供します。
 * <p>
 * TODO カスタマイズしやすくする。
 * 
 * @author Naoki Takezoe
 */
public class DefaultValidator implements IValidator {

	@Override
	public void validateValue(String label, Field field, String value) {
		
		Required required = field.getAnnotation(Required.class);
		if(required != null && StringUtils.isEmpty(value)){
			throwValidationException(required.message(), label);
		}
		
		if(value != null){
			MinLength minLength = field.getAnnotation(MinLength.class);
			if(minLength != null && value.length() < minLength.length()){
				throwValidationException(minLength.message(), label, minLength.length());
			}
			MaxLength maxLength = field.getAnnotation(MaxLength.class);
			if(maxLength != null && value.length() > maxLength.length()){
				throwValidationException(maxLength.message(), label, maxLength.length());
			}
			
			ByteMinLength byteMinLength = field.getAnnotation(ByteMinLength.class);
			try {
				if(byteMinLength != null && value.getBytes(byteMinLength.encoding()).length < byteMinLength.length()){
					throwValidationException(byteMinLength.message(), label, byteMinLength.length());
				}
			} catch(UnsupportedEncodingException ex){
				throw new RuntimeException(ex); // TODO 例外はこれでいいか？
			}
			
			ByteMaxLength byteMaxLength = field.getAnnotation(ByteMaxLength.class);
			try {
				if(byteMaxLength != null && value.getBytes(byteMaxLength.encoding()).length > byteMaxLength.length()){
					throwValidationException(byteMaxLength.message(), label, byteMaxLength.length());
				}
			} catch(UnsupportedEncodingException ex){
				throw new RuntimeException(ex); // TODO 例外はこれでいいか？
			}
			
			Numeric numeric = field.getAnnotation(Numeric.class);
			if(numeric != null && !StringUtils.isNumeric(value)){
				throwValidationException(numeric.message(), label);
			}
			
			Pattern pattern = field.getAnnotation(Pattern.class);
			if(pattern != null && !value.matches(pattern.pattern())){
				throwValidationException(pattern.message(), label, pattern.pattern());
			}
		}
	}

	@Override
	public void validateValues(String label, Field field, String[] values) {
		
		Required required = field.getAnnotation(Required.class);
		if(required != null && (values==null || values.length==0)){
			throwValidationException(required.message(), label);
		}
		
		if(values != null){
			ArrayMinLength arrayMinLength = field.getAnnotation(ArrayMinLength.class);
			if(arrayMinLength != null && values.length < arrayMinLength.length()){
				throwValidationException(arrayMinLength.message(), label, arrayMinLength.length());
			}
			ArrayMaxLength arrayMaxLength = field.getAnnotation(ArrayMaxLength.class);
			if(arrayMaxLength != null && values.length > arrayMaxLength.length()){
				throwValidationException(arrayMaxLength.message(), label, arrayMaxLength.length());
			}
			
			for(String value: values){
				validateValue(label, field, value);
			}
		}
	}
	
	/**
	 * {@link ValidationException}をスローします。
	 * 
	 * @param message エラーメッセージ
	 * @param label 項目名
	 * @param args エラーメッセージに埋め込む変数
	 */
	protected void throwValidationException(String message, String label, Object... args){
		if(StringUtils.isEmpty(label)){
			throw new ValidationException();
		} else {
			Object[] array = new Object[args.length + 1];
			array[0] = label;
			System.arraycopy(args, 0, array, 1, args.length);
			throw new ValidationException(String.format(message, array));
		}
	}

}
