
package jp.sourceforge.nicoro;

import android.os.Handler;
import android.os.Looper;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;

import jp.sourceforge.nicoro.SurfaceVideoDrawer.DrawMessage;
import jp.sourceforge.nicoro.SurfaceVideoDrawer.VideoDrawBufferPool;

public interface SurfaceVideoDrawerInterface {

    boolean isNull();

    void quit();

    void quitAsync(ExecutorService executorService, final CountDownLatch latch);

    /**
     * 描画をポスト
     * @param videoBuffer
     */
    void postDraw(VideoDrawBuffer<?> videoBuffer);

    /**
     * 描画待機中の先頭バッファを、時間を無視して直ちに描画させる
     */
    void drawAtOnce();

    /**
     * 先頭バッファを強制的に取り返す
     * @return 強参照状態で返す
     */
    VideoDrawBuffer<?> getBackVideoBufferForcibly();

    boolean hasLongFrameskip();

    /**
     * {@link android.view.SurfaceView} のサイズを設定する
     * @param width
     * @param height
     */
    void setViewSize(int width, int height);

    /**
     * 動画のサイズを設定する
     * @param width
     * @param height
     */
    void setVideoSize(int width, int height);

    void registerEventHandler(Handler handler, int whatReady, int whatDestroyed,
            int whatPlayFinished);

    void setDrawMessage(DrawMessage drawMessage);

    void setVideoBufferPool(VideoDrawBufferPool videoBufferPool);

    void setFrameDurationMs(int duration);

    void setFrameRate(int num, int den);

    void pause();

    void restart();

    void startSeek();

    void updatePlayStartTime();

    void updatePlayStartTime(int frameCurrent);

    void fixPlayStartTime(int diff);

    void getCurrentPosition(Rational rational);

    void waitIdlePlay();

    /**
     * 描画待機中のバッファの個数を取得
     * @return
     */
    int getStandbyBufferSize();

    Handler getViewHandler();

    void setDecodeFinished(boolean isFinished);

    int getVpos();

    void start();

    int getViewWidth();

    int getViewHeight();

    public static class NullObject implements SurfaceVideoDrawerInterface {
        private static final NullObject sInstance = new NullObject();
        public static NullObject getInstance() {
            return sInstance;
        }

        private NullObject() {
        }

        @Override
        public boolean isNull() {
            return true;
        }

        @Override
        public void quit() {
        }

        @Override
        public void quitAsync(ExecutorService executorService, CountDownLatch latch) {
            latch.countDown();
        }

        @Override
        public void postDraw(VideoDrawBuffer<?> videoBuffer) {
        }

        @Override
        public void drawAtOnce() {
        }

        @Override
        public VideoDrawBuffer<?> getBackVideoBufferForcibly() {
            return null;
        }

        @Override
        public boolean hasLongFrameskip() {
            return false;
        }

        @Override
        public void setViewSize(int width, int height) {
        }

        @Override
        public void setVideoSize(int width, int height) {
        }

        @Override
        public void registerEventHandler(Handler handler, int whatReady, int whatDestroyed,
                int whatPlayFinished) {
        }

        @Override
        public void setDrawMessage(DrawMessage drawMessage) {
        }

        @Override
        public void setVideoBufferPool(VideoDrawBufferPool videoBufferPool) {
        }

        @Override
        public void setFrameDurationMs(int duration) {
        }

        @Override
        public void setFrameRate(int num, int den) {
        }

        @Override
        public void pause() {
        }

        @Override
        public void restart() {
        }

        @Override
        public void startSeek() {
        }

        @Override
        public void updatePlayStartTime() {
        }

        @Override
        public void updatePlayStartTime(int frameCurrent) {
        }

        @Override
        public void fixPlayStartTime(int diff) {
        }

        @Override
        public void getCurrentPosition(Rational rational) {
            rational.num = 0;
            rational.den = 1;
        }

        @Override
        public void waitIdlePlay() {
        }

        @Override
        public int getStandbyBufferSize() {
            return 0;
        }

        @Override
        public Handler getViewHandler() {
            return new Handler(Looper.getMainLooper());
        }

        @Override
        public void setDecodeFinished(boolean isFinished) {
        }

        @Override
        public int getVpos() {
            return 0;
        }

        @Override
        public void start() {
        }

        @Override
        public int getViewWidth() {
            return 0;
        }

        @Override
        public int getViewHeight() {
            return 0;
        }
    }
}
