package jp.sourceforge.nicoro;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CoderResult;
import java.nio.charset.CodingErrorAction;

import jp.sourceforge.nicoro.MessageLoader.EventListener;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.protocol.HTTP;

import android.util.Log;

public abstract class XmlLoader implements Runnable {
	private static final boolean DEBUG_LOGD = Release.IS_DEBUG && true;
	private static final String LOG_TAG = "NicoRo";

	protected Thread mThread = new Thread(this);
	protected boolean mIsStarted = false;
	protected volatile boolean mIsFinish = false;
	
	public void startLoad() {
		if (mIsStarted) {
			Log.d(LOG_TAG, "it has started");
			return;
		}
		mIsStarted = true;
		mIsFinish = false;
		
		mThread.start();
	}

	public void finish() {
		mIsFinish = true;
		try {
			mThread.join(2000L);
		} catch (InterruptedException e) {
			Log.d(LOG_TAG, e.getMessage(), e);
		}
		mIsStarted = false;
	}
	
	protected abstract HttpUriRequest createRequest();
	protected abstract boolean createDataFromXml(String xmlBody);
	protected abstract void dispatchOnFinished();
	protected abstract void dispatchOnOccurredError(String errorMessage);
	
	@Override
	public void run() {
		DefaultHttpClient httpClient = Util.createHttpClient();
		httpClient.getCookieStore().clear();
		InputStream inDownload = null;

		try {
			HttpUriRequest httpRequest = createRequest();
			HttpResponse httpResponse = httpClient.execute(
					httpRequest
					);
			if (DEBUG_LOGD) {
				Log.d(LOG_TAG, "XmlLoader HTTP response>");
				Log.d(LOG_TAG, httpResponse.getStatusLine().getReasonPhrase());
				Util.logHeaders(LOG_TAG, httpResponse.getAllHeaders());
			}
			
			int httpStatusCode = httpResponse.getStatusLine().getStatusCode();
			if (httpStatusCode == HttpStatus.SC_OK) {
				HttpEntity entityInput = httpResponse.getEntity();
				inDownload = entityInput.getContent();
				
				String xmlBody = readEntityAndDecodeDefault(inDownload);
//				String xmlBody = readEntityAndDecodeSpecial(inDownload);
				boolean result = createDataFromXml(xmlBody);
				
				if (result) {
					dispatchOnFinished();
				} else {
					dispatchOnOccurredError("XML parse failed");
				}
			} else {
				// エラー
				String errorMessage = "HTTP Status Code: " + httpStatusCode;
				dispatchOnOccurredError(errorMessage);
			}
			
		} catch (UnsupportedEncodingException e) {
			dispatchOnOccurredError("UnsupportedEncodingException");
			Log.d(LOG_TAG, e.getMessage(), e);
		} catch (ClientProtocolException e) {
			dispatchOnOccurredError("ClientProtocolException");
			Log.d(LOG_TAG, e.getMessage(), e);
		} catch (IOException e) {
			dispatchOnOccurredError("IOException");
			Log.d(LOG_TAG, e.getMessage(), e);
		} finally {
			if (inDownload != null) {
				try {
					inDownload.close();
				} catch (IOException e) {
					Log.d(LOG_TAG, e.getMessage(), e);
				}
			}
		}
		
	}
	
	private String readEntityAndDecodeDefault(InputStream inDownload) throws IOException {
		InputStreamReader inReader = null;
		try {
			inReader = new InputStreamReader(inDownload, HTTP.UTF_8);
			
			StringBuilder stringBody = new StringBuilder(1024 * 4);
			char[] buffer = new char[1024*4];
			while (!mIsFinish) {
				int read = inReader.read(buffer, 0, buffer.length);
				if (read < 0) {
					break;
				}
				stringBody.append(buffer, 0, read);
			}
			
			String xmlBody = stringBody.toString();
			if (DEBUG_LOGD) {
				Log.d(LOG_TAG, "read xml=" + xmlBody);
			}
			return xmlBody;
		} finally {
			if (inReader != null) {
				try {
					inReader.close();
				} catch (IOException e) {
					Log.d(LOG_TAG, e.getMessage(), e);
				}
			}
		}
	}

	private String readEntityAndDecodeSpecial(InputStream inDownload) throws IOException {
		final CharsetDecoder decoder = Charset.forName(HTTP.UTF_8).newDecoder()
		.onMalformedInput(CodingErrorAction.REPLACE)
		.onUnmappableCharacter(CodingErrorAction.REPLACE);
//		.onMalformedInput(CodingErrorAction.REPORT)
//		.onUnmappableCharacter(CodingErrorAction.REPORT);
		
		final StringBuilder stringBody = new StringBuilder(1024 * 4);
		final CharBuffer charBuffer = CharBuffer.allocate(1024 * 4);
		final ByteBuffer byteBuffer = ByteBuffer.allocate(1024 * 8);
		final char[] charBufferArray = charBuffer.array();
		final byte[] byteBufferArray = byteBuffer.array();
		int lastByteOffset = 0;
		while (!mIsFinish) {
			final int readByte = inDownload.read(
					byteBufferArray,
					lastByteOffset,
					byteBufferArray.length - lastByteOffset);
			if (readByte < 0) {
				byteBuffer.position(0).limit(lastByteOffset);
				charBuffer.position(0);
				final CoderResult result = decoder.decode(byteBuffer, charBuffer, true);
				stringBody.append(charBufferArray, 0, charBuffer.position());
				break;
			}
			
			final int byteBufferSize = lastByteOffset + readByte; 
			
			int end;
			for (end = byteBufferSize - 1; end > 0; --end) {
				final byte b = byteBufferArray[end];
				if (b < 0x80 || b > 0xbf) {
					break;
				}
			}
			
			byteBuffer.position(0).limit(end);
			charBuffer.position(0);
			final CoderResult result = decoder.decode(byteBuffer, charBuffer, false);
			stringBody.append(charBufferArray, 0, charBuffer.position());
			final int byteBufferPositionAfterDecode = byteBuffer.position();
			if (byteBufferPositionAfterDecode != byteBufferSize) {
				lastByteOffset = byteBufferSize - byteBufferPositionAfterDecode;
				assert lastByteOffset <= byteBufferPositionAfterDecode : "lastByteOffset=" + lastByteOffset + " byteBufferPositionAfterDecode=" + byteBufferPositionAfterDecode;
				System.arraycopy(byteBufferArray, byteBufferPositionAfterDecode,
						byteBufferArray, 0,
						lastByteOffset);
			}
		}
		
		String xmlBody = stringBody.toString();
		if (DEBUG_LOGD) {
			Log.d(LOG_TAG, "read xml=" + xmlBody);
		}
		return xmlBody;
	}
}
