package jp.sourceforge.nicoro;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.util.AttributeSet;
import android.view.View;

/**
 * 単純なテキスト表示View
 * 
 * SDKのApiDemosのLabelViewクラスを参考に改造
 *
 */
public class VariableLabelView extends View {
	private Paint mTextPaint;
	private StringBuilder mTextBuilder;
	private String mText;
    private float mAscent;

	public VariableLabelView(Context context) {
		this(context, null);
	}
	public VariableLabelView(Context context, AttributeSet attrs) {
		super(context, attrs);
		
		mTextPaint = new Paint();
		
		TypedArray a = context.obtainStyledAttributes(attrs,
				R.styleable.VariableLabelView);
		
//		String source =
//			a.getString(R.styleable.VariableLabelView_text);
//		int textColor =
//			a.getColor(R.styleable.VariableLabelView_textColor, Color.BLACK);
//		int textSize =
//			a.getDimensionPixelOffset(R.styleable.VariableLabelView_textSize, 16);
//		boolean antialias =
//			a.getBoolean(R.styleable.VariableLabelView_antialias, false);
//		int defaultCapacity =
//			a.getInt(R.styleable.VariableLabelView_defaultCapacity, 256);
		
		String source = null;
		int textColor = Color.BLACK;
		int textSize = 16;
		boolean antialias = false;
		int defaultCapacity = 256;
		
		final int indexCount = a.getIndexCount();
		for (int i = 0; i < indexCount; i++) {
			int attr = a.getIndex(i);
			switch (attr) {
			case R.styleable.VariableLabelView_text:
				source =
					a.getString(R.styleable.VariableLabelView_text);
				break;
			case R.styleable.VariableLabelView_textColor:
				textColor =
					a.getColor(R.styleable.VariableLabelView_textColor, Color.BLACK);
				break;
			case R.styleable.VariableLabelView_textSize:
				textSize =
					a.getDimensionPixelOffset(R.styleable.VariableLabelView_textSize, 16);
				break;
			case R.styleable.VariableLabelView_antialias:
				antialias =
					a.getBoolean(R.styleable.VariableLabelView_antialias, false);
				break;
			default:
				defaultCapacity =
					a.getInt(R.styleable.VariableLabelView_defaultCapacity, 256);
				break;
			}
		}
		
		if (source == null) {
			mTextBuilder = new StringBuilder(defaultCapacity);
			mText = null;
		} else {
			mTextBuilder = new StringBuilder(source);
			mText = source;
		}
		mTextPaint.setColor(textColor);
		mTextPaint.setTextSize(textSize);
		mTextPaint.setAntiAlias(antialias);
		
		a.recycle();
	}
	
	public StringBuilder getTextBuilder() {
		mText = null;
		return mTextBuilder;
	}
	
	public void notifyUpdateText() {
		mText = mTextBuilder.toString();
        requestLayout();
        invalidate();
	}
	
    public void setText(String text) {
    	mTextBuilder.delete(0, mTextBuilder.length());
        mTextBuilder.append(text);
        mText = text;
        requestLayout();
        invalidate();
    }
    
    public void setTextSize(int size) {
        mTextPaint.setTextSize(size);
        requestLayout();
        invalidate();
    }
    
    public void setTextColor(int color) {
        mTextPaint.setColor(color);
        invalidate();
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        setMeasuredDimension(measureWidth(widthMeasureSpec),
                measureHeight(heightMeasureSpec));
    }

    /**
     * Determines the width of this view
     * @param measureSpec A measureSpec packed into an int
     * @return The width of the view, honoring constraints from measureSpec
     */
    private int measureWidth(int measureSpec) {
        int result = 0;
        int specMode = MeasureSpec.getMode(measureSpec);
        int specSize = MeasureSpec.getSize(measureSpec);

        if (specMode == MeasureSpec.EXACTLY) {
            // We were told how big to be
            result = specSize;
        } else {
            // Measure the text
        	if (mText == null) {
        		mText = mTextBuilder.toString();
        	}
            result = (int) mTextPaint.measureText(mText) + getPaddingLeft()
                    + getPaddingRight();
            if (specMode == MeasureSpec.AT_MOST) {
                // Respect AT_MOST value if that was what is called for by measureSpec
                result = Math.min(result, specSize);
            }
        }

        return result;
    }

    /**
     * Determines the height of this view
     * @param measureSpec A measureSpec packed into an int
     * @return The height of the view, honoring constraints from measureSpec
     */
    private int measureHeight(int measureSpec) {
        int result = 0;
        int specMode = MeasureSpec.getMode(measureSpec);
        int specSize = MeasureSpec.getSize(measureSpec);

        mAscent = mTextPaint.ascent();
        if (specMode == MeasureSpec.EXACTLY) {
            // We were told how big to be
            result = specSize;
        } else {
            // Measure the text (beware: ascent is a negative number)
            result = (int) (-mAscent + mTextPaint.descent()) + getPaddingTop()
                    + getPaddingBottom();
            if (specMode == MeasureSpec.AT_MOST) {
                // Respect AT_MOST value if that was what is called for by measureSpec
                result = Math.min(result, specSize);
            }
        }
        return result;
    }

    /**
     * Render the text
     * 
     * @see android.view.View#onDraw(android.graphics.Canvas)
     */
    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        canvas.drawText(mText, getPaddingLeft(), getPaddingTop() - mAscent, mTextPaint);
    }
}
