package jp.sourceforge.nicoro;

import java.util.LinkedList;


import jp.sourceforge.nicoro.R;
import android.content.Intent;
import android.content.SharedPreferences;
import android.graphics.Canvas;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.preference.PreferenceManager;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.MotionEvent;
import android.view.View;
import android.widget.ImageView;
import android.widget.VideoView;

public class NicoroMediaPlayer extends AbstractNicoroPlayer {
	private static final boolean DEBUG_LOGD = Release.IS_DEBUG && true;
	private static final String LOG_TAG = "NicoRo";

	private static final int MSG_ID_VIDEO_DOWNLOAD_FINISHED = MSG_ID_SUB_OFFSET + 0;
	private static final int MSG_ID_MESSAGE_DRAW_MESSAGE = MSG_ID_SUB_OFFSET + 1;

	private boolean mIsVideoDownloadOk;
	
	private VideoView mVideoView;
	private MessageView mMessageView;
	private ImageView mIconPause;
	
	public NicoroMediaPlayer() {
		mHandler = new MessageHandler() {
			@Override
			public void handleMessage(Message msg) {
				if (mHandler == null) {
					if (DEBUG_LOGD) {
						Log.d(LOG_TAG, "Activity was destroyed. ignore message=" + msg.toString());
					}
					return;
				}
				switch (msg.what) {
				case MSG_ID_VIDEO_DOWNLOAD_FINISHED:
					mIsVideoDownloadOk = true;
					if (canStartPlay()) {
						startPlay();
					}
					break;
				case MSG_ID_MESSAGE_DRAW_MESSAGE:
					if (!isFinishing()) {
						mMessageView.invalidate();
						
						Message message = mHandler.obtainMessage(MSG_ID_MESSAGE_DRAW_MESSAGE);
						mHandler.sendMessageDelayed(message, 33L);
					}
					break;
				default:
					super.handleMessage(msg);
					break;
				}
			}
		};
	}
	
	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		
        if (DEBUG_LOGD) {
        	Log.d(LOG_TAG, "NicoroMediaPlayer onCreate");
        }
		
        mIsVideoDownloadOk = false;
        
		setContentView(R.layout.nicoro_mediaplayer);
		mVideoView = (VideoView) findViewById(R.id.video);
		DisplayMetrics metrics = new DisplayMetrics();
		getWindowManager().getDefaultDisplay().getMetrics(metrics);
		mVideoView.getLayoutParams().width =
//			(int) (metrics.heightPixels * 4 / 3 / metrics.density);
			(int) (metrics.heightPixels * 4 / 3);
		mVideoView.getLayoutParams().height =
//			(int) (metrics.heightPixels / metrics.density);
			(int) (metrics.heightPixels);
		mVideoView.requestLayout();
		
		mVideoView.setOnTouchListener(new View.OnTouchListener() {
			@Override
			public boolean onTouch(View v, MotionEvent event) {
				if (event.getAction() == MotionEvent.ACTION_DOWN) {
					if (mVideoView.isPlaying()) {
						mIconPause.setVisibility(View.VISIBLE);
						mVideoView.pause();
					} else {
						mIconPause.setVisibility(View.INVISIBLE);
						mVideoView.start();
					}
					return true;
				}
				return false;
			}
		});
		
		mIconPause = (ImageView) findViewById(R.id.icon_pause);
		mIconPause.setAlpha(192);
		
		mMessageView = (MessageView) findViewById(R.id.message_view);
		mMessageView.setPlayer(this);
		initializeView();
		
        Intent intent = getIntent();
        mVideoLoader = createVideoLoader(intent, getApplicationContext());
        if (mVideoLoader != null) {
        	mVideoLoader.setEventListener(new VideoLoader.EventListener() {
				@Override
				public void onCached(VideoLoader videoLoader) {
					// nothing
				}
				@Override
				public void onFinished(VideoLoader videoLoader) {
					if (mHandler != null) {
						mHandler.sendEmptyMessage(MSG_ID_VIDEO_DOWNLOAD_FINISHED);
					}
				}
				@Override
				public void onOccurredError(VideoLoader videoLoader, String errorMessage) {
					if (mHandler != null) {
						Message message = mHandler.obtainMessage(MSG_ID_VIDEO_OCCURRED_ERROR, errorMessage);
						mHandler.sendMessage(message);
					}
				}
				@Override
				public void onNotifyProgress(int num, int den) {
					if (mHandler != null) {
						if (mVideoView == null || !mVideoView.isPlaying()) {
							mHandler.removeMessages(MSG_ID_VIDEO_NOTIFY_PROGRESS);
							Message message = mHandler.obtainMessage(MSG_ID_VIDEO_NOTIFY_PROGRESS, num, den);
							mHandler.sendMessage(message);
						}
					}
				}
        	});
        	// 最初にダウンロード要求
        	mVideoLoader.startLoad();
        }
        
	}

	@Override
	protected void onDestroy() {
		mVideoView.stopPlayback();
		super.onDestroy();
		mVideoView = null;
		mMessageView = null;
	}
	
	@Override
	protected boolean canStartPlay() {
		return mIsVideoDownloadOk && mIsMessageOk;
	}
	
	@Override
	protected void startPlay() {
		super.startPlay();
    	
		String file = mVideoLoader.getFilePath();
        mVideoView.setVideoPath(file);
		
		mVideoView.start();
		if (DEBUG_LOGD) {
			Log.d(LOG_TAG, "VideoView start");
		}
		
		Message message = mHandler.obtainMessage(MSG_ID_MESSAGE_DRAW_MESSAGE);
		mHandler.sendMessageDelayed(message, 33L);
	}
	
	@Override
	protected StringBuilder appendCurrentPlayTime(StringBuilder builder) {
		getCurrentPositionVideoPlay(mRatinalCurrentPlayTime);
		final int posNum = mRatinalCurrentPlayTime.num;
		final int posDen = mRatinalCurrentPlayTime.den;
		int hours = posNum / (posDen * 60 * 60);
		int minutes = posNum / (posDen * 60) % 60;
		int seconds = posNum / posDen % 60;
		if (hours != 0) {
			builder.append(hours).append(':');
		}
		if (minutes < 10) {
			builder.append('0');
		}
		builder.append(minutes).append(':');
		if (seconds < 10) {
			builder.append('0');
		}
		return builder.append(seconds);
	}
	
	@Override
	protected void getCurrentPositionVideoPlay(Rational rational) {
		rational.num = mVideoView.getCurrentPosition();
		rational.den = 1000;	// ms
	}
	@Override
	protected void getCurrentPositionAudioPlay(Rational rational) {
		// Videoと同じ
		getCurrentPositionVideoPlay(rational);
	}
	@Override
	protected void getCurrentPositionVideoDecode(Rational rational) {
		// playの時間を代わりに使う
		getCurrentPositionVideoPlay(rational);
	}
	@Override
	protected void getCurrentPositionAudioDecode(Rational rational) {
		// playの時間を代わりに使う
		getCurrentPositionAudioPlay(rational);
	}
	
	public void drawMessageView(MessageView messageView, Canvas canvas) {
		canvas.drawColor(0x00000000);
//		if (mVideoView.isPlaying()) {
		if (mIsMessageOk) {
			int vpos = mVideoView.getCurrentPosition() / 10;
			drawMessage(canvas,
					mChatsWait,
					mChatsRunningNaka,
					mChatsRunningShita,
					mChatsRunningUe,
					vpos, messageView.getWidth(), messageView.getHeight());
		}
	}
}
