package jp.sourceforge.nicoro;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;


import jp.sourceforge.nicoro.R;
import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.preference.PreferenceManager;
import android.util.Log;
import android.view.View;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;

public abstract class AbstractNicoroPlayer extends Activity {
	private static final boolean DEBUG_LOGV = Release.IS_DEBUG && false;
	private static final boolean DEBUG_LOGD = Release.IS_DEBUG && true;
	private static final String LOG_TAG = "NicoRo";
	
	public static final String INTENT_NAME_VIDEO_URL = "VIDEO_URL";
	public static final String INTENT_NAME_COOKIE = "COOKIE";
	public static final String INTENT_NAME_VIDEO_NUMBER = "VIDEO_NUMBER";
	public static final String INTENT_NAME_MESSAGE_URL = "MESSAGE_URL";
	public static final String INTENT_NAME_THREAD_ID = "THREAD_ID";
	public static final String INTENT_NAME_USER_ID = "USER_ID";
	public static final String INTENT_NAME_THREAD_KEY = "THREAD_KEY";
	public static final String INTENT_NAME_FORCE_184 = "FORCE_184";
	
	protected static final int MSG_ID_MESSAGE_FINISHED = 0;
	protected static final int MSG_ID_MESSAGE_OCCURRED_ERROR = 1;
	protected static final int MSG_ID_THUMBINFO_FINISHED = 2;
	protected static final int MSG_ID_THUMBINFO_OCCURRED_ERROR = 3;
	protected static final int MSG_ID_VIDEO_OCCURRED_ERROR = 4;
	protected static final int MSG_ID_VIDEO_NOTIFY_PROGRESS = 5;
	protected static final int MSG_ID_INFO_TIME_UPDATE = 6;
	protected static final int MSG_ID_SUB_OFFSET = 0x100;
	
	public static final int REAL_PLAYER_WIDTH_PX = 512;
	public static final int REAL_PLAYER_HEIGHT_PX_4_3 = 384;
	public static final int REAL_PLAYER_HEIGHT_PX_16_9 = 288;
	
	protected VideoLoader mVideoLoader;
	protected MessageLoader mMessageLoader;
	protected RelativeLayout mProgressGroup;
	protected ProgressBar mProgressBar;
	protected VariableLabelView mProgressText;
	protected VariableLabelView mInfoCountPlay;
	protected VariableLabelView mInfoCountComment;
	protected VariableLabelView mInfoCountMylist;
	protected VariableLabelView mInfoTime;
	protected ThumbInfo mThumbInfo;

	protected boolean mIsMessageOk;
	protected List<MessageChat> mChatsWait = null;
	protected List<MessageChat> mChatsRunningNaka = null;
	protected List<MessageChat> mChatsRunningShita = null;
	protected List<MessageChat> mChatsRunningUe = null;

    protected Paint mPaintText = new Paint();
    
    protected volatile MessageHandler mHandler;
    
    private Matrix mMatrixMessage = new Matrix();
    private Random mRandom = new Random();
    
    protected abstract class MessageHandler extends Handler {
		@Override
		public void handleMessage(Message msg) {
			if (mHandler == null) {
				if (DEBUG_LOGD) {
					Log.d(LOG_TAG, "Activity was destroyed. ignore message=" + msg.toString());
				}
				return;
			}
			switch (msg.what) {
			case MSG_ID_MESSAGE_FINISHED:
				mIsMessageOk = true;
				mChatsWait = 
						new LinkedList<MessageChat>(mMessageLoader.getChats());
				mChatsRunningNaka = 
						new LinkedList<MessageChat>();
				mChatsRunningShita = 
					new LinkedList<MessageChat>();
				mChatsRunningUe = 
					new LinkedList<MessageChat>();
				if (canStartPlay()) {
					startPlay();
				}
				break;
			case MSG_ID_MESSAGE_OCCURRED_ERROR:
				Util.showErrorDialog(AbstractNicoroPlayer.this,
						(String) msg.obj, true);
				break;
			case MSG_ID_THUMBINFO_FINISHED:
				mMessageLoader.setVideoLength((String) msg.obj);
				mMessageLoader.startLoad();
				break;
			case MSG_ID_THUMBINFO_OCCURRED_ERROR:
				Util.showErrorDialog(AbstractNicoroPlayer.this,
						(String) msg.obj, true);
				break;
			case MSG_ID_VIDEO_OCCURRED_ERROR:
				Util.showErrorDialog(AbstractNicoroPlayer.this,
						(String) msg.obj, true);
				break;
			case MSG_ID_VIDEO_NOTIFY_PROGRESS:
				StringBuilder textData = mProgressText.getTextBuilder();
				textData.delete(0, textData.length());
				textData.append(msg.arg1).append('/').append(msg.arg2);
				mProgressText.notifyUpdateText();
				break;
			case MSG_ID_INFO_TIME_UPDATE:
				if (!isFinishing()) {
					StringBuilder infoTimeData = mInfoTime.getTextBuilder();
					infoTimeData.delete(0, infoTimeData.length());
					appendCurrentPlayTime(infoTimeData).append('/').append(mThumbInfo.getFormattedLengthForPlayer());
					mInfoTime.notifyUpdateText();
					
					Message message = mHandler.obtainMessage(MSG_ID_INFO_TIME_UPDATE);
					mHandler.sendMessageDelayed(message, 33L);
				}
				break;
			default:
				assert false : msg.what;
				break;
			}
		}
    }
    
	protected abstract boolean canStartPlay();
	protected abstract StringBuilder appendCurrentPlayTime(StringBuilder builder);
	
	protected void startPlay() {
    	mProgressGroup.setVisibility(View.GONE);
		Message message = mHandler.obtainMessage(MSG_ID_INFO_TIME_UPDATE);
		mHandler.sendMessageDelayed(message, 33L);
		
		mInfoCountPlay.getTextBuilder()
		.append("再生：").append(mThumbInfo.getViewCounter());
		mInfoCountPlay.notifyUpdateText();
		mInfoCountComment.getTextBuilder()
		.append("コメント：").append(mThumbInfo.getCommentNum());
		mInfoCountComment.notifyUpdateText();
		mInfoCountMylist.getTextBuilder()
		.append("マイリスト：").append(mThumbInfo.getMylistCounter());
		mInfoCountMylist.notifyUpdateText();
	}
	
	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		
        mIsMessageOk = false;

        Intent intent = getIntent();
        
        mThumbInfo = createThumbInfo(intent);
        if (mThumbInfo != null) {
        	mThumbInfo.setEventListener(new ThumbInfo.EventListener() {
				@Override
				public void onFinished(ThumbInfo thumbInfo) {
					if (mHandler != null) {
						Message message = mHandler.obtainMessage(MSG_ID_THUMBINFO_FINISHED, thumbInfo.getLength());
						mHandler.sendMessage(message);
					}
				}
				@Override
				public void onOccurredError(ThumbInfo thumbInfo,
						String errorMessage) {
					if (mHandler != null) {
						Message message = mHandler.obtainMessage(MSG_ID_THUMBINFO_OCCURRED_ERROR, errorMessage);
						mHandler.sendMessage(message);
					}
				}
        	});
        	mThumbInfo.startLoad();
        }
        
		SharedPreferences sharedPreference = PreferenceManager.getDefaultSharedPreferences(
				getApplicationContext());
		String userSession = sharedPreference.getString(NicoroConfig.COOKIE_USER_SESSION, null);
        mMessageLoader = createMessageLoader(intent, getApplicationContext(), userSession);
        if (mMessageLoader != null) {
        	mMessageLoader.setEventListener(new MessageLoader.EventListener() {
				@Override
				public void onFinished(MessageLoader messageLoader) {
					if (mHandler != null) {
						mHandler.sendEmptyMessage(MSG_ID_MESSAGE_FINISHED);
					}
				}
				@Override
				public void onOccurredError(MessageLoader messageLoader,
						String errorMessage) {
					if (mHandler != null) {
						Message message = mHandler.obtainMessage(MSG_ID_MESSAGE_OCCURRED_ERROR, errorMessage);
						mHandler.sendMessage(message);
					}
				}
        	});
//        	mMessageLoader.startLoad();
        }
        
        boolean messageAntialias = sharedPreference.getBoolean(
				getString(R.string.pref_key_message_antialias), false);
        mPaintText.setAntiAlias(messageAntialias);
	}
	
	@Override
	protected void onDestroy() {
    	if (mVideoLoader != null) {
    		mVideoLoader.finish();
    		mVideoLoader = null;
    	}
    	if (mMessageLoader != null) {
    		mMessageLoader.finish();
    		mMessageLoader = null;
    	}
    	if (mThumbInfo != null) {
    		mThumbInfo.finish();
    		mThumbInfo = null;
    	}
    	mProgressGroup = null;
    	mProgressBar = null;
    	mProgressText = null;
    	mChatsWait = null;
    	mChatsRunningNaka = null;
    	mChatsRunningShita = null;
    	mChatsRunningUe = null;
    	mPaintText = null;
    	mHandler = null;
		super.onDestroy();
	}
	
    @Override
    protected void finalize() throws Throwable {
    	try {
    		super.finalize();
    	} finally {
        	if (mVideoLoader != null) {
        		mVideoLoader.finish();
        	}
        	if (mMessageLoader != null) {
        		mMessageLoader.finish();
        	}
        	if (mThumbInfo != null) {
        		mThumbInfo.finish();
        	}
    	}
    }
	
	protected void initializeView() {
		mProgressGroup = (RelativeLayout) findViewById(R.id.progress_group);
        mProgressBar = (ProgressBar) findViewById(R.id.progress);
        mProgressText = (VariableLabelView) findViewById(R.id.progress_text);
        mInfoCountPlay = (VariableLabelView) findViewById(R.id.info_count_play);
        mInfoCountComment = (VariableLabelView) findViewById(R.id.info_count_comment);
        mInfoCountMylist = (VariableLabelView) findViewById(R.id.info_count_mylist);
        mInfoTime = (VariableLabelView) findViewById(R.id.info_time);
	}
	
	protected static VideoLoader createVideoLoader(Intent intent, Context context) {
		VideoLoader videoLoader = null;
        String url = intent.getStringExtra(AbstractNicoroPlayer.INTENT_NAME_VIDEO_URL);
        String cookie = intent.getStringExtra(AbstractNicoroPlayer.INTENT_NAME_COOKIE);
        String videoNumber = intent.getStringExtra(AbstractNicoroPlayer.INTENT_NAME_VIDEO_NUMBER);
        if (DEBUG_LOGD) {
        	Log.d(LOG_TAG, " url=" + url + " cookie=" + cookie + " videoNumber=" + videoNumber);
        }
        if (url != null && cookie != null) {
        	videoLoader = new VideoLoader(url, cookie, videoNumber,
        			context);
        }
		return videoLoader;
	}

	protected static MessageLoader createMessageLoader(Intent intent, Context context,
			String userSession) {
		MessageLoader messageLoader = null;
        String cookie = intent.getStringExtra(AbstractNicoroPlayer.INTENT_NAME_COOKIE);
        String messageUrl = intent.getStringExtra(AbstractNicoroPlayer.INTENT_NAME_MESSAGE_URL);
        String threadId = intent.getStringExtra(AbstractNicoroPlayer.INTENT_NAME_THREAD_ID);
        String userId = intent.getStringExtra(AbstractNicoroPlayer.INTENT_NAME_USER_ID);
        String threadKey = intent.getStringExtra(AbstractNicoroPlayer.INTENT_NAME_THREAD_KEY);
        String force184 = intent.getStringExtra(AbstractNicoroPlayer.INTENT_NAME_FORCE_184);
        if (DEBUG_LOGD) {
        	Log.d(LOG_TAG, " cookie=" + cookie + " messageUrl=" + messageUrl
        			+ " threadId=" + threadId + " userId=" + userId
        			+ " threadKey=" + threadKey + " force184=" + force184);
        }
        if (messageUrl != null && threadId != null && cookie != null) {
        	messageLoader = new MessageLoader(messageUrl, threadId, userSession,
        			userId, threadKey, force184, 
        			context);
        }
		return messageLoader;
	}
	
	protected static ThumbInfo createThumbInfo(Intent intent) {
		ThumbInfo thumbInfo = null;
        String videoNumber = intent.getStringExtra(AbstractNicoroPlayer.INTENT_NAME_VIDEO_NUMBER);
        if (DEBUG_LOGD) {
        	Log.d(LOG_TAG, " videoNumber=" + videoNumber);
        }
        if (videoNumber != null) {
        	thumbInfo = new ThumbInfo(videoNumber);
        }
        return thumbInfo;
	}
    
    protected void drawMessage(Canvas canvas,
    		List<MessageChat> chatsWait,
    		List<MessageChat> chatsRunningNaka,
    		List<MessageChat> chatsRunningShita,
    		List<MessageChat> chatsRunningUe,
    		int vpos, int videoWidth, int videoHeight) {
    	
    	mMatrixMessage.reset();
    	mMatrixMessage.setScale(
    			(float) videoWidth / (float) REAL_PLAYER_WIDTH_PX,
    			(float) videoHeight / (float) REAL_PLAYER_HEIGHT_PX_4_3);
    	canvas.setMatrix(mMatrixMessage);
    	
    	assert chatsWait != null;
    	assert chatsRunningNaka != null;
    	assert chatsRunningShita != null;
    	assert chatsRunningUe != null;
    	
    	if (DEBUG_LOGV) {
    		Log.v(LOG_TAG, "vpos=" + vpos);
    	}
    	
    	for (Iterator<MessageChat> it = chatsRunningNaka.iterator(); it.hasNext(); ) {
    		MessageChat chat = it.next();
    		if (vpos > chat.vpos + MessageChat.DISPLAY_TIME_VPOS_NAKA) {
    			it.remove();
    			continue;
    		}
    	}
    	for (Iterator<MessageChat> it = chatsRunningShita.iterator(); it.hasNext(); ) {
    		MessageChat chat = it.next();
    		if (vpos > chat.vpos + MessageChat.DISPLAY_TIME_VPOS_SHITA) {
    			it.remove();
    			continue;
    		}
    	}
    	for (Iterator<MessageChat> it = chatsRunningUe.iterator(); it.hasNext(); ) {
    		MessageChat chat = it.next();
    		if (vpos > chat.vpos + MessageChat.DISPLAY_TIME_VPOS_UE) {
    			it.remove();
    			continue;
    		}
    	}
    	
    	for (Iterator<MessageChat> it = chatsWait.iterator(); it.hasNext(); ) {
    		MessageChat chat = it.next();
    		if (chat.vpos >= vpos) {
    			int vposOffset;
    			if (chat.getPos() == MessageChat.POS_NAKA) {
    				vposOffset = MessageChat.DISPLAY_TIME_VPOS_NAKA;
    			} else if (chat.getPos() == MessageChat.POS_SHITA) {
    				vposOffset = MessageChat.DISPLAY_TIME_VPOS_SHITA;
    			} else if (chat.getPos() == MessageChat.POS_UE) {
    				vposOffset = MessageChat.DISPLAY_TIME_VPOS_UE;
    			} else {
    				assert false;
    				vposOffset = MessageChat.DISPLAY_TIME_VPOS_NAKA;
    			}
    			if (chat.vpos <= (vpos + vposOffset)) {
	    			it.remove();
	    			
	    			chat.computeWidth(mPaintText);
	    			chat.computeSpeed();
	    			if (chat.getPos() == MessageChat.POS_NAKA) {
	    				int nextY = chat.computeNakaNextY(vpos, chatsRunningNaka, mRandom);
	    				chat.setY(nextY, mPaintText);
	    				
	    				chat.addNakaOrder(chatsRunningNaka);
	    			} else if (chat.getPos() == MessageChat.POS_SHITA) {
	    				int nextY = chat.computeShitaNextY(chatsRunningShita, mRandom);
	    				chat.setY(nextY, mPaintText);
		
	    				chat.addShitaOrder(chatsRunningShita);
	    			} else if (chat.getPos() == MessageChat.POS_UE) {
						int nextY = chat.computeUeNextY(chatsRunningUe, mRandom);
	    				chat.setY(nextY, mPaintText);
		
	    				chat.addUeOrder(chatsRunningUe);
	    			} else {
	    				assert false;
	    			}
	    			
	        		if (DEBUG_LOGD) {
	        			Log.d(LOG_TAG, "cur vpos=" + vpos + " vpos=" + chat.vpos + " chat=" + chat.getText());
	        		}
    			}
    		}
    	}
    	
    	for (Iterator<MessageChat> it = chatsRunningNaka.iterator(); it.hasNext(); ) {
    		MessageChat chat = it.next();
    		chat.draw(vpos, canvas, mPaintText);
    	}
		
    	for (Iterator<MessageChat> it = chatsRunningShita.iterator(); it.hasNext(); ) {
    		MessageChat chat = it.next();
    		chat.draw(vpos, canvas, mPaintText);
    	}
		
    	for (Iterator<MessageChat> it = chatsRunningUe.iterator(); it.hasNext(); ) {
    		MessageChat chat = it.next();
    		chat.draw(vpos, canvas, mPaintText);
    	}
		
    }
}
