package jp.sourceforge.nicoro;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;

import jp.sourceforge.nicoro.MessageLoader.EventListener;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.protocol.HTTP;

import android.util.Log;

public abstract class XmlLoader implements Runnable {
	private static final boolean DEBUG_LOGD = Release.IS_DEBUG && true;
	private static final String LOG_TAG = "NicoRo";

	protected Thread mThread = new Thread(this);
	protected boolean mIsStarted = false;
	protected volatile boolean mIsFinish = false;
	
	public void startLoad() {
		if (mIsStarted) {
			Log.d(LOG_TAG, "it has started");
			return;
		}
		mIsStarted = true;
		mIsFinish = false;
		
		mThread.start();
	}

	public void finish() {
		mIsFinish = true;
		try {
			mThread.join(1000L);
		} catch (InterruptedException e) {
			Log.d(LOG_TAG, "", e);
		}
		mIsStarted = false;
	}
	
	protected abstract HttpUriRequest createRequest();
	protected abstract boolean createDataFromXml(String xmlBody);
	protected abstract void dispatchOnFinished();
	protected abstract void dispatchOnOccurredError(String errorMessage);
	
	@Override
	public void run() {
		DefaultHttpClient httpClient = Util.createHttpClient();
		httpClient.getCookieStore().clear();
		InputStream inDownload = null;
		InputStreamReader inReader = null;

		try {
			HttpUriRequest httpRequest = createRequest();
			HttpResponse httpResponse = httpClient.execute(
					httpRequest
					);
			if (DEBUG_LOGD) {
				Log.d(LOG_TAG, "XmlLoader HTTP response>");
				Log.d(LOG_TAG, httpResponse.getStatusLine().getReasonPhrase());
				Util.logHeaders(LOG_TAG, httpResponse.getAllHeaders());
			}
			
			int httpStatusCode = httpResponse.getStatusLine().getStatusCode();
			if (httpStatusCode == HttpStatus.SC_OK) {
				HttpEntity entityInput = httpResponse.getEntity();
				inDownload = entityInput.getContent();
				inReader = new InputStreamReader(inDownload, HTTP.UTF_8);
				
				StringBuilder stringBody = new StringBuilder(1024 * 4);
				char[] buffer = new char[1024*4];
				while (!mIsFinish) {
					int read = inReader.read(buffer, 0, buffer.length);
					if (read < 0) {
						break;
					}
					stringBody.append(buffer, 0, read);
				}
				
				String xmlBody = stringBody.toString();
				if (DEBUG_LOGD) {
					Log.d(LOG_TAG, xmlBody);
				}
				boolean result = createDataFromXml(xmlBody);
				
				if (result) {
					dispatchOnFinished();
				} else {
					dispatchOnOccurredError("XML parse failed");
				}
			} else {
				// エラー
				String errorMessage = "HTTP Status Code: " + httpStatusCode;
				dispatchOnOccurredError(errorMessage);
			}
			
		} catch (UnsupportedEncodingException e) {
			dispatchOnOccurredError("UnsupportedEncodingException");
			Log.d(LOG_TAG, "", e);
		} catch (ClientProtocolException e) {
			dispatchOnOccurredError("ClientProtocolException");
			Log.d(LOG_TAG, "", e);
		} catch (IOException e) {
			dispatchOnOccurredError("IOException");
			Log.d(LOG_TAG, "", e);
		} finally {
			if (inReader != null) {
				try {
					inReader.close();
				} catch (IOException e) {
					Log.d(LOG_TAG, "", e);
				}
			}
			if (inDownload != null) {
				try {
					inDownload.close();
				} catch (IOException e) {
					Log.d(LOG_TAG, "", e);
				}
			}
		}
		
	}
}
