#!/usr/bin/env python2.6.2
# -*- coding: utf-8 -*-

""" Nicolib - CommentServer
Notes:
  ■コメントサーバについて
    ニコニコ生放送のコメント情報を取得するために接続するサーバです。
    下記のようなフォーマットでRequest/Responseのやり取り(XMLSocket)を行います。

  ■Request Format
    '<thread thread="%d" res_from="%d" version="%d" />\0'
      - thread   ... ?
      - res_from ... 取得するコメント数。取得しない場合は0。過去に遡って取得するため負数で指定(1件取得 = -1)
      - version  ... ?

  ■Response Format
    1.初回接続時の応答
    '<thread last_res="%d" resultcode="%d" revision="%d" server_time="%d" thread="%d" ticket="%d"/>'
      - last_res    ... ?
      - resultcode  ... ?
      - revision    ... ?
      - server_time ... ?
      - thread      ... ?
      - ticket      ... ?
    2.コメント情報
    '<chat date="%d" no="%d" premium="%d" thread="%d" user_id="%s" vpos="%d">コメント本文</chat>'
      - date        ... ?
      - no          ... ?
      - premium     ... ?
      - thread      ... ?
      - user_id     ... ?
      - vpos        ... ?

Example:
  >>> address = "hoge.com" # CommentServer Address(getPlayerStatus/getAlertStatusで得られます)
  >>> port    = "80"       # CommentServer Port   (getPlayerStatus/getAlertStatusで得られます)
  >>> server = CommentServer(address, port)
  >>> server.connect()
  >>> server.send('<thread thread="1" res_from="-1" version="20061206"')
  >>> server.receive()
"""

import socket
import thread

import CommentMessage as cmes
import AlertMessage as ames

class CommentServer(object):
    """ CommentServer - CommentServerと直接やり取りするclass """
    REQUEST_FORMAT  = '<thread thread="%s" res_from="%d" version="%s" />\0'
    REQUEST_VERSION = '20061206'

    def __init__(self, address, port, thread):
        self._address  = address   # CommentServerへ接続するAddress
        self._port     = port      # CommentServerへ接続するPort
        self._thread   = thread    # スレッド情報
        self._socket   = None      # CommentServer接続用Socket
        self._observer = None      # Comment情報通知先
        self._buffer   = ''        # Comment情報を格納するBuffer
        
    def connect(self):
        """ CommentServerに接続する """
        try:
            self._socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            self._socket.connect((self._address, int(self._port)))
        except socket.error, msg:
            if self._sock:
                self._socket.close()
            self._socket = None
            raise socket.error, msg

    def startgetcomment(self):
        """ Comment取得開始 """
        senddata = self.REQUEST_FORMAT % (self._thread, 0, self.REQUEST_VERSION)
        self.connect()
        self.send(senddata)        # <thread>送信
        self.receive()             # <thread>受信
        thread.start_new_thread(self._getComment, ())

    def _getComment(self):
        """ コメント取得 """
        senddata = self.REQUEST_FORMAT % (self._thread, -1, self.REQUEST_VERSION)

        while(1):
            # 送信 - '<thread thread="?" res_from="?" version="20061206" />\0'
            self.send(senddata)

            # 受信 - コメント情報は、1コメントに満たない状態で送られてきたり
            #        複数コメント(\0区切り)が一気に送られてくる場合があるため、buffer管理
            self._buffer = self._buffer + self.receive()

            # 1コメント単位(\0区切り)でParse/通知処理
            while '\0' in self._buffer:
                datas = self._buffer.split('\0',1)
                self._buffer = datas[1]
                # Parse
                commentdata = cmes.CommentMessage.parse(datas[0])
                # 通知
                self._notifyObserver(commentdata)

    def startAlert(self):
        """ アラート受信開始 """
        senddata = self.REQUEST_FORMAT % (self._thread, 0, self.REQUEST_VERSION)
        self.connect()
        self.send(senddata)        # <thread>送信
        self.receive()             # <thread>受信
        thread.start_new_thread(self._checkAlert, ())
        
    def _checkAlert(self):
        """ アラート通知待ち """
        senddata = self.REQUEST_FORMAT % (self._thread, -1, self.REQUEST_VERSION)

        while(1):
            # 送信 - '<thread thread="?" res_from="?" version="20061206" />\0'
            self.send(senddata)

            # 受信 - アラート情報は、1アラートに満たない状態で送られてきたり
            #        複数アラート(\0区切り)が一気に送られてくる場合があるため、buffer管理
            self._buffer = self._buffer + self.receive()

            # 1アラート単位(\0区切り)でParse/通知処理
            while '\0' in self._buffer:
                datas = self._buffer.split('\0',1)
                self._buffer = datas[1]
                # Parse
                alertdata = ames.AlertMessage.parse(datas[0])
                # 通知
                if alertdata.get('community_id',None):
                    self._notifyObserver(alertdata)

    def send(self, senddata):
        """ CommentServerにData送信 """
        self._socket.sendall(senddata)
        
    def receive(self):
        """ CommentServerからのData受信 """
        return self._socket.recv(1024)

    def addObserver(self, observer):
        """ コメント受信通知先登録 """
        self._observer = observer

    def _notifyObserver(self, message):
        """ コメント受信通知 """
        self._observer.notify(message)


