package nicolib.api.live;

import java.util.Date;

import nicolib.api.AsyncLoadHandler;
import nicolib.api.NicoApiException;
import nicolib.api.XmlApiBase;
import nicolib.util.Logger;

/**
 * 生放送配信情報取得用API
 * @author hal
 *
 */
public class PublishStatus extends XmlApiBase {

	private static final String API_URLFORMAT = "http://watch.live.nicovideo.jp/api/getpublishstatus?v=%s";
	
	/**
	 * 指定した放送IDの配信情報を取得する
	 * [APIUrl] http://watch.live.nicovideo.jp/api/getpublishstatus?v=[放送ID]
	 * @param id 放送ID（URL形式でもよい）
	 * @return PublishStatus
	 * @throws NicoApiException
	 */
	public static PublishStatus get(String id) throws NicoApiException{
		String lid = checkAndGetLiveId(id);
		PublishStatus status = new PublishStatus();
		status.access(String.format(API_URLFORMAT, lid));
		
		return status;
	}
	
	/**
	 * 指定した放送IDの配信情報を非同期で取得する
	 * [APIUrl] http://watch.live.nicovideo.jp/api/getpublishstatus?v=[放送ID]
	 * @param id 放送ID（URL形式でもよい）
	 * @param id
	 * @param handler
	 * @throws NicoApiException
	 */
	public static void getAsync(String id, AsyncLoadHandler handler) throws NicoApiException{
		String lid = checkAndGetLiveId(id);
		PublishStatus status = new PublishStatus();
		status.accessAsync(String.format(API_URLFORMAT, lid), handler);
		
	}
	
	private PublishStatus(){
		
	}
	
	public String getLiveId(){
		return select("getpublishstatus/stream/id");
	}
	
	public int getWatchCount(){
		return selectInt("getpublishstatus/stream/watch_count", 0);
	}
	
	public int getCommentCount(){
		return selectInt("getpublishstatus/stream/comment_count", 0);
	}
	
	public String getToken(){
		return select("getpublishstatus/stream/token");
	}
	
	public int getExclude(){
		return selectInt("getpublishstatus/stream/exclude", 0);
	}

	public String getProviderType(){
		return select("getpublishstatus/stream/provider_type");
	}
	
	public String getDefaultCommunity(){
		return select("getpublishstatus/stream/default_community");
	}
	
	public Date getBaseTime(){
		return selectDate("getpublishstatus/stream/base_time");
	}
	
	public Date getOpenTime(){
		return selectDate("getpublishstatus/stream/open_time");
	}
	
	public Date getStartTime(){
		return selectDate("getpublishstatus/stream/start_time");
	}
	
	public Date getEndTime(){
		return selectDate("getpublishstatus/stream/end_time"); 
	}
	
	public int getAllowVote(){
		return selectInt("getpublishstatus/steram/allow_vote", 0);
	}
	
	public int getDisableAdaptiveBitrate(){
		return selectInt("getpublishstatus/steram/disable_adaptive_bitrate", 0);
	}
	
	public String getNickname(){
		return select("getpublishstatus/user/nickname");
	}
	
	public int getIsPremium(){
		return selectInt("getpublishstatus/user/is_premium", 0);
	}
	
	public int getUserId(){
		return selectInt("getpublishstatus/user/user_id", 0);
	}
	
	public String getRtmpUrl(){
		return select("getpublishstatus/rtmp/url");
	}
	
	public String getRtmpTicket(){
		return select("getpublishstatus/rtmp/Ticket");
	}
	
	@Override
	public String getErrorCode() {
		return select("getpublishstatus/error/code");
	}

	@Override
	public String getErrorMessage() {
		String errorCode = getErrorCode();
		
		if(hasError()){
			if(errorCode == null){
				return "getpublishstatusの取得に失敗しました。";
			}else if(errorCode.equals("permission_denied")){
				return "アクセス許可がありません。";
			}else if(errorCode.equals("notfound")){
				return "放送が見つかりませんでした。";
			}else if(errorCode.equals("notlogin")){
				return "ログインが完了していません。";
			}else if(errorCode.equals("unknown")){
				return "未知のエラーが発生しました。";
			}else{
				Logger.write("getpublishstatus UnknownErrorCode:" + errorCode);
				return "未定義のエラーが発生しました";
			}
		}else{
			return "エラーはありません";
		}
	}

	@Override
	public boolean hasError() {
		String status = select("getpublishstatus/@status");
		return status == null || !status.equals("ok");
	}

}
