package nicolib.api.live;

import java.util.Date;

import nicolib.api.AsyncLoadHandler;
import nicolib.api.NicoApiException;
import nicolib.api.XmlApiBase;
import nicolib.util.Logger;

/**
 * 生放送接続情報取得用API
 * @author hal
 *
 */
public class PlayerStatus extends XmlApiBase {
	
	private static final String API_URLFORMAT = "http://live.nicovideo.jp/api/getplayerstatus?v=%s";
	
	/**
	 * 指定した放送IDの接続情報を取得する
	 * [APIUrl] http://live.nicovideo.jp/api/getplayerstatus?v=[放送ID]
	 * @param id 放送ID（URL形式でもよい）
	 * @return PlayerStatus
	 * @throws NicoApiException
	 */
	public static PlayerStatus get(String id) throws NicoApiException{
		String lid = checkAndGetLiveId(id);
		PlayerStatus status = new PlayerStatus();
		status.access(String.format(API_URLFORMAT, lid));
		
		return status;
	}
	
	/**
	 * 指定した放送IDの接続情報を非同期で取得する
	 * [APIUrl] http://live.nicovideo.jp/api/getplayerstatus?v=[放送ID]
	 * @param id
	 * @param handler
	 * @throws NicoApiException
	 */
	public static void getAsync(String id, AsyncLoadHandler handler) 
		throws NicoApiException
	{
		String lid = checkAndGetLiveId(id);
		PlayerStatus status = new PlayerStatus();
		status.accessAsync(String.format(API_URLFORMAT, lid), handler);
	}
	
	private PlayerStatus() 
	{
	}
	
	/**
	 * 放送ID
	 * @return
	 */
	public String getLiveId(){
		return select("getplayerstatus/stream/id");
	}
	
	/**
	 * 来場者数
	 * @return
	 */
	public int getWatchCount(){
		return selectInt("getplayerstatus/stream/watch_count", 0);
	}
	
	/**
	 * コメント数
	 * @return
	 */
	public int getCommentCount(){
		return selectInt("getplayerstatus/stream/comment_count", 0);
	}
	
	/**
	 * 男女コメントモード
	 * @return
	 */
	public int getDanjoCommentMode(){
		return selectInt("getplayerstatus/stream/danjo_comment_mode", 0);
	}

	public int getRelayComment(){
		return selectInt("getplayerstatus/stream/relay_comment", 0);
	}

	public int getCommentLock(){
		return selectInt("getplayerstatus/stream/comment_lock", 0);
	}

	public String getProviderType(){
		return select("getplayerstatus/stream/provider_type");
	}
	
	public String getDefaultCommunity(){
		return select("getplayerstatus/stream/default_community");
	}
	
	/**
	 * 基準時間 (vposなどはこの時間からカウントされる)
	 * @return
	 */
	public Date getBaseTime(){
		return selectDate("getplayerstatus/stream/base_time");
	}
	
	/**
	 * 放送ページが開けるようになる時間
	 * @return
	 */
	public Date getOpenTime(){
		return selectDate("getplayerstatus/stream/open_time");
	}
	
	/**
	 * 放送開始時間
	 * @return
	 */
	public Date getStartTime(){
		return selectDate("getplayerstatus/stream/start_time");
	}
	
	/**
	 * 部屋名
	 * @return
	 */
	public String getRoomLabel(){
		return select("getplayerstatus/user/room_label");
	}
	
	/**
	 * 座席番号
	 * @return
	 */
	public int getRoomSeetNo(){
		return selectInt("getplayerstatus/user/room_seetno", 0);
	}
	
	/**
	 * APIを取得した人の年齢
	 * @return
	 */
	public int getUserAge(){
		return selectInt("getplayerstatus/user/userAge", 0);
	}
	
	/**
	 * APIを取得した人の性別
	 * @return
	 */
	public int getUserSex(){
		return selectInt("getplayerstatus/user/userSex", 0);
	}
	
	/**
	 * APIを取得した人の住んでいる都道府県
	 * @return
	 */
	public int getUserPrefecture(){
		return selectInt("getplayerstatus/user/userPrefecture", 0);
	}
	
	/**
	 * APIを取得した人のユーザー名
	 * @return
	 */
	public String getNickname(){
		return select("getplayerstatus/user/nickname");
	}
	
	/**
	 * プレミアム会員かどうか
	 * @return
	 */
	public int getIsPremium(){
		return selectInt("getplayerstatus/user/is_premium", 0);
	}
	
	/**
	 * APIを取得した人のユーザーID
	 * @return
	 */
	public int getUserId(){
		return selectInt("getplayerstatus/user/user_id", 0);
	}
	
	/**
	 * 放送コミュニティに参加しているか
	 * @return
	 */
	public int getIsJoin(){
		return selectInt("getplayerstatus/user/is_join", 0);
	}
	
	/**
	 * RTMPのURL
	 * @return
	 */
	public String getRtmpUrl(){
		return select("getplayerstatus/rtmp/url");
	}
	
	/**
	 * RTMP通信用のチケット
	 * @return
	 */
	public String getRtmpTicket(){
		return select("getplayerstatus/rtmp/Ticket");
	}
	
	/**
	 * コメントサーバーのアドレス
	 * @return
	 */
	public String getAddr(){
		return select("getplayerstatus/ms/addr");
	}
	
	/**
	 * コメントサーバーのポート
	 * @return
	 */
	public int getPort(){
		return selectInt("getplayerstatus/ms/port", 0);
	}
	
	/**
	 * コメントのスレッド
	 * @return
	 */
	public int getThread(){
		return selectInt("getplayerstatus/ms/thread", 0);
	}
	
	@Override
	public String getErrorCode() {
		return select("getplayerstatus/error/code");
	}
	
	@Override
	public String getErrorMessage() {
		String errorCode = getErrorCode();
		
		if(hasError()){
			if(errorCode == null){
				return "getplayerstatusの取得に失敗しました。";
			}else if(errorCode.equals("closed")){
				return "放送は終了しています。";
			}else if(errorCode.equals("notfound")){
				return "放送が見つかりませんでした。";
			}else if(errorCode.equals("notlogin")){
				return "ログインが完了していません。";
			}else if(errorCode.equals("full")){
				return "満員のため放送にアクセスできませんでした。";
			}else if(errorCode.equals("require_community_member")){
				return "メンバー限定のためアクセスできませんでした。";
			}else{
				Logger.write("PlayerStatus UnknownErrorCode:" + errorCode);
				return "未定義のエラーが発生しました";
			}
		}else{
			return "エラーはありません";
		}
	}

	@Override
	public boolean hasError() {
		String status = select("getplayerstatus/@status");
		return status == null || !status.equals("ok");
	}

}
