package nicolib.api.alert;

import nicolib.api.ApiBase;
import nicolib.api.AsyncLoadHandler;
import nicolib.api.NicoApiException;
import nicolib.util.Logger;

/**
 * アラート認証API
 * @author hal
 *
 */
public class AlertStatus extends AlertInfo {
	
	private static final String apiUrl = "http://live.nicovideo.jp/api/getalertstatus";
	private static final String paramFormat = "ticket=%s";
	
	
	/**
	 * アカウント情報からAlertStatusを取得する
	 * [APIUrl] http://live.nicovideo.jp/api/getalertstatus
	 * @param mail メールアドレス
	 * @param password パスワード
	 * @return　AlertStatus
	 * @throws NicoApiException
	 */
	public static AlertStatus get(String mail, String password) throws NicoApiException{
		AlertLogin login = AlertLogin.get(mail, password);
		
		if(login.hasError()){
			throw new NicoApiException("ログインに失敗しました。");
		}
		
		return AlertStatus.get(login);
	}
		
	/**
	 * 事前に取得したAlertLoginを元にAlertStatusを取得する
	 * [APIUrl] http://live.nicovideo.jp/api/getalertstatus
	 * @param login 事前に取得したログイン情報
	 * @return　AlertStatus
	 * @throws NicoApiException
	 */
	public static AlertStatus get(AlertLogin login) throws NicoApiException{
		
		if(login == null){
			throw new IllegalArgumentException("loginがnullです。");
		}
		
		if(login.hasError()){
			throw new NicoApiException("ログインに失敗しています。");
		}
		
		AlertStatus status = new AlertStatus();
		status.access(apiUrl, String.format(paramFormat, login.getTicket()), null);
		
		return status;
	}
	
	/**
	 * アカウント情報からAlertStatusを非同期で取得する
	 * [APIUrl] http://live.nicovideo.jp/api/getalertstatus
	 * @param mail メールアドレス
	 * @param password パスワード
	 * @param handler
	 * @throws NicoApiException
	 */
	public static void getAsync(String mail, String password, final AsyncLoadHandler handler) throws NicoApiException{
		AlertLogin.getAsync(mail, password, new AsyncLoadHandler() {
			
			@Override
			public void onLoaded(ApiBase api) {
				try {
					AlertStatus.getAsync((AlertLogin)api, handler);
				} catch (NicoApiException e) {
					handler.onError(e);
				}
			}
			
			@Override
			public void onError(NicoApiException e) {
				handler.onError(e);
			}
		});
	}
	
	/**
	 * 事前に取得したAlertLoginを元にAlertStatusを非同期で取得する
	 * [APIUrl] http://live.nicovideo.jp/api/getalertstatus
	 * @param login
	 * @param handler
	 * @throws NicoApiException
	 */
	public static void getAsync(AlertLogin login, AsyncLoadHandler handler) throws NicoApiException{
		if(login == null){
			throw new IllegalArgumentException("loginがnullです。");
		}
		
		if(login.hasError()){
			throw new NicoApiException("ログインに失敗しています。");
		}
		
		AlertStatus status = new AlertStatus();
		status.accessAsync(apiUrl, String.format(paramFormat, login.getTicket()), null, handler);
		
	}
	
	
	private AlertStatus(){
	}
	
	/**
	 * アカウント名
	 * @return
	 */
	public String getUserName(){
		return select("getalertstatus/user_name");
	}
	
	/**
	 * 都道府県情報
	 * @return
	 */
	public int getUserPrefecture(){
		return selectInt("getalertstatus/user_prefecture", 0);
	}
	
	/**
	 * 年齢
	 * @return
	 */
	public int getAge(){
		return selectInt("getalertstatus/user_age", 0);
	}
	
	/**
	 * 性別
	 * @return
	 */
	public int getSex(){
		return selectInt("getalertstatus/user_sex", 0);
	}
	
	/**
	 * ユーザーが所属しているコミュニティ一覧
	 * @return
	 */
	public String[] getCommunityIds(){
		return selectStrings("getalertstatus/communities/community_id");
	}

	@Override
	public String getErrorMessage() {
		String errorCode = getErrorCode();
		
		if(hasError()){
			if(errorCode == null){
				return "AlertStatusの取得に失敗しました。";
			}else if(errorCode.equals("incorrect_account_data")){
				return "ログイン情報が正しくありません。";
			}else{
				Logger.write("AlertStatus UnknownErrorCode:" + errorCode);
				return "未定義のエラーが発生しました";
			}
		}else{
			return "エラーはありません";
		}
	}

}
