package nicolib.api;

import java.io.ByteArrayInputStream;
import java.io.IOException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import nicolib.util.Http;
import nicolib.util.Logger;

import org.w3c.dom.Document;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;


/**
 * XML系APIの抽象クラス
 * APIはxmlドキュメントとして保持し、適時XPathにより値を取得する
 * @author hal
 */
public abstract class XmlApiBase extends ApiBase{
	
	protected static DocumentBuilder builder;
	protected static XPath xpath;
	
	static{
		DocumentBuilderFactory dbfactory = DocumentBuilderFactory.newInstance();
		try {
			builder = dbfactory.newDocumentBuilder();
		} catch (ParserConfigurationException e) {
			Logger.writeException(e, true);
		}
		
		XPathFactory xpfactory = XPathFactory.newInstance();
		xpath= xpfactory.newXPath();
	}
	
	protected Document xdoc;
	
	/**
	 * 指定したURLのAPIへアクセスする
	 * 失敗時には既定では３回まで接続を試みる
	 * @param url
	 * @throws NicoApiException
	 */
	protected void access(String url)
		throws NicoApiException
	{
		try {
			String res = Http.TryGet(url, retryCount);
			if(res.equals("")) throw new NicoApiException(String.format("API(%s)の取得に失敗しました。", url));
			
			ByteArrayInputStream bais = new ByteArrayInputStream(res.getBytes("UTF-8"));		
			xdoc = builder.parse(bais);
		} catch (SAXException e) {
			throw new NicoApiException(String.format("XML(%s)の解析に失敗しました", url), e);
		} catch (IOException e) {
			throw new NicoApiException(String.format("XML(%s)の取得に失敗しました", url), e);
		}
	}
	
	/**
	 * 指定したURLのAPIへPOSTでアクセスする
	 * 失敗時には既定では３回まで接続を試みる
	 * @param url
	 * @param param
	 * @param referer
	 * @throws NicoApiException
	 */
	protected void access(String url, String param, String referer)
		throws NicoApiException
	{
		try {
			String res = Http.TryPost(url, param, referer, retryCount);
			if(res.equals("")) throw new NicoApiException(String.format("API(%s)の取得に失敗しました。", url));
			
			ByteArrayInputStream bais = new ByteArrayInputStream(res.getBytes("UTF-8"));		
			xdoc = builder.parse(bais);
									
		} catch (SAXException e) {
			throw new NicoApiException(String.format("XML(%s)の解析に失敗しました", url), e);
		} catch (IOException e) {
			throw new NicoApiException(String.format("XML(%s)の取得に失敗しました", url), e);
		}
	}
	
	/**
	 * XPathで指定された位置にある文字データを取得する
	 * @param location
	 * @return 文字列　データが見つからない場合はnullを返す
	 */
	public String select(String location) {
		try{
			return xpath.evaluate(location, xdoc);
		}catch(XPathExpressionException e){
			Logger.writeException(e);
			return null;
		}
	}	
	
	/**
	 * XPathで指定された位置にある文字列を全て取得する
	 * @param location
	 * @return 見つからなかった場合はnullを返す
	 */
	public String[] selectStrings(String location){
		try{
			XPathFactory xpfactory = XPathFactory.newInstance();
			XPath xpath = xpfactory.newXPath();
			
			NodeList entries = (NodeList)xpath.evaluate(location, xdoc, XPathConstants.NODESET);
			String[] results = new String[entries.getLength()];
			for(int i = 0; i < entries.getLength(); i++){
				results[i] = entries.item(i).getTextContent();
			}
			
			return results;		
		}catch(XPathExpressionException e){
			Logger.writeException(e);
			return null;
		}
	}
	
	/**
	 * 取得したAPIがエラーデータでないか調べる
	 * @return エラーが発生しているか
	 */
	public abstract boolean hasError();
	
	/**
	 * 取得したAPIのエラーコードを取得する
	 * @return　エラーが発生していない場合はnullを返す
	 */
	public abstract String getErrorCode();
	
	/**
	 * エラーの説明文を取得する
	 * （error/descriptionは値自体がなかったりエラーコードと同一だったりするので参照しない）
	 * @return
	 */
	public abstract String getErrorMessage();

}
