﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Net;
using CommonSystem;
using Nicolib.API;
using Nicolib.Cookie.BrowserCookie;

namespace Nicolib.Cookie {
	/// <summary>
	/// ニコニコのセッションを取得します。
	/// </summary>
	public static class Session {

		/// <summary>
		/// ニコニコ関係のドメインを定義します。
		/// </summary>
		private static string[] Domains = {
			"live.nicovideo.jp",
			"watch.live.nicovideo.jp",
			"www.nicovideo.jp",
			"ch.nicovideo.jp",
			"com.nicovideo.jp",
			"nicovideo.jp",
			"seiga.nicovideo.jp",
			"dvd.nicovideo.jp"
		};

		private static int checkSessionWait  = 0;

		/// <summary>
		/// ニコニコのアカウント
		/// </summary>
		internal static string Account {
			get;
			set;
		}
		/// <summary>
		/// ニコニコのパスワード
		/// </summary>
		internal static string Password {
			get;
			set;
		}

		/// <summary>
		/// ブラウザからCookieの取得を試みます。
		/// </summary>
		private static CookieBase cookie {
			get;
			set;
		}

		/// <summary>
		/// ブラウザからセッションを取得します。
		/// 最後にログイン成功したブラウザから優先してセッションを取得します。
		/// セッションが必要な場合は、このメソッドを呼び出して下さい。
		/// </summary>
		/// <returns>true=セッションが取得できた。false=セッションを取得できなかった</returns>
		public static bool AutoGet() {
			if(SetNicoNicosession( false , GetTryOnBrowser )) {
				if(TryOnEvent != null) {
					TryOnEvent( null , EventArgs.Empty );
				}
				return ( true );
			} else {
				GetTryOnBrowser = Browser.Error;
				if(TryOnEvent != null) {
					TryOnEvent( null , EventArgs.Empty );
				}
				return ( false );
			}
		}
		/// <summary>
		/// ブラウザからニコニコ動画のセッションを保持したクッキーを取得する。
		/// CookieContainer NicoSession_Cookieにセットされる。
		/// </summary>
		/// <param field="Refresh">強制的にセッションを取得し直す</param>
		/// <param field="SelectBrowser">はじめに試みるブラウザ</param>
		/// <returns>トライオンできているかどうか</returns>
		public static bool SetNicoNicosession(bool Refresh , Browser SelectBrowser) {
			// 既にセッションが取得済みでないかチェックします。
			if(Refresh == false && GetTryOnBrowser == Browser.Error) {
				if(Environment.TickCount - checkSessionWait >= 10000) {
					checkSessionWait = Environment.TickCount;
					if(CheckSession()) {
						return ( true );
					}
				} else {
					return ( true );
				}
			}

			while(true) {

				#region ブラウザからセッションの取得を試みます。
				if(SelectBrowser == Browser.Auto || SelectBrowser == Browser.FireFox) {
					cookie = new FirefoxCookieGet();
					if(cookie.SessionGet()) {
						SelectBrowser = Browser.FireFox;
						break;
					}
				}
				if(SelectBrowser == Browser.Auto || SelectBrowser == Browser.Opera) {
					cookie = new OperaCookieGet();
					if(cookie.SessionGet()) {
						SelectBrowser = Browser.Opera;
						break;
					}
				}
				if(SelectBrowser == Browser.Auto || SelectBrowser == Browser.Chrome) {
					cookie = new ChromeCookieGet();
					if(cookie.SessionGet()) {
						SelectBrowser = Browser.Chrome;
						break;
					}
				}
				if(SelectBrowser == Browser.Auto || SelectBrowser == Browser.Safari) {
					cookie = new SafariCookieGet();
					if(cookie.SessionGet()) {
						SelectBrowser = Browser.Chrome;
						break;
					}
				}
				if(SelectBrowser == Browser.Auto || SelectBrowser == Browser.InternetExplorer) {
					cookie = new InternetExplorerCookieGet();
					if(cookie.SessionGet()) {
						SelectBrowser = Browser.InternetExplorer;
						break;
					}
				}
				#endregion

				if(SelectBrowser != Browser.Auto) {
					if(SelectBrowser == Browser.Auto || SelectBrowser == Browser.Library) {
						if(Account.Length > 0 && Password.Length > 0 && NicoAPI.Login( Account , Password )) {
							SelectBrowser = Browser.Library;
							break;
						} else {
							SelectBrowser = Browser.Error;
						}
					}
					if(SelectBrowser == Browser.Auto) {
						SelectBrowser = Browser.Error;
						break;
					} else {
						SelectBrowser = Browser.Auto;
					}
				} else {
					break;
				}
			}
			GetTryOnBrowser = SelectBrowser;
			if(TryOnEvent != null) {
				TryOnEvent( null , EventArgs.Empty );
			}
			return ( SelectBrowser != Browser.Error );
		}
		/// <summary>
		/// 現在、トライオンしているブラウザの種類を取得します。
		/// </summary>
		public static Browser GetTryOnBrowser {
			get;
			private set;
		}

		/// <summary>
		/// 現在のセッションが有効かどうかチェックします。
		/// </summary>
		/// <returns></returns>
		public static bool CheckSession() {
			string Test = HTTP.Get( "http://live.nicovideo.jp/api/getflv?v=lv999999" );
			if(Test == null || Test.Contains( "notlogin" )) {
				Test = HTTP.Get( "http://www.nicovideo.jp/" );
				if(Test != null && Test.Contains( "https://secure.nicovideo.jp/secure/login_form" ) == false) {
					return ( true );
				}
				return ( false );
			}
			return ( true );
		}

		/// <summary>
		/// セッションを登録します。
		/// IEのCookieをWIN32-APIを通じて更新します。
		/// </summary>
		/// <param field="session">セッション文字列</param>
		/// <returns>true = 有効なセッション</returns>
		internal static bool SessionSet(string session) {
			if(session != null && session.Length > 0) {
				string cookieText = "user_session={0};expires = Fri,10-Sep-2030 00:00:00 GMT";
				CookieContainer cookie = new CookieContainer();
				foreach(string domain in Domains) {
					cookie.Add( new System.Net.Cookie( "user_session" , session , "/" , domain ) );
				}
				HTTP.SetCookie( cookie );
				if(Session.CheckSession()) {	// ←セッションが正しいかどうかチェックします。
					// APIを通じてIE に書き込む
					foreach(string domain in Domains) {
						WIN32API.InternetSetCookie( string.Format( "http://{0}/" , domain ) , null , string.Format( cookieText , session ) );
					}
					return ( true );
				}
			}
			return ( false );
		}

		/// <summary>
		/// トライオン(ブラウザ)の共有が変更されたら通知します。
		/// </summary>
		public static event EventHandler TryOnEvent = null;
	}

	/// <summary>
	/// ブラウザの種類
	/// </summary>
	public enum Browser {
		/// <summary>
		/// IE
		/// </summary>
		InternetExplorer ,
		/// <summary>
		/// Opera
		/// </summary>
		Opera ,
		/// <summary>
		/// FireFox
		/// </summary>
		FireFox ,
		/// <summary>
		/// Chrome
		/// </summary>
		Chrome ,
		/// <summary>
		/// Safari
		/// </summary>
		Safari ,
		/// <summary>
		/// 自動
		/// </summary>
		Auto ,
		/// <summary>
		/// ライブラリ自信がログイン
		/// </summary>
		Library ,
		/// <summary>
		/// エラー
		/// </summary>
		Error
	}
}
