/* instruction.cpp
   Copyright (C) 2005 Free Software Foundation, Inc.

This file is part of Mysaifu JVM

Mysaifu JVM is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; version 2 of the License.

Mysaifu JVM is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.
*/

#include "StdAfx.h"
#include "ClassFile.h"
#include "frame.h"
#include "java_object.h"
#include "class_loader.h"
#include "instruction.h"
#include "jni_funcs.h"
#include "monitor.h"
#include "java_utf8.h"
#include "common_funcs.h"
#include "Profiler.h"
#include <Armintr.h>
#include <jvmti.h>

// fobO}N
#ifdef DEBUG
void
dbg_utf8 (const java_utf8 * msg)
{
  int bufflen = strlen (msg) * 2 + 1;
  TCHAR *buff = (TCHAR *) alloca (sizeof (TCHAR) * bufflen);
  convert_to_TCHAR (msg, buff, bufflen);
  DBG (buff);
}

void
dbg_int (int i)
{
  int bufflen = 12;
  TCHAR *buff = (TCHAR *) alloca (sizeof (TCHAR) * bufflen);
  _stprintf (buff, _T ("%i"), i);
  DBG (buff);
}

void
dbg_uint (unsigned int i)
{
  int bufflen = 12;
  TCHAR *buff = (TCHAR *) alloca (sizeof (TCHAR) * bufflen);
  _stprintf (buff, _T ("%u"), i);
  DBG (buff);
}

static TCHAR *g_instruction_names[256] = {
  _T ("nop"),
  _T ("aconst_null"),
  _T ("iconst_m1"),
  _T ("iconst_0"),
  _T ("iconst_1"),
  _T ("iconst_2"),
  _T ("iconst_3"),
  _T ("iconst_4"),
  _T ("iconst_5"),
  _T ("lconst_0"),
  _T ("lconst_1"),
  _T ("fconst_0"),
  _T ("fconst_1"),
  _T ("fconst_2"),
  _T ("dconst_0"),
  _T ("dconst_1"),
  _T ("bipush"),
  _T ("sipush"),
  _T ("ldc"),
  _T ("ldc_w"),
  _T ("ldc2_w"),
  _T ("iload"),
  _T ("lload"),
  _T ("fload"),
  _T ("dload"),
  _T ("aload"),
  _T ("iload_0"),
  _T ("iload_1"),
  _T ("iload_2"),
  _T ("iload_3"),
  _T ("lload_0"),
  _T ("lload_1"),
  _T ("lload_2"),
  _T ("lload_3"),
  _T ("fload_0"),
  _T ("fload_1"),
  _T ("fload_2"),
  _T ("fload_3"),
  _T ("dload_0"),
  _T ("dload_1"),
  _T ("dload_2"),
  _T ("dload_3"),
  _T ("aload_0"),
  _T ("aload_1"),
  _T ("aload_2"),
  _T ("aload_3"),
  _T ("iaload"),
  _T ("laload"),
  _T ("faload"),
  _T ("daload"),
  _T ("aaload"),
  _T ("baload"),
  _T ("caload"),
  _T ("saload"),
  _T ("istore"),
  _T ("lstore"),
  _T ("fstore"),
  _T ("dstore"),
  _T ("astore"),
  _T ("istore_0"),
  _T ("istore_1"),
  _T ("istore_2"),
  _T ("istore_3"),
  _T ("lstore_0"),
  _T ("lstore_1"),
  _T ("lstore_2"),
  _T ("lstore_3"),
  _T ("fstore_0"),
  _T ("fstore_1"),
  _T ("fstore_2"),
  _T ("fstore_3"),
  _T ("dstore_0"),
  _T ("dstore_1"),
  _T ("dstore_2"),
  _T ("dstore_3"),
  _T ("astore_0"),
  _T ("astore_1"),
  _T ("astore_2"),
  _T ("astore_3"),
  _T ("iastore"),
  _T ("lastore"),
  _T ("fastore"),
  _T ("dastore"),
  _T ("aastore"),
  _T ("bastore"),
  _T ("castore"),
  _T ("sastore"),
  _T ("pop"),
  _T ("pop2"),
  _T ("dup"),
  _T ("dup_x1"),
  _T ("dup_x2"),
  _T ("dup2"),
  _T ("dup2_x1"),
  _T ("dup2_x2"),
  _T ("swap"),
  _T ("iadd"),
  _T ("ladd"),
  _T ("fadd"),
  _T ("dadd"),
  _T ("isub"),
  _T ("lsub"),
  _T ("fsub"),
  _T ("dsub"),
  _T ("imul"),
  _T ("lmul"),
  _T ("fmul"),
  _T ("dmul"),
  _T ("idiv"),
  _T ("ldiv"),
  _T ("fdiv"),
  _T ("ddiv"),
  _T ("irem"),
  _T ("lrem"),
  _T ("frem"),
  _T ("drem"),
  _T ("ineg"),
  _T ("lneg"),
  _T ("fneg"),
  _T ("dneg"),
  _T ("ishl"),
  _T ("lshl"),
  _T ("ishr"),
  _T ("lshr"),
  _T ("iushr"),
  _T ("lushr"),
  _T ("iand"),
  _T ("land"),
  _T ("ior"),
  _T ("lor"),
  _T ("ixor"),
  _T ("lxor"),
  _T ("iinc"),
  _T ("i2l"),
  _T ("i2f"),
  _T ("i2d"),
  _T ("l2i"),
  _T ("l2f"),
  _T ("l2d"),
  _T ("f2i"),
  _T ("f2l"),
  _T ("f2d"),
  _T ("d2i"),
  _T ("d2l"),
  _T ("d2f"),
  _T ("i2b"),
  _T ("i2c"),
  _T ("i2s"),
  _T ("lcmp"),
  _T ("fcmpl"),
  _T ("fcmpg"),
  _T ("dcmpl"),
  _T ("dcmpg"),
  _T ("ifeq"),
  _T ("ifne"),
  _T ("iflt"),
  _T ("ifge"),
  _T ("ifgt"),
  _T ("ifle"),
  _T ("if_icmpeq"),
  _T ("if_icmpne"),
  _T ("if_icmplt"),
  _T ("if_icmpge"),
  _T ("if_icmpgt"),
  _T ("if_icmple"),
  _T ("if_acmpeq"),
  _T ("if_acmpne"),
  _T ("goto"),
  _T ("jsr"),
  _T ("ret"),
  _T ("tableswitch"),
  _T ("lookupswitch"),
  _T ("ireturn"),
  _T ("lreturn"),
  _T ("freturn"),
  _T ("dreturn"),
  _T ("areturn"),
  _T ("return"),
  _T ("getstatic"),
  _T ("putstatic"),
  _T ("getfield"),
  _T ("putfield"),
  _T ("invokevirtual"),
  _T ("invokespecial"),
  _T ("invokestatic"),
  _T ("invokeinterface"),
  _T ("xxxunusedxxx1"),
  _T ("new"),
  _T ("newarray"),
  _T ("anewarray"),
  _T ("arraylength"),
  _T ("athrow"),
  _T ("checkcast"),
  _T ("instanceof"),
  _T ("monitorenter"),
  _T ("monitorexit"),
  _T ("wide"),
  _T ("multianewarray"),
  _T ("ifnull"),
  _T ("ifnonnull"),
  _T ("goto_w"),
  _T ("jsr_w"),
  _T ("breakpoint"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("(UNDEFINED)"),
  _T ("impdep1"),
  _T ("impdep2"),
};

#endif /* DEBUG */

#ifdef DEBUG
/**
 * checkcast߂̓v擾ꍇɂ͂OȊO̒l`
 */
#define CHECKCAST_STATISTICS	0

#if CHECKCAST_STATISTICS
 // checkcast߂̓v
static unsigned int g_checkcast_count;
static unsigned int g_checkcast_hits;
#endif /* CHECKCAST_STATISTICS */

#endif /* DEBUG */

/**
 * OutOfMemoryError̃CX^X
 * JVMNɊmۂĂ
 */
static jthrowable g_OutOfMemoryError_instance;

/**
 * StackOverflowError̃CX^X
 * JVMNɊmۂĂ
 */
static jthrowable g_StackOverflowError_instance;

// static֐̐錾
static unsigned int find_exception_handler (frame * rt,
					    Code_attribute * code_attr);
static void run_code (frame * rt);
static void invoke_native_method (frame * caller_frame,
				  ClassFile * target_class_file,
				  method_info * target_method_info);
static jarray create_multiarray (frame * current_frame, class_loader * loader,
				 const char *class_name, u1 dimensions,
				 __int32 * counts, frame * tmp_frame);
static bool run_code (ClassFile * cfile, java_utf8 * name);
static jstring alloc_utf8_string (frame * frame, const java_utf8 * utf8,
				  bool permanent);

/**
 * w肳ꂽ CONSTANT_Fieldref_info ς݂ǂԂB
 */
static inline bool is_resolved(const CONSTANT_Fieldref_info* fieldref)
{
  return fieldref->finfo != NULL;
}


/**
 * w肳ꂽ CONSTANT_Fieldref_info 
 */
static bool resolve(frame* current_frame, CONSTANT_Fieldref_info* fieldref, bool instance_field);

/**
 * ߎs֐
 */
static void execute_tableswitch (frame * frame, u1 * bytecode);
static void execute_lookupswitch (frame * frame, u1 * bytecode);
// static void execute_getstatic (frame * frame, u1 * bytecode);
// static void execute_putstatic (frame * frame, u1 * bytecode);
static void execute_arraylength (frame * frame, u1 * bytecode);
static void execute_athrow (frame * frame, u1 * bytecode);
static void execute_wide (frame * frame, u1 * bytecode);
static void execute_multianewarray (frame * frame, u1 * bytecode);
static inline void execute_ldc_w (frame * current_frame, u1 index1,
				  u1 index2);
static void execute_jsr (frame * current_frame, __int32 branch);

/**
 * ߎs֐p[eBeB֐
 */
#define GET_PARAMETER(bytecode, index) bytecode[index]

#define MAKE_INT16(index1, index2) \
	((__int16) index1 << 8) | (index2 & 0x00ff)

#define MAKE_U2(index1, index2) \
	(((u2) index1 << 8) | (index2 & 0x00ff))

#define MAKE_INT32(b1, b2, b3, b4) \
	(((__int32) b1 << 24) | ((__int32) b2 << 16) | ((__int32) b3 << 8) | ((__int32) b4))

#define READ_BIG_ENDIAN(value) \
	((value << 24) | ((value & 0xff00) << 8) | ((value & 0xff0000) >> 8) | ((value & 0xff000000) >> 24))

#define EXCEPTION_OCCURRED(current_frame)	(current_frame->body->current_exception)

#define NO_EXCEPTION_HANDLER	(0xffffffff)

/**
 * staticϐ
 */
static const java_utf8 *g_Math_class_name;
static const java_utf8 *g_Math_sin_method_name;
static const java_utf8 *g_Math_cos_method_name;
static const java_utf8 *g_Math_sqrt_method_name;

/**
 * ߎs
 */
void
init_instruction_settings ()
{
  // Math֘ANX̍s
  g_Math_class_name = intern_utf8 ("java/lang/Math");
  g_Math_sin_method_name = intern_utf8 ("sin");
  g_Math_cos_method_name = intern_utf8 ("cos");
  g_Math_sqrt_method_name = intern_utf8 ("sqrt");
}

/**
 * OutOfMemoryError̃CX^XĂ
 */
bool
prepare_OutOfMemoryError ()
{
  frame *frm = alloc_root_frame (NULL);
  ClassFile *cfile =
    find_ClassFile (frm, intern_utf8 ("java/lang/OutOfMemoryError"));
  if (cfile == NULL)
    {
      return false;
    }
  // mۂAIyhX^bNɃt@Xς
  jobject obj;
  if ((obj = alloc_object (cfile, frm)) == NULL)
    {
      return false;
    }
  DUP_STACK (frm);

  // ȂRXgN^T
  method_info *constructor =
    find_declaring_method_info (cfile, INIT_METHOD_NAME,
				VOID_NOARG_METHOD_DESCRIPTOR);
  if (constructor == NULL)
    {
      POP2_DISCARD (frm);
      return false;
    }

  // RXgN^N
  invoke_method (frm, obj, cfile, constructor);

  // QƂO[oϐɓĂ
  g_OutOfMemoryError_instance = (jthrowable) new_global_reference (obj);
  POP_DISCARD (frm);

  free_root_frame (frm);

  return true;
}

/**
 * StackOverflowError̃CX^XĂ
 */
bool
prepare_StackOverflowError ()
{
  frame *frm = alloc_root_frame (NULL);
  ClassFile *cfile =
    find_ClassFile (frm, intern_utf8 ("java/lang/StackOverflowError"));
  if (cfile == NULL)
    {
      return false;
    }
  // mۂAIyhX^bNɃt@Xς
  jobject obj;
  if ((obj = alloc_object (cfile, frm)) == NULL)
    {
      return false;
    }
  DUP_STACK (frm);

  // ȂRXgN^T
  method_info *constructor =
    find_declaring_method_info (cfile, INIT_METHOD_NAME,
				VOID_NOARG_METHOD_DESCRIPTOR);
  if (constructor == NULL)
    {
      POP2_DISCARD (frm);
      return false;
    }

  // RXgN^N
  invoke_method (frm, obj, cfile, constructor);

  // QƂO[oϐɓĂ
  g_StackOverflowError_instance = (jthrowable) new_global_reference (obj);
  POP_DISCARD (frm);

  free_root_frame (frm);

  return true;
}

/**
 * w肳ꂽt[ŁApc߂s
 */
static void
run_code (frame * current_frame)
{
  const frame *start_frame = current_frame;
  u1 *code;
  u1 *bytecode;
  __int32 *stack_base;
  __int32 *local_variables_base;
  unsigned int *pposition;

	/**
	 * current_frame̓erun_code()Ɂu[hv
	 */
#define LOAD_CONTEXT() \
	{ \
		code = current_frame->current_method_info->code_attribute->code; \
		bytecode = code + current_frame->current_pc; \
		stack_base = current_frame->operand_stack; \
		local_variables_base = current_frame->local_variables; \
		pposition = &current_frame->operand_stack_position; \
	}

	/**
	 * run_code()̏current_frameɁuXgAv
	 */
#define STORE_CONTEXT() \
	{ \
		current_frame->current_pc = (int) (bytecode - code); \
	}

	/**
     * WvAhXݒ肷}N
	 *  condition  true ̏ꍇɂ̂݃Wv
	 */
#define SET_JUMP_ADDRESS(current_frame, condition, index1, index2, instruction_length) \
	{ \
		CHECK_EXCEPTION; \
		if (condition) { \
			__int16 offset = MAKE_INT16(index1, index2); \
			bytecode += offset; \
		} else { \
			bytecode += instruction_length; \
		} \
	}

  // O`FbNp}N
#define CHECK_EXCEPTION	\
	{\
		if (EXCEPTION_OCCURRED(current_frame)) { \
			while (true) { \
				/* OĂꍇAOnh */ \
				const unsigned int handler_pc = find_exception_handler(current_frame, current_frame->current_method_info->code_attribute); \
				if (handler_pc == NO_EXCEPTION_HANDLER) { \
					/* OnhȂꍇ̓X^bN1߂ */ \
					if (current_frame == start_frame) { \
						DBG(_T("Exception handler not found. return\n")); \
						goto END; \
					} \
					current_frame = leave_method(current_frame); \
					DBG(_T("Exception handler not found. rewind stack.\n")); \
					LOAD_CONTEXT(); \
				} else { \
					/* OnhL */ \
					bytecode = code + handler_pc; \
					/* IyhX^bNۂɂAOIuWFNgւ̎QƂpush */ \
					current_frame->operand_stack_position = 0; \
					PUSH_OBJECT(current_frame, \
								current_frame->body->current_exception); \
					/* ONA */ \
					exception_clear(current_frame); \
					break; \
				} \
			} \
			/* [v̐擪ɖ߂ */ \
			continue; \
		} \
	}

  // OThrowp}N
#define THROW(cname, msg) \
	{ \
		STORE_CONTEXT(); \
		throw_exception(current_frame, cname, msg); \
		CHECK_EXCEPTION; \
	}

  // \bhsp}N
#define INVOKE_METHOD(oref, target_class, target_method) \
	{ \
		if (is_native(target_method->access_flags)) { \
			STORE_CONTEXT(); \
			invoke_method(current_frame, oref, target_class, target_method); \
			CHECK_EXCEPTION; \
		} else { \
			/* Java\bh */ \
			frame* new_frame = enter_method(current_frame, target_class, target_method); \
			if (! new_frame) { \
				CHECK_EXCEPTION; \
			} else { \
				STORE_CONTEXT(); \
				current_frame = new_frame; \
				LOAD_CONTEXT(); \
				continue; \
			} \
		} \
	}

  //--- }N`I ---


  LOAD_CONTEXT ();

#ifdef ENABLE_PROFILER
        Profiler* profiler = Profiler::BYTECODE_PROFILER;
        LARGE_INTEGER startTime;
        profiler->queryTime(&startTime);
        unsigned int slot = 0;
#endif
  while (true)
    {
#ifdef ENABLE_PROFILER
        profiler->recordEnd(slot, &startTime);
        profiler->queryTime(&startTime);
        slot = (unsigned int)*bytecode;
#endif
      switch (*bytecode)
	{
	default:
	  DBG (_T ("Unknown instruction "));
	  DBG_UINT (*bytecode);
	  DBG (_T ("\n"));
	  assert (false);
	  fatal_error (_T ("Unknown instruction"));

	  // ʂ悭邽߁Aߎs֐{̂́Aʃt@CŒ`
#include "instruction_inc.h"

        }
    }

END:
  ;				// ߎsI

// ̊֐ł̂ݗLȃ}N
#undef SET_JUMP_ADDRESS
#undef CHECK_EXCEPTION

}

/**
 * ldc/ldc_wߋʂ̏s
 */
static inline void
execute_ldc_w (frame * current_frame, u1 index1, u1 index2)
{
  u2 index = MAKE_U2 (index1, index2);
  ClassFile *cfile = current_frame->current_class_file;
  cp_info *cp = cfile->constant_pool[index];

  switch (cfile->constant_pool_tags[index])
    {
    case CONSTANT_Integer:
    case CONSTANT_Float:
      {
	// l̂܂PUSH
	__int32 value = (__int32) cp;
	PUSH_INT (current_frame, value);
	break;
      }

    case CONSTANT_String:
      {
	jobject string_reference;
	if (IS_SPECIAL_POINTER (cp))
	  {
	    // jstringĂ
	    string_reference = (jstring) MAKE_NORMAL_POINTER (cp);
	  }
	else
	  {
	    // java_utf8* Ă
	    java_utf8 *utf = (java_utf8 *) cp;
	    string_reference = intern_string (utf, current_frame);
	    if (exception_occurred (current_frame))
	      {
		return;
	      }
	    // lLbV
	    cfile->constant_pool[index] =
	      (cp_info *) MAKE_SPECIAL_POINTER (string_reference);
	  }
	PUSH_OBJECT (current_frame, string_reference);

	break;
      }

    case CONSTANT_Class:
      {
	// Java5ldc/ldc_w߂CONSTANT_Class悤ɂȂ

	// ΏۂƂȂClassFile\̂ւ̃|C^𓾂
	ClassFile *target_class_file = resolve_ClassFile (current_frame,
							  cfile,
							  index);
	if (exception_occurred (current_frame))
	  {
	    // O
	    return;
	  }
	// ClassIuWFNg𐶐
	jobject clazz = find_class_object (current_frame,
					   get_static_data
					   (target_class_file),
					   NULL);
	PUSH_OBJECT (current_frame, clazz);

	break;
      }

    default:
      // Othrow
      throw_exception (current_frame, "java/lang/ClassFormatError",
		       "Illegal index in ldc/ldc_w");
      break;
    }
}

/**
 * w肳ꂽ̔z쐬
 * 쐬̔z񂪃Kx[WRN^ɂĉĂ܂Ȃ悤ɁAtmp_stack
 * 쐬zւ̎QƂpushB
 *
 * @param	current_frame	Jgt[
 * @param	loader		NX[_
 * @param	cname	쐬z̃NX
 * @param	dimensions	z̎
 * @param	counts		zȇ傫i[intzւ̃|C^
 * @param	tmp_stack	쐬̗vfꎞIpushĂX^bN
 */
static jarray
create_multiarray (frame * current_frame,
		   class_loader * loader,
		   const char *cname,
		   u1 dimensions, __int32 * counts, frame * tmp_stack)
{
  const java_utf8 *class_name = intern_utf8 (cname);
  ClassFile *cfile = find_ClassFile (current_frame, loader, class_name);
  if (cfile == NULL)
    {
      return NULL;
    }

  int count = *counts;
  jarray obj;
  if ((obj = alloc_array (cfile, count, tmp_stack)) == NULL)
    {
      return NULL;
    }

  if (dimensions > 1 && count > 0)
    {
      // vf̃NX쐬
      // [[I ̏ꍇ [I vf̃NXɂȂ
      u4 length = strlen (class_name) - 1;	// vf͂P
      char *component_class_name = (char *) malloc (length + 1);
      if (!component_class_name)
	{
	  fatal_error (FATAL_ERROR_NO_MEMORY);
	}
      strcpy (component_class_name, class_name + 1);

      int newdimensions = dimensions - 1;
      __int32 *newcounts = counts + 1;

      // vfċAIɍ쐬Ă
      for (int index = 0; index < count; ++index)
	{
	  jobject component;
	  if ((component = create_multiarray (current_frame,
					      loader,
					      component_class_name,
					      newdimensions,
					      newcounts, tmp_stack)) == NULL)
	    {
	      // ċAŃG[
	      free (component_class_name);
	      return NULL;
	    }
	  store_object_array (obj, index, component);
	}
      free (component_class_name);
    }
  return obj;
}

/**
 * native\bhs
 */
static void
invoke_native_method (frame * caller_frame, ClassFile * target_class_file,
		      method_info * target_method_info)
{
  void *func_ptr =
    find_native_proc_ptr (target_class_file, target_method_info);
  if (func_ptr == NULL)
    {
      // ΉlCeBu֐݂ȂꍇAUnsatisfiedLinkError  throw 
      int len =
	strlen (target_class_file->this_class_name) + 1 +
	strlen (target_method_info->name) + 1;
      char *buff = (char *) malloc (len);
      if (buff == NULL)
	{
	  fatal_error (FATAL_ERROR_NO_MEMORY);
	}
      strcpy (buff, target_class_file->this_class_name);
      strcat (buff, ".");
      strcat (buff, target_method_info->name);
      throw_exception (caller_frame, "java/lang/UnsatisfiedLinkError", buff);
      free (buff);
      return;
    }
  call_native_function (func_ptr, caller_frame, target_class_file,
			target_method_info);
}

/**
 * w肳ꂽ\bhs
 *
 * @param	caller_frame	Ăяõt[B
 * @param	ref				IuWFNg̎QƁB
 *                          CX^X\bh̏ꍇ̓CX^X̎QƂw肷B
 *							iX^eBbN\bh̏ꍇ͖j
 * @param	declaring_class_file		Y\bh`ĂClassFile\
 * @param	minfo			ĂяoΏۃ\bh
 */
void
invoke_method (frame * caller_frame,
	       jobject ref, ClassFile * declaring_file, method_info * minfo)
{
  assert (declaring_file == minfo->declaring_ClassFile);

  // AbstractMethod̏ꍇ́AO𔭐
  if (is_abstract (minfo->access_flags))
    {
      throw_exception (caller_frame, "java/lang/AbstractMethodError",
		       minfo->name);
      return;
    }

  // native\bh𔻒f
  if (is_native (minfo->access_flags))
    {
      // native\bhN
      invoke_native_method (caller_frame, declaring_file, minfo);
    }
  else
    {
      // t[
      frame *new_frame = enter_method (caller_frame, declaring_file, minfo);
      if (!new_frame)
	{
	  assert (false);
	  // ĂяoStackOverflowErrorm
	}
      else
	{
	  // R[hs
	  run_code (new_frame);
	  // t[ɖ߂
	  leave_method (new_frame);
	}
    }
}

/**
 * OnhAYpclԂ
 * ΉnhȂꍇANO_EXCEPTION_HANDLER Ԃ
 *
 */
static unsigned int
find_exception_handler (frame * rt, Code_attribute * code_attr)
{
  u2 exception_table_length = code_attr->exception_table_length;
  unsigned int pc = rt->current_pc;
  unsigned int handler_pc = NO_EXCEPTION_HANDLER;

  // Javã\bhĂяo邽߁AꎞIɗONAĂ
  jthrowable current_exception = rt->body->current_exception;
  jthrowable global_throwable = NULL;
  if (current_exception)
    {
      // Kx[WRNgȂ悤ɃO[oQƂƂĕێĂ
      global_throwable =
	(jthrowable) new_global_reference (current_exception);
    }
  rt->body->current_exception = NULL;

  for (u2 i = 0; i < exception_table_length; ++i)
    {
      exception_handler_info *handler = code_attr->exception_table + i;
      if (pc >= handler->start_pc && pc < handler->end_pc)
	{
	  if (handler->catch_type == 0)
	    {
	      // catches all exceptions.
	      DBG (_T ("Exception handler found at "));
	      DBG_UTF8 (rt->current_class_file->this_class_name);
	      DBG (_T ("."));
	      DBG_UTF8 (rt->current_method_info->name);
	      DBG (_T (" (all exceptions)\n"));
	      handler_pc = handler->handler_pc;
	      break;
	    }
	  else
	    {
	      // O𒲂ׂ
	      const java_utf8 *cname =
		get_CONSTANT_Class_info (rt->current_class_file,
					 handler->catch_type);
	      // catchnhClassFile[h
	      ClassFile *catch_cfile = find_ClassFile (rt,
						       get_defining_loader
						       (rt->
							current_class_file),
						       cname);
	      if (catch_cfile)
		{
		  if (is_assignable_from
		      (rt, get_ClassFile (current_exception), catch_cfile))
		    {
		      DBG (_T ("Exception handler found at "));
		      DBG_UTF8 (rt->current_class_file->this_class_name);
		      DBG (_T ("."));
		      DBG_UTF8 (rt->current_method_info->name);
		      DBG (_T (" Exception name="));
		      DBG_UTF8 (cname);
		      DBG (_T ("\n"));
		      handler_pc = handler->handler_pc;
		      break;
		    }
		}
	      else
		{
		  // NX̃[hɎs
		  DBG (_T ("Failed to load "));
		  DBG_UTF8 (cname);
		  DBG (_T ("\n"));
		  break;
		}
	    }
	}
    }

  // ޔĂOɖ߂
  if (current_exception)
    {
      rt->body->current_exception = current_exception;
      delete_global_reference (global_throwable);
    }
  return handler_pc;
}

/**
 * w肳ꂽOthrowꂽԂɂB
 * w肳ꂽNX java.lang.Throwable ̔hNXł邩̓`FbNȂ
 *
 * @param	current_frame	Jgt[
 * @param	cfile			NXt@C
 * @param	message			bZ[W
 * @return	ꍇtrueB
 */
bool
throw_exception (frame * current_frame, ClassFile * cfile,
		 const java_utf8 * message)
{
  // mۂAIyhX^bNɃt@Xς
  jobject obj;
  if ((obj = alloc_object (cfile, current_frame)) == NULL)
    {
      // ǂ悤Ȃ̂ŁAOutOfMemoryError𓊂ďI
      throw_OutOfMemoryError (current_frame);
      return false;
    }
  DUP_STACK (current_frame);

  // StringƂRXgN^s
  method_info *constructor = find_declaring_method_info (cfile,
							 INIT_METHOD_NAME,
							 STRING_ARG_CONSTRUCTOR_DESCRIPTOR);
  if (constructor != NULL)
    {
      // StringIuWFNg쐬iɃX^bNpushj
      if (message != NULL)
	{
	  alloc_utf8_string (current_frame, message, false);
	}
      else
	{
	  PUSH_OBJECT (current_frame, NULL);
	}
    }
  else
    {
      // StringɂƂRXgN^ȂꍇAȂRXgN^T
      constructor =
	find_declaring_method_info (cfile, INIT_METHOD_NAME,
				    VOID_NOARG_METHOD_DESCRIPTOR);
      if (constructor == NULL)
	{
	  POP2_DISCARD (current_frame);
	  return false;
	}
    }

  // RXgN^N
  invoke_method (current_frame, obj, cfile, constructor);

  // QƂJgt[ɓ
  bool result = throw_exception (current_frame, (jthrowable) obj);
  POP_DISCARD (current_frame);

  return result;
}

/**
 * w肳ꂽOthrowꂽԂɂB
 * w肳ꂽNX java.lang.Throwable ̔hNXł邩̓`FbNȂ
 *
 * @param	current_frame	Jgt[
 * @param	cfile			NXt@C
 * @param	param			p[^
 * @return	ꍇtrueB
 */
bool
throw_exception_int (frame * current_frame, ClassFile * cfile, jint param)
{
  // mۂAIyhX^bNɃt@Xς
  jobject obj;
  if ((obj = alloc_object (cfile, current_frame)) == NULL)
    {
      // ǂ悤Ȃ̂ŁAOutOfMemoryError𓊂ďI
      throw_OutOfMemoryError (current_frame);
      return false;
    }
  DUP_STACK (current_frame);

  // intƂRXgN^s
  method_info *constructor =
    find_declaring_method_info (cfile, INIT_METHOD_NAME,
				intern_utf8 ("(I)V"));
  if (constructor != NULL)
    {
      // p[^ς
      PUSH_INT (current_frame, param);
    }
  else
    {
      // intɂƂRXgN^ȂꍇAȂRXgN^T
      constructor =
	find_declaring_method_info (cfile, INIT_METHOD_NAME,
				    VOID_NOARG_METHOD_DESCRIPTOR);
      if (constructor == NULL)
	{
	  // ȂRXgN^Ȃꍇ́AG[^[
	  POP2_DISCARD (current_frame);
	  return false;
	}
    }

  // RXgN^N
  invoke_method (current_frame, obj, cfile, constructor);

  // QƂJgt[ɓ
  bool result = throw_exception (current_frame, (jthrowable) obj);
  POP_DISCARD (current_frame);

  return result;
}

bool
throw_exception (frame * current_frame, jthrowable t)
{
  current_frame->body->current_exception = t;
  return true;
}

/**
 * w肳ꂽO̗OthrowꂽԂɂB
 * w肳ꂽÕNX java.lang.Throwable ̔hNXł邩̓`FbNȂ߁A
 * ĂяoƂ̐ӔCɂȂB
 *
 * @param	current_frame	Jgt[
 * @param	throwÕNX
 * @return	ꍇtrueB
 */
bool
throw_exception (frame * current_frame, const char *classname)
{
  bool result =
    throw_exception (current_frame, classname, (java_utf8 *) NULL);
  return result;
}

bool
throw_exception (frame * current_frame, const char *classname,
		 const char *message)
{
  // ɗOĂꍇɔAONA
  exception_clear (current_frame);

  DBG (_T ("JVM throws "));
  DBG_UTF8 (classname);
  DBG (_T ("\n"));

  class_loader *loader =
    get_defining_loader (current_frame->current_class_file);
  ClassFile *cfile =
    find_ClassFile (current_frame, loader, intern_utf8 (classname));

  if (cfile == NULL)
    {
      // NXt@C[hłȂBMuAbv
      DBG (_T ("Cannot throw "));
      DBG_UTF8 (classname);
      DBG (_T ("\n"));
      assert (false);

      return false;
    }
  return throw_exception (current_frame, cfile, message);
}

bool
throw_exception_int (frame * current_frame, const char *classname, jint param)
{
  // ɗOĂꍇɔAONA
  exception_clear (current_frame);

  DBG (_T ("JVM throws "));
  DBG_UTF8 (classname);
  DBG (_T ("\n"));

  class_loader *loader =
    get_defining_loader (current_frame->current_class_file);
  ClassFile *cfile =
    find_ClassFile (current_frame, loader, intern_utf8 (classname));

  if (cfile == NULL)
    {
      // NXt@C[hłȂBMuAbv
      DBG (_T ("Cannot throw "));
      DBG_UTF8 (classname);
      DBG (_T ("\n"));
      assert (false);

      return false;
    }
  return throw_exception_int (current_frame, cfile, param);
}

/**
 * OutOfMemoryErrorthrow
 */
bool
throw_OutOfMemoryError (frame * current_frame)
{
  DBG (_T ("JVM throws OutOfMemoryError\n"));
  return throw_exception (current_frame, g_OutOfMemoryError_instance);
}

/**
 * StackOverflowErrorthrow
 */
bool
throw_StackOverflowError (frame * current_frame)
{
  DBG (_T ("JVM throws StackOverflowError\n"));
  return throw_exception (current_frame, g_StackOverflowError_instance);
}

/**
 * ݗOĂ邩Ԃ
 *
 * @param	current_frame	JgEt[
 * @return	OĂꍇ̗͂OBĂȂꍇNULL.
 */
jthrowable
exception_occurred (frame * current_frame)
{
  return current_frame->body->current_exception;
}

/**
 * ONA
 *
 * @param	current_frame	JgEt[
 */
void
exception_clear (frame * current_frame)
{
  current_frame->body->current_exception = NULL;
}

/**
 * s  t ɑ\𒲂ׂ
 * iJNIIsAssignableFrom֐Ɠ̈ӖɂȂj
 *
 * @param	s	ΏۂƂȂNX
 * @param	t	NX
 * @return	cfile instanceof name ̏ꍇ true
 */
bool
is_assignable_from (frame * current_frame, ClassFile * s, ClassFile * t)
{
  if (s == t)
    {
      // NX
      return true;
    }
  if (!is_interface (s->access_flags))
    {
      // SNX
      if (!is_array_class (s))
	{
	  // SzNX
	  if (!is_interface (t->access_flags))
	    {
	      // T NXłꍇAS  T ܂ T̃TuNXłȂ΂ȂȂ
	      if (s->this_class_name == t->this_class_name)
		{
		  return true;
		}
	      ClassFile *super_cfile = get_superclass_ClassFile (s);
	      if (!super_cfile)
		{
		  return false;
		}
	      // ċAIɒׂ
	      return is_assignable_from (current_frame, super_cfile, t);
	    }
	  else
	    {
	      // T C^tF[X̏ꍇASTimplementsĂȂ΂ȂȂ
	      u2 interfaces_count = s->interfaces_count;
	      class_loader *loader = get_defining_loader (s);
	      for (u2 i = 0; i < interfaces_count; ++i)
		{
		  const java_utf8 *interface_name =
		    get_CONSTANT_Class_info (s, s->interfaces[i]);
		  ClassFile *interface_class_file =
		    get_interface_ClassFile (s, i);

		  // C^tF[X T ƈv邩ׂ
		  if (interface_class_file->this_class_name ==
		      t->this_class_name)
		    {
		      return true;
		    }

		  // ċAIɒׂ
		  if (is_assignable_from
		      (current_frame, interface_class_file, t))
		    {
		      return true;
		    }
		}
	      // X[p[NXŎĂC^tF[XrĂ
	      const java_utf8 *super_class_name = s->super_class_name;
	      if (super_class_name == NULL)
		{
		  return false;
		}
	      // ċAIɒׂ
	      return is_assignable_from (current_frame,
					 get_superclass_ClassFile (s), t);
	    }
	}
      else
	{
	  // SzNX
	  if (!is_interface (t->access_flags))
	    {
	      // TNX
	      if (!is_array_class (t))
		{
		  // TzNX^̏ꍇAT java/lang/Object łȂ΂ȂȂ
		  if (JAVA_LANG_OBJECT_CLASS_NAME == t->this_class_name)
		    {
		      return true;
		    }
		  else
		    {
		      return false;
		    }
		}
	      else
		{
		  // Tz^ TC[] ̏ꍇAȉ̂ꂩ̏𖞂Ȃ΂ȂȂ
		  // ETC  SC v~eBu^łB
		  // ETCSCQƌ^łASCTCɃLXg\ł
		  // iLɉāuSTƂꎟłvƂOKvHj

		  // NXv邩𒲂ׂ
		  if (s->this_class_name == t->this_class_name)
		    {
		      // ҂^ł邱ƂۏႳ
		      return true;
		    }
		  else
		    {
		      if (is_primitive_array_class (s)
			  || is_primitive_array_class (t))
			{
			  return false;
			}
		    }

		  // zvfċAIɒׂĂ
		  // [hɂ́AzNX̒`[_pȂ΂ȂȂ_ɒӂKv
		  java_utf8 *tc_name =
		    get_array_component_name (t->this_class_name);
		  java_utf8 *sc_name =
		    get_array_component_name (s->this_class_name);
		  ClassFile *tc =
		    find_ClassFile (current_frame, get_defining_loader (t),
				    tc_name);
		  ClassFile *sc =
		    find_ClassFile (current_frame, get_defining_loader (s),
				    sc_name);
		  return is_assignable_from (current_frame, sc, tc);
		}
	    }
	  else
	    {
	      // TC^tF[X^łꍇ
	      // zNXĂC^tF[XTƈv邩𒲂ׂ
	      u2 interfaces_count = s->interfaces_count;
	      class_loader *loader = get_defining_loader (s);
	      for (u2 i = 0; i < interfaces_count; ++i)
		{
		  const java_utf8 *interface_name =
		    get_CONSTANT_Class_info (s, s->interfaces[i]);
		  // C^tF[X T ƈv邩ׂ
		  if (interface_name == t->this_class_name)
		    {
		      return true;
		    }
		}
	      return false;
	    }
	}
    }
  else
    {
      // SC^tF[X̏ꍇ
      if (!is_interface (t->access_flags))
	{
	  // T NX^̏ꍇAT "java/lang/Object" łȂ΂ȂȂ
	  if (JAVA_LANG_OBJECT_CLASS_NAME == t->this_class_name)
	    {
	      return true;
	    }
	  else
	    {
	      return false;
	    }
	}
      else
	{
	  // TC^tF[X^̏ꍇA
	  // TSƓC^tF[X
	  if (s->this_class_name == t->this_class_name)
	    {
	      return true;
	    }
	  // TS̃X[p[C^tF[X
	  u2 interfaces_count = s->interfaces_count;
	  class_loader *loader = get_defining_loader (s);
	  for (u2 i = 0; i < interfaces_count; ++i)
	    {
	      const java_utf8 *interface_name =
		get_CONSTANT_Class_info (s, s->interfaces[i]);
	      ClassFile *interface_class_file =
		get_interface_ClassFile (s, i);

	      // C^tF[X T ƈv邩ׂ
	      if (interface_class_file->this_class_name == t->this_class_name)
		{
		  return true;
		}

	      // ċAIɒׂ
	      if (is_assignable_from (current_frame, interface_class_file, t))
		{
		  return true;
		}
	    }
	  return false;
	}
    }
}

/**
 * w肳ꂽUTF-8̓eێStringIuWFNg쐬B
 * 쐬StringIuWFNg̎QƂ́A frame Ŏw肳ꂽt[
 * pushB
 *
 * @param	frame	t[B쐬String̎QƂpushB
 * @param	utf8	UTF-8
 */
jstring
new_utf8_string (frame * frame, java_utf8 * utf8)
{
  return alloc_utf8_string (frame, utf8, false);
}

/**
 * w肳ꂽUTF-8̓eێStringIuWFNg쐬B
 * 쐬StringIuWFNg̎QƂ́A frame Ŏw肳ꂽt[
 * pushB
 * ̊֐gpč쐬́AKx[WRNV̑ΏۂƂȂȂB
 *
 * @param	frame	t[B쐬String̎QƂpushB
 * @param	utf8	UTF-8
 */
jstring
permanent_utf8_string (frame * frame, const java_utf8 * utf8)
{
  return alloc_utf8_string (frame, utf8, true);
}

/**
 * w肳ꂽUTF-8̓eێStringIuWFNg쐬B
 * 쐬StringIuWFNg̎QƂ́A frame Ŏw肳ꂽt[
 * pushB
 * permanenttruȅꍇAKx[WRNV̑ΏۂƂȂȂB
 *
 * @param	frame	t[B쐬String̎QƂpushB
 * @param	utf8	UTF-8
 * @param	permanent	
 */
static jstring
alloc_utf8_string (frame * frame, const char *utf8, bool permanent)
{
  static ClassFile *string_class_file;
  static ClassFile *char_array_class_file;
  static method_info *string_constructor_minfo;

  // StringClassFile[h
  if (!string_class_file)
    {
      string_class_file = find_ClassFile (frame, STRING_CLASS_NAME);
      if (!string_class_file)
	{
	  return NULL;
	}
    }

  // 蓖Ă
  jstring string_reference =
    (jstring) alloc_object (string_class_file, frame);
  if (string_reference == NULL)
    {
      return NULL;
    }
  if (permanent)
    {
      // Kx[WRNVΏۊOƂ
      new_global_reference (string_reference);
    }

  DUP_STACK (frame);		// String̎QƂ𕡐

  // charz쐬
  // char[]NXt@C擾
  if (!char_array_class_file)
    {
      char_array_class_file = find_ClassFile (frame, CHAR_ARRAY_CLASS_NAME);
      if (!char_array_class_file)
	{
	  return NULL;
	}
    }

  jobject char_array_reference =
    alloc_char_array (char_array_class_file, utf8, frame);
  if (char_array_reference == NULL)
    {
      return NULL;
    }

  // RXgN^Ăяo(String(char[]))
  if (!string_constructor_minfo)
    {
      string_constructor_minfo
	= find_declaring_method_info (string_class_file,
				      INIT_METHOD_NAME,
				      CHAR_ARRAY_ARG_CONSTRUCTOR_DESCRIPTOR);
    }
  invoke_method (frame,
		 string_reference,
		 string_class_file, string_constructor_minfo);

  // dup()Ă̂ŁAX^bNgbv String ̎QƂĂ

  return string_reference;
}

/**
 * w肳ꂽNX from ANX target  access_flags 
 * NX^\bh^tB[hɃANZX\𔻒肷
 */
static bool
is_accessible (frame * current_frame, ClassFile * from, ClassFile * target,
	       u2 access_flags)
{
  if (from == target)
    {
      // g̃tB[h^\bhɂ͕KANZX\
      return true;
    }

  if (is_public (access_flags))
    {
      // public̏ꍇ͏ɃANZX\
      return true;
    }
  if (is_protected (access_flags))
    {
      // protected̏ꍇɂ͈ȉ̂ꂩ̏𖞂ĂȂ΂ȂȂ
      // EpbP[W
      // Efrom  target̔hNXłKv
      class_loader *loader = get_defining_loader (from);
      if (is_same_package (loader, from, target))
	{
	  return true;
	}

      while ((from = get_superclass_ClassFile (from)) != NULL)
	{
	  if (from == target)
	    {
	      return true;
	    }
	}
      return false;
    }
  if (is_private (access_flags))
    {
      // privatȅꍇɂ͎głȂ΂ȂȂ
      return from == target;
    }

  // pbP[WvCx[g̏ꍇ
  class_loader *loader = get_defining_loader (from);
  return is_same_package (loader, from, target);
}

/**
 * jsr߂si[eBeBj
 */
static inline void
execute_jsr (frame * current_frame, __int32 branch)
{
  // jsr߂̒3Ȃ̂ŁApc3ZlPUSH
  // returnAddress^ł邱Ƃߍŏʃrbg𗧂ĂĂ
  // iretߎsɂ̃rbg̓NAj
  PUSH_INT (current_frame,
	    (unsigned int) (current_frame->current_pc + 3) | 0x80000000);
  current_frame->current_pc += branch;
}

// ȍ~ EXECUTOR֐

/**
 * tableswitch߂s
 */
static void
execute_tableswitch (frame * current_frame, u1 * bytecode)
{
  assert (false);
}

/**
 * lookupswitch߂s
 */
static void
execute_lookupswitch (frame * current_frame, u1 * bytecode)
{
  assert (false);
}

/**
 * getstatic߂s
 */
#if 0
static void
execute_getstatic (frame * current_frame, u1 * bytecode)
{
  u1 index1 = GET_PARAMETER (bytecode, 1);
  u1 index2 = GET_PARAMETER (bytecode, 2);
  u2 index = MAKE_U2 (index1, index2);

  // CfbNXŎw肳ꂽ CONSTANT_Fieldref_info \̂ւ̃|C^擾
  CONSTANT_Fieldref_info *fieldref =
    get_CONSTANT_Fieldref_info (current_frame->current_class_file, index);

  // FieldrefŎw肳ꂽNX擾 
  ClassFile *target_class_file = fieldref->class_file;
  if (target_class_file == NULL)
    {
      class_loader *loader =
	get_defining_loader (current_frame->current_class_file);
      target_class_file =
	find_ClassFile (current_frame, loader, fieldref->class_name);
      if (!target_class_file)
	{
	  // Ɏs
	  return;
	}
      fieldref->class_file = target_class_file;
    }

  // tB[hID擾
  jfieldID fid =
    get_static_field_id (target_class_file, fieldref->field_name,
			 fieldref->field_descriptor);
  if (fid == NULL)
    {
      // ToDo: Oݒ肷
      return;
    }

  // [hǉ
  field_info *finfo = get_static_field_info (target_class_file, fid);
  if (!add_loading_constraints_of_field (current_frame,
					 target_class_file,
					 finfo->descriptor))
    {
      return;
    }
  // ANZX`FbN
  if (!is_accessible (current_frame,
		      current_frame->current_class_file,
		      target_class_file, finfo->access_flags))
    {
      throw_exception (current_frame, "java/lang/IllegalAccessError",
		       finfo->name);
      return;
    }

  // X^eBbNtB[h擾
  jobject static_instance = get_static_data (target_class_file);
  get_static_field (current_frame, static_instance, fid);

}
#endif

/**
 * putstatic߂s
 */
#if 0
static void
execute_putstatic (frame * current_frame, u1 * bytecode)
{
  u1 index1 = GET_PARAMETER (bytecode, 1);
  u1 index2 = GET_PARAMETER (bytecode, 2);
  u2 index = MAKE_U2 (index1, index2);

  // CfbNXŎw肳ꂽ CONSTANT_Fieldref_info \̂ւ̃|C^擾
  CONSTANT_Fieldref_info *fieldref =
    get_CONSTANT_Fieldref_info (current_frame->current_class_file, index);

  // FieldrefŎw肳ꂽNX擾
  ClassFile *target_class_file = fieldref->class_file;
  if (target_class_file == NULL)
    {
      class_loader *loader =
	get_defining_loader (current_frame->current_class_file);
      target_class_file =
	find_ClassFile (current_frame, loader, fieldref->class_name);
      if (!target_class_file)
	{
	  // Ɏs
	  return;
	}
      fieldref->class_file = target_class_file;
    }

  // tB[hID擾
  jfieldID fid =
    get_static_field_id (target_class_file, fieldref->field_name,
			 fieldref->field_descriptor);
  if (fid == NULL)
    {
      // Oݒ肷
      return;
    }

  // [hǉ
  field_info *finfo = get_static_field_info (target_class_file, fid);
  if (!add_loading_constraints_of_field (current_frame,
					 target_class_file,
					 finfo->descriptor))
    {
      return;
    }

  // ANZX`FbN
  if (!is_accessible (current_frame,
		      current_frame->current_class_file,
		      target_class_file, finfo->access_flags))
    {
      throw_exception (current_frame, "java/lang/IllegalAccessError",
		       finfo->name);
      return;
    }

  jobject static_instance = get_static_data (target_class_file);
  put_static_field (current_frame, static_instance, fid);
}
#endif

/**
 * wide߂s
 */
static void
execute_wide (frame * current_frame, u1 * bytecode)
{
  assert (false);
}

/**
 * multianewarray߂
 */
static void
execute_multianewarray (frame * current_frame, u1 * bytecode)
{
  u1 indexbyte1 = GET_PARAMETER (bytecode, 1);
  u1 indexbyte2 = GET_PARAMETER (bytecode, 2);
  u1 dimensions = GET_PARAMETER (bytecode, 3);
  u2 index = MAKE_U2 (indexbyte1, indexbyte2);

  // e̗vfIyhX^bN擾
  jint *counts = (__int32 *) alloca (sizeof (int) * dimensions);
  for (int i = dimensions - 1; i >= 0; --i)
    {
      POP_INT (current_frame, counts[i]);
    }
  // Svf̍vvZ
  jint tmp_stack_size = counts[0];
  if (tmp_stack_size == 0)
    {
      // vfȌꍇɔ
      tmp_stack_size = 1;
    }
  else if (tmp_stack_size < 0)
    {
      // NegativeArraySizeException  throw 
      throw_exception_int (current_frame,
			   "java/lang/NegativeArraySizeException",
			   tmp_stack_size);
      return;
    }

  for (int i = 1; i < dimensions; ++i)
    {
      unsigned int count = counts[i];
      if (count < 0)
	{
	  // NegativeArraySizeException  throw 
	  throw_exception_int (current_frame,
			       "java/lang/NegativeArraySizeException",
			       tmp_stack_size);
	  return;
	}
      tmp_stack_size *= count;
    }

  // Zievf"e"̔z񕪁j
  tmp_stack_size += dimensions;

  // NX̏擾
  ClassFile *cfile = current_frame->current_class_file;
  const java_utf8 *cname = get_CONSTANT_Class_info (cfile, index);
  class_loader *loader =
    get_defining_loader (current_frame->current_class_file);

  // e|̃X^bN쐬i쐬̔z񂪃Kx[WRNgȂ悤ɂ邽߁j
  frame *tmp_stack =
    alloc_root_frame (current_frame->current_class_file, tmp_stack_size + 1);
  if (tmp_stack == NULL)
    {
      throw_OutOfMemoryError (current_frame);
      return;
    }

  jobject arrayref =
    create_multiarray (current_frame, loader, cname, dimensions, counts,
		       tmp_stack);

  PUSH_OBJECT (current_frame, arrayref);

  // e|X^bN폜
  free_root_frame (tmp_stack);
}

/**
 * w肳ꂽ CONSTANT_Fieldref_info 
 */
static bool resolve(frame* current_frame, CONSTANT_Fieldref_info* fieldref, bool instance_field)
{
  if (fieldref->finfo)
  {
    // łɉς
    return true;
  }

  class_loader *loader = get_defining_loader (current_frame->current_class_file);
  ClassFile* target_class_file = find_ClassFile (current_frame, loader, fieldref->class_name);
  if (!target_class_file)
  {
    // Ɏs
    return false;
  }
  
  // tB[hID擾
  jfieldID fid;
  if (instance_field)
  {
    fid = get_field_id (target_class_file,
                        fieldref->field_name,
                        fieldref->field_descriptor);
  }
  else
  {
    fid = get_static_field_id (target_class_file,
                               fieldref->field_name,
			       fieldref->field_descriptor);
  }

  if (! fid)
  {
    // vIȖ肪Ă
    throw_exception(current_frame, "java/lang/VirtualMachineError", "Cannot get fid");
    return false;
  }

  // [hǉ
  field_info* finfo;
  if (instance_field)
  {
    finfo = get_field_info (target_class_file, fid);
  }
  else
  {
    finfo = get_static_field_info (target_class_file, fid);
  }
  assert(finfo);

  ClassFile *declaring_class_file = finfo->declaring_ClassFile;
  assert (declaring_class_file);

  if (!add_loading_constraints_of_field (current_frame,
					 declaring_class_file,
					 finfo->descriptor))
  {
    return false;
  }
  
  // ANZX`FbN
  if (!is_accessible (current_frame,
		      current_frame->current_class_file,
		      declaring_class_file,
                      finfo->access_flags))
  {
    throw_exception(current_frame, "java/lang/IllegalAccessError", finfo->name);
    return false;
  }

  // field_infoLbViŁuς݁vɂȂj
  set_field_info(fieldref, finfo);

  return true;
}